# retrieve movement patterns for a species from the IUCN Red List

# %%
api_token = ""

# %%
import requests
# import json
import time

import pandas as pd
import matplotlib.pyplot as plt
import seaborn as sns
import numpy as np
import matplotlib.dates as mdates
from datetime import datetime

# %%
def get_species_assessment(api_token, latin_name, assessment_no):
    # wait for 1 second to avoid hitting the API rate limit
    time.sleep(1)

    genus_name, species_name = latin_name.split(" ")

    # correct request URL
    request_url = f"https://api.iucnredlist.org/api/v4/taxa/scientific_name?genus_name={genus_name}&species_name={species_name}"

    # Set the API endpoint and parameters
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"{api_token}"
    } 

    # Make the API request
    response = requests.get(request_url, headers=headers)


    # Check if the request was successful
    if response.status_code == 200:
        # Parse the JSON response
        data = response.json()
        # Print the data
        # print(json.dumps(data, indent=4))

        # Extract the latest assessment id
        assessment_id = data["assessments"][assessment_no]["assessment_id"]

        return assessment_id
    else:
        raise Exception(f"Error: {response.status_code} - {response.text}")


# %%
def movement_patterns_from_assessment(api_token, assessment_id):
    # wait for 1 second to avoid hitting the API rate limit
    time.sleep(1)
    assessment_request_url = f"https://api.iucnredlist.org/api/v4/assessment/{assessment_id}"

    # Set the API endpoint and parameters
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"{api_token}"
    }

    # Make the API request
    response = requests.get(assessment_request_url, headers=headers)
    
    # Check if the request was successful
    if response.status_code == 200:
        # Parse the JSON response
        data = response.json()
        # Print the data
        # print(json.dumps(data, indent=4))
        supplementary_info = data["supplementary_info"]
        # print(json.dumps(supplementary_info, indent=4))
        movement_patterns = supplementary_info["movement_patterns"]
        # print(json.dumps(movement_patterns, indent=4))
        return movement_patterns
    else:
        raise Exception(f"Error: {response.status_code} - {response.text}")

# %%
# read in birdsnap-species.txt as csv separated by tabs or spaces, first row is header, id column is index
birdsnap_species = pd.read_csv("birdsnap-species.txt", sep="\t", header=0, index_col=0)
# %%
birdsnap_species.head()

# %%
# extract id and scientific_name columns
birdsnap_species = birdsnap_species[["scientific"]]
birdsnap_species.head()
# %% 
# prepare a new column for movement patterns
birdsnap_species["movement_patterns"] = None
# iterate over rows
for index, row in birdsnap_species.iterrows():
    # get the scientific name
    scientific_name = row["scientific"]
    # get the latest assessment id and movement patterns
    try:
        assessment_id = get_species_assessment(api_token, scientific_name, assessment_no=0)
        # get the movement patterns
        movement_patterns = movement_patterns_from_assessment(api_token, assessment_id)
        print(f"{index}: {scientific_name}: {movement_patterns}")

        # set the movement patterns in the dataframe
        birdsnap_species.at[index, "movement_patterns"] = movement_patterns
    # exception for http error
    except requests.exceptions.HTTPError as e:
        print(f"{index} : HTTPError {e.response.status_code}: {scientific_name} not found in IUCN Red List")
    # exception for other errors
    except Exception as e:
        print(f"{index} : Error: {e}: {scientific_name} not found in IUCN Red List")
    
# %%
# save the dataframe to a csv file
birdsnap_species.to_csv("birdsnap-species-movement-patterns.csv", sep="\t", index=True)

# %%
# summarize the movement patterns, include stats on Na values
movement_patterns_column_description = birdsnap_species["movement_patterns"].describe()
movement_patterns_column_description
# %%
# summarize all values taken by the movement patterns column, treat none / na as a separate category
movement_patterns_column_summary = birdsnap_species["movement_patterns"].value_counts()
movement_patterns_column_summary

# %%
# read in inat2018-categories.json, make id column the index column
inat2018_categories = pd.read_json("inat2018-categories.json", orient="records")
inat2018_categories.set_index("id", inplace=True)

# %%
inat2018_categories.head()

# %%
# filter inat2018_categories to only include rows with class "Aves" - birds
inat2018_categories_birds = inat2018_categories[inat2018_categories["class"] == "Aves"]

# %%
inat2018_categories_birds.head()
print(len(inat2018_categories_birds))

# %%
# print length of inat2018_categories
# print(len(inat2018_categories))

# %%
# create small version containing only id
inat2018_categories_small = inat2018_categories_birds[["name"]]
inat2018_categories_small.head()
# %%
# prepare a new column for movement patterns
inat2018_categories_small["movement_patterns"] = None

# iterate through the rows of inat2018_categories_small 
for index, row in inat2018_categories_small.iterrows():
    # get the scientific name
    scientific_name = row["name"]
    # get the latest assessment id and movement patterns
    try:
        assessment_id = get_species_assessment(api_token, scientific_name, assessment_no=0)
        # get the movement patterns
        movement_patterns = movement_patterns_from_assessment(api_token, assessment_id)
        print(f"{index}: {scientific_name}: {movement_patterns}")

        # set the movement patterns in the dataframe
        inat2018_categories_small.at[index, "movement_patterns"] = movement_patterns
    # exception for http error
    except requests.exceptions.HTTPError as e:
        print(f"{index} : HTTPError {e.response.status_code}: {scientific_name} not found in IUCN Red List")
    # exception for other errors
    except Exception as e:
        print(f"{index} : Error: {e}: {scientific_name} not found in IUCN Red List")

# %%
# save the dataframe to a csv file
inat2018_categories_small.to_csv("inat2018-categories-movement-patterns.csv", sep="\t", index=True)

# %%
# summarize the movement patterns, include stats on Na values
movement_patterns_column_description = inat2018_categories_small["movement_patterns"].describe()
movement_patterns_column_description

# %%
# summarize all values taken by the movement patterns column, treat none / na as a separate category
movement_patterns_column_summary = inat2018_categories_small["movement_patterns"].value_counts()
movement_patterns_column_summary

# %%
print(len(inat2018_categories))
# %%
