# mapper-lib

**mapper-lib** is a Python library based on the Mapper algorithm, a key tool in
Topological Data Analysis (TDA). Designed for efficient computations and backed
by advanced spatial search techniques, it scales seamlessly to high dimensional
data, making it suitable for applications in machine learning, data mining, and
exploratory data analysis.

### Main Features

- **Fast Mapper graph construction**: Accelerates computations with efficient spatial search, enabling analysis of large, high-dimensional datasets.

- **Scikit-learn compatibility**: Easily integrate Mapper as a part of your machine learning workflows.

- **Flexible visualization options**: Visualize Mapper graphs with multiple supported backends, tailored to your needs.

### Background

The Mapper algorithm transforms complex datasets into graph representations
that highlight clusters, transitions, and topological features. These insights
reveal hidden patterns in data, applicable across fields like social sciences,
biology, and machine learning. For an in-depth coverage of Mapper, including
its mathematical foundations and applications, read the 
[the original paper](https://research.math.osu.edu/tgda/mapperPBG.pdf).

## Quick Start

### Installation

To install from source, run from the root

```bash
pip install -e .
```

### How to Use

Here's a minimal example using the **circles dataset** from `scikit-learn` to
demonstrate how to use **mapper-lib**:

```python
import numpy as np

from sklearn.datasets import make_circles
from sklearn.decomposition import PCA
from sklearn.cluster import DBSCAN

from mapperlib.learn import MapperAlgorithm
from mapperlib.cover import CubicalCover
from mapperlib.plot import MapperPlot

# load a labelled dataset
X, labels = make_circles(n_samples=5000, noise=0.05, factor=0.3, random_state=42)
y = PCA(2, random_state=42).fit_transform(X)

cover = CubicalCover(n_intervals=10, overlap_frac=0.3)
clust = DBSCAN()
graph = MapperAlgorithm(cover, clust).fit_transform(X, y)

# color according to labels
fig = MapperPlot(graph, dim=2, seed=42, iterations=60).plot_plotly(colors=labels)
fig.show(config={'scrollZoom': True})
```
