import os

import numpy as np
import torch

from fairseq.data import (
    data_utils,
    Dictionary,
    encoders,
    IdDataset,
    MaskTokensDataset,
    TnfMaskTokensDataset,
    NestedDictionaryDataset,
    NumelDataset,
    NumSamplesDataset,
    PadDataset,
    PrependTokenDataset,
    SortDataset,
    TokenBlockDataset,
)
from fairseq.tasks import FairseqTask, register_task


@register_task('tnf_masked_lm')
class TnfMaskedLMTask(FairseqTask):
    """Task for training masked language models (e.g., BERT, RoBERTa)."""

    @staticmethod
    def add_args(parser):
        """Add task-specific arguments to the parser."""
        parser.add_argument('data', help='colon separated path to data directories list, \
                            will be iterated upon during epochs in round-robin manner')
        parser.add_argument('--tnf-data', required=True, help='colon separated path to tnf \
                            data directories list, will be iterated upon during epochs \
                            in round-robin manner')
        parser.add_argument('--tnf-subword-data', help='colon separated path to tnf \
                            data directories list, will be iterated upon during epochs \
                            in round-robin manner')
        parser.add_argument('--sample-break-mode', default='complete',
                            choices=['none', 'complete', 'complete_doc', 'eos'],
                            help='If omitted or "none", fills each sample with tokens-per-sample '
                                 'tokens. If set to "complete", splits samples only at the end '
                                 'of sentence, but may include multiple sentences per sample. '
                                 '"complete_doc" is similar but respects doc boundaries. '
                                 'If set to "eos", includes only one sentence per sample.')
        parser.add_argument('--tokens-per-sample', default=512, type=int,
                            help='max number of total tokens over all segments '
                                 'per sample for BERT dataset')
        parser.add_argument('--mask-prob', default=0.15, type=float,
                            help='probability of replacing a token with mask')
        parser.add_argument('--leave-unmasked-prob', default=0.1, type=float,
                            help='probability that a masked token is unmasked')
        parser.add_argument('--random-token-prob', default=0.1, type=float,
                            help='probability of replacing a token with a random token')
        parser.add_argument('--freq-weighted-replacement', action='store_true',
                            help='sample random replacement words based on word frequencies')
        parser.add_argument('--mask-whole-words', default=False, action='store_true',
                            help='mask whole words; you may also want to set --bpe')
        parser.add_argument('--loss-lamda', default=10.0, type=float,
                            help='lamda trade-off between generator loss and discriminator loss')
        parser.add_argument('--tnf-lambda', default=None, type=float,
                            help='tnf lambda when merge with original embeddings')
        parser.add_argument('--tnf-gamma', default=None, type=float,
                            help='moving average parameter gamma when updating the tnf embedding')

    def __init__(self, args, dictionary, tnf_dictionary, tnf_subword_dictionary):
        super().__init__(args)
        self.dictionary = dictionary
        self.tnf_dictionary = tnf_dictionary
        self.tnf_subword_dictionary = tnf_subword_dictionary
        self.seed = args.seed

        # add mask token
        self.mask_idx = dictionary.add_symbol('<mask>')
        self.tnf_mask_idx = tnf_dictionary.add_symbol('<mask>')
        if self.tnf_subword_dictionary is not None:
            self.tnf_subword_mask_idx = tnf_subword_dictionary.add_symbol('<mask>')
            # subword mask tokens
            self.is_subword_idx = tnf_subword_dictionary.index('1')
        else:
            self.is_subword_idx = None

    @classmethod
    def setup_task(cls, args, **kwargs):
        paths = args.data.split(':')
        tnf_paths = args.tnf_data.split(':')
        assert len(paths) > 0
        assert len(tnf_paths) > 0
        dictionary = Dictionary.load(os.path.join(paths[0], 'dict.txt'))
        print('| dictionary: {} types'.format(len(dictionary)))
        tnf_dictionary = Dictionary.load(os.path.join(tnf_paths[0], 'dict.txt'))
        print('| tnf dictionary: {} types'.format(len(tnf_dictionary)))
        if hasattr(args, 'tnf_subword_data') and args.tnf_subword_data is not None:
            tnf_subword_paths = args.tnf_subword_data.split(':')
            assert len(tnf_subword_paths) > 0
            tnf_subword_dictionary = Dictionary.load(os.path.join(tnf_subword_paths[0], 'dict.txt'))
            print('| tnf subword dictionary: {} types'.format(len(tnf_subword_dictionary)))
        else:
            tnf_subword_dictionary = None
        return cls(args, dictionary, tnf_dictionary, tnf_subword_dictionary)

    def load_dataset(self, split, epoch=0, combine=False, data_selector=None):
        """Load a given dataset split.

        Args:
            split (str): name of the split (e.g., train, valid, test)
        """
        paths = self.args.data.split(':')
        assert len(paths) > 0
        data_path = paths[epoch % len(paths)]
        split_path = os.path.join(data_path, split)

        dataset = data_utils.load_indexed_dataset(
            split_path,
            self.source_dictionary,
            self.args.dataset_impl,
            combine=combine,
        )
        if dataset is None:
            raise FileNotFoundError('Dataset not found: {} ({})'.format(split, split_path))

        # create continuous blocks of tokens
        dataset = TokenBlockDataset(
            dataset,
            dataset.sizes,
            self.args.tokens_per_sample - 1,  # one less for <s>
            pad=self.source_dictionary.pad(),
            eos=self.source_dictionary.eos(),
            break_mode=self.args.sample_break_mode,
        )
        print('| loaded {} batches from: {}'.format(len(dataset), split_path))

        # prepend beginning-of-sentence token (<s>, equiv. to [CLS] in BERT)
        dataset = PrependTokenDataset(dataset, self.source_dictionary.bos())


        # load tnf dataset
        tnf_paths = self.args.tnf_data.split(':')
        assert len(tnf_paths) > 0
        tnf_data_path = tnf_paths[epoch % len(tnf_paths)]
        tnf_split_path = os.path.join(tnf_data_path, split)

        tnf_dataset = data_utils.load_indexed_dataset(
            tnf_split_path,
            self.tnf_source_dictionary,
            self.args.dataset_impl,
            combine=combine,
        )
        if tnf_dataset is None:
            raise FileNotFoundError('tnf dataset not found: {} ({})'.format(split, tnf_split_path))

        # create continuous blocks of tokens
        tnf_dataset = TokenBlockDataset(
            tnf_dataset,
            tnf_dataset.sizes,
            self.args.tokens_per_sample - 1,  # one less for <s>
            pad=self.tnf_source_dictionary.pad(),
            eos=self.tnf_source_dictionary.eos(),
            break_mode=self.args.sample_break_mode,
        )
        print('| loaded {} batches from: {}'.format(len(tnf_dataset), tnf_split_path))

        # prepend beginning-of-sentence token (<s>, equiv. to [CLS] in BERT)
        tnf_dataset = PrependTokenDataset(tnf_dataset, self.tnf_source_dictionary.bos())

        # load tnf subword dataset
        if hasattr(self.args, 'tnf_subword_data') and self.args.tnf_subword_data is not None:
            tnf_subword_path = self.args.tnf_subword_data.split(':')
            assert len(tnf_subword_path) > 0
            tnf_subword_data_path = tnf_subword_path[epoch % len(tnf_subword_path)]
            tnf_subword_split_path = os.path.join(tnf_subword_data_path, split)

            tnf_subword_dataset = data_utils.load_indexed_dataset(
                tnf_subword_split_path,
                self.tnf_subword_dictionary,
                self.args.dataset_impl,
                combine=combine,
            )
            if tnf_subword_dataset is None:
                raise FileNotFoundError('tnf subword dataset not found: {} ({})'.format(split, tnf_subword_split_path))

            # create continuous blocks of tokens
            tnf_subword_dataset = TokenBlockDataset(
                tnf_subword_dataset,
                tnf_subword_dataset.sizes,
                self.args.tokens_per_sample - 1,  # one less for <s>
                pad=self.tnf_subword_dictionary.pad(),
                eos=self.tnf_subword_dictionary.eos(),
                break_mode=self.args.sample_break_mode,
            )
            print('| loaded {} batches from: {}'.format(len(tnf_subword_dataset), tnf_subword_split_path))

            # prepend beginning-of-sentence token (<s>, equiv. to [CLS] in BERT)
            tnf_subword_dataset = PrependTokenDataset(tnf_subword_dataset, self.tnf_source_dictionary.bos())
        else:
            tnf_subword_dataset = None


        # create masked input and targets
        if self.args.mask_whole_words:
            bpe = encoders.build_bpe(self.args)
            if bpe is not None:

                def is_beginning_of_word(i):
                    if i < self.source_dictionary.nspecial:
                        # special elements are always considered beginnings
                        return True
                    tok = self.source_dictionary[i]
                    if tok.startswith('madeupword'):
                        return True
                    try:
                        return bpe.is_beginning_of_word(tok)
                    except ValueError:
                        return True

                mask_whole_words = torch.ByteTensor(list(
                    map(is_beginning_of_word, range(len(self.source_dictionary)))
                ))
        else:
            mask_whole_words = None

        src_dataset, tnf_src_dataset, tnf_src_dataset_nomask, tgt_dataset = TnfMaskTokensDataset.apply_mask(
            dataset,
            tnf_dataset,
            tnf_subword_dataset,
            self.is_subword_idx,
            self.source_dictionary,
            self.tnf_source_dictionary,
            pad_idx=self.source_dictionary.pad(),
            tnf_pad_idx=self.tnf_source_dictionary.pad(),
            mask_idx=self.mask_idx,
            tnf_mask_idx=self.tnf_mask_idx,
            seed=self.args.seed,
            mask_prob=self.args.mask_prob,
            leave_unmasked_prob=self.args.leave_unmasked_prob,
            random_token_prob=self.args.random_token_prob,
            freq_weighted_replacement=self.args.freq_weighted_replacement,
            mask_whole_words=mask_whole_words,
        )

        with data_utils.numpy_seed(self.args.seed + epoch):
            shuffle = np.random.permutation(len(src_dataset))

        self.datasets[split] = SortDataset(
            NestedDictionaryDataset(
                {
                    'id': IdDataset(),
                    'net_input': {
                        'src_tokens': PadDataset(
                            src_dataset,
                            pad_idx=self.source_dictionary.pad(),
                            left_pad=False,
                        ),
                        'tnf_src_tokens': PadDataset(
                            tnf_src_dataset,
                            pad_idx=self.tnf_source_dictionary.pad(),
                            left_pad=False,
                        ),
                        'tnf_src_tokens_nomask': PadDataset(
                            tnf_src_dataset,
                            pad_idx=self.tnf_source_dictionary.pad(),
                            left_pad=False,
                        ),
                        'src_lengths': NumelDataset(src_dataset, reduce=False),
                    },
                    'target': PadDataset(
                        tgt_dataset,
                        pad_idx=self.source_dictionary.pad(),
                        left_pad=False,
                    ),
                    'nsentences': NumSamplesDataset(),
                    'ntokens': NumelDataset(src_dataset, reduce=True),
                },
                sizes=[src_dataset.sizes],
            ),
            sort_order=[
                shuffle,
                src_dataset.sizes,
            ],
        )

    def build_dataset_for_inference(self, src_tokens, tnf_src_tokens, src_lengths, tnf_src_lengths,
                                    sort=True):
        src_dataset = PadDataset(
            TokenBlockDataset(
                src_tokens,
                src_lengths,
                self.args.tokens_per_sample - 1,  # one less for <s>
                pad=self.source_dictionary.pad(),
                eos=self.source_dictionary.eos(),
                break_mode='eos',
            ),
            pad_idx=self.source_dictionary.pad(),
            left_pad=False,
        )
        src_dataset = PrependTokenDataset(src_dataset, self.source_dictionary.bos())

        tnf_src_dataset = PadDataset(
            TokenBlockDataset(
                tnf_src_tokens,
                tnf_src_lengths,
                self.args.tokens_per_sample - 1,  # one less for <s>
                pad=self.tnf_source_dictionary.pad(),
                eos=self.tnf_source_dictionary.eos(),
                break_mode='eos',
            ),
            pad_idx=self.tnf_source_dictionary.pad(),
            left_pad=False,
        )
        tnf_src_dataset = PrependTokenDataset(tnf_src_dataset, self.tnf_source_dictionary.bos())


        src_dataset = NestedDictionaryDataset(
            {
                'id': IdDataset(),
                'net_input': {
                    'src_tokens': src_dataset,
                    'src_lengths': NumelDataset(src_dataset, reduce=False),
                },
                'tnf_src_tokens': tnf_src_dataset,
            },
            sizes=src_lengths,
        )
        if sort:
            src_dataset = SortDataset(src_dataset, sort_order=[src_lengths])
        return src_dataset

    @property
    def source_dictionary(self):
        return self.dictionary

    @property
    def tnf_source_dictionary(self):
        return self.tnf_dictionary

    @property
    def target_dictionary(self):
        return self.dictionary
