##########################################################################################
# Machine Environment 
DEBUG_MODE = False
USE_CUDA = not DEBUG_MODE
CUDA_DEVICE_NUM = 1

##########################################################################################
# Path Config
import os
import sys
import torch
import numpy as np

os.chdir(os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, "..")  # for problem_def
sys.path.insert(0, "../..")  # for utils

##########################################################################################
# import
import logging
from utils.utils import create_logger, copy_all_src


from MOTSPTester import TSPTester as Tester
from MOTSProblemDef import get_random_problems

##########################################################################################
import time
import hvwfg

from matplotlib import pyplot as plt
import matplotlib as mpl


import pickle

def load_dataset(file_path):
    if not os.path.exists(file_path):
        raise FileNotFoundError(f"Dataset file {file_path} not found.")
    
    with open(file_path, 'rb') as f:
        data = pickle.load(f)
    
    return data


def load_TSPLIB_dataset(path):
    return torch.load(path)



mpl.style.use('default')
##########################################################################################
# parameters
env_params = {
    'problem_size': 200,
    'pomo_size': 200,
}

model_params = {
    'num_experts': 4,
    'topk': 2,
    'routing_level': 'node',
    'routing_method': 'input_choice',
    'embedding_dim': 128,
    'sqrt_embedding_dim': 128**(1/2),
    'encoder_layer_num': 6,
    'qkv_dim': 16,
    'head_num': 8,
    'logit_clipping': 10,
    'ff_hidden_dim': 512,
    'eval_type': 'argmax',
}

tester_params = {
    'use_cuda': USE_CUDA,
    'cuda_device_num': CUDA_DEVICE_NUM,
    'model_load': {
        'path': './result/saved_TSP100_model',  # directory path of pre-trained model and log files saved.
        'epoch': 200, 
    },
    'test_episodes': 1, 
    'test_batch_size': 1,
    'augmentation_enable': True,
    'aug_factor': 1, #64,
    'aug_batch_size': 1
}
if tester_params['augmentation_enable']:
    tester_params['test_batch_size'] = tester_params['aug_batch_size']

logger_params = {
    'log_file': {
        'desc': 'test__tsp_n100',
        'filename': 'run_log'
    }
}

##########################################################################################
def _set_debug_mode():
    global tester_params
    tester_params['test_episodes'] = 100

def _print_config():
    logger = logging.getLogger('root')
    logger.info('DEBUG_MODE: {}'.format(DEBUG_MODE))
    logger.info('USE_CUDA: {}, CUDA_DEVICE_NUM: {}'.format(USE_CUDA, CUDA_DEVICE_NUM))
    [logger.info(g_key + "{}".format(globals()[g_key])) for g_key in globals().keys() if g_key.endswith('params')]

##########################################################################################
def main(n_sols = 101):

    timer_start = time.time()
    logger_start = time.time()
    
    if DEBUG_MODE:
        _set_debug_mode()
    
    create_logger(**logger_params)
    _print_config()
    
    tester = Tester(env_params=env_params,
                    model_params=model_params,
                    tester_params=tester_params)
    
    copy_all_src(tester.result_folder)
    
    sols = np.zeros([n_sols, 2])
    
    # shared_problem = get_random_problems(tester_params['test_episodes'], env_params['problem_size'])
    
    problem_size = 200
    
    
    device = torch.device('cuda:1' if USE_CUDA is True else 'cpu')

    # loaded_problem = load_dataset('/data/liuw2/test_data/motsp/motsp%d_test_seed1234.pkl'%(problem_size))
    
    loaded_problem = load_TSPLIB_dataset(f"../data/kro/kro{env_params['problem_size']}.pt")
    
    
    shared_problem = torch.FloatTensor(loaded_problem).to(device)

    
    
    
    
    
    for i in range(n_sols):
        pref = torch.zeros(2).cuda()
        pref[0] = 1 - 0.01 * i
        pref[1] = 0.01 * i
        pref = pref / torch.sum(pref)
    
        aug_score = tester.run(shared_problem,pref)
        sols[i] = np.array(aug_score)
    
    timer_end = time.time()
    
    total_time = timer_end - timer_start
   
    # MOTSP 20
    #single_task = [3.83, 3.83]
    
    # MOTSP 50
    #single_task = [5.69, 5.69]
    
    # MOTSP 100
    single_task = [7.76, 7.76]
    
    fig = plt.figure()
    
    plt.axvline(single_task[0],linewidth=3 , alpha = 0.25)
    plt.axhline(single_task[1],linewidth=3,alpha = 0.25, label = 'Single Objective TSP (Concorde)')
    plt.plot(sols[:,0],sols[:,1], marker = 'o', c = 'C1',ms = 3,  label='PSL-MOCO (Ours)')
    
    plt.legend()
    
    ref = np.array([115, 115])  # 200
    
    hv = hvwfg.wfg(sols.astype(float), ref.astype(float))
    hv_ratio =  hv / (ref[0] * ref[1])
    
    print('Run Time(s): {:.4f}'.format(total_time))
    print('HV Ratio: {:.4f}'.format(hv_ratio))
    
##########################################################################################
if __name__ == "__main__":
    main()