import argparse
from functools import wraps
from typing import Callable, List


def output_added_arguments(add_algo_arguments: Callable) -> Callable:
    @wraps(add_algo_arguments)
    def decorated(parser: argparse.ArgumentParser) -> List[str]:
        unfiltered_old_arguments = list(parser._option_string_actions.keys())

        add_algo_arguments(parser)

        unfiltered_arguments = list(parser._option_string_actions.keys())
        unfiltered_added_arguments = [
            argument for argument in unfiltered_arguments if argument not in unfiltered_old_arguments
        ]

        return [
            argument.strip("-")
            for argument in unfiltered_added_arguments
            if argument.startswith("--") and argument not in ["--help"]
        ]

    return decorated


@output_added_arguments
def add_base_arguments(parser: argparse.ArgumentParser):
    parser.add_argument(
        "-en",
        "--experiment_name",
        help="Experiment name.",
        type=str,
        required=True,
    )
    parser.add_argument(
        "-s",
        "--seed",
        help="Seed of the experiment.",
        type=int,
        required=True,
    )
    parser.add_argument(
        "-dw",
        "--disable_wandb",
        help="Disable wandb.",
        default=False,
        action="store_true",
    )
    parser.add_argument(
        "-f",
        "--features",
        nargs="*",
        help="List of features for the Q-networks.",
        type=int,
        default=[200, 200],
    )
    parser.add_argument(
        "-rbc",
        "--replay_buffer_capacity",
        help="Replay Buffer capacity.",
        type=int,
        default=10_000,
    )
    parser.add_argument(
        "-bs",
        "--batch_size",
        help="Batch size for training.",
        type=int,
        default=32,
    )
    parser.add_argument(
        "-n",
        "--update_horizon",
        help="Value of n in n-step TD update.",
        type=int,
        default=1,
    )
    parser.add_argument(
        "-lr",
        "--learning_rate",
        help="Learning rate.",
        type=float,
        default=3e-4,
    )
    parser.add_argument(
        "-horizon",
        "--horizon",
        help="Horizon for truncation.",
        type=int,
        default=1000,
    )
    parser.add_argument(
        "-ne",
        "--n_epochs",
        help="Number of epochs to perform.",
        type=int,
        default=50,
    )
    parser.add_argument(
        "-ntspe",
        "--n_training_steps_per_epoch",
        help="Number of training steps per epoch.",
        type=int,
        default=10_000,
    )
    parser.add_argument(
        "-utd",
        "--update_to_data",
        help="Number of data points to collect per online Q-network update.",
        type=float,
        default=1,
    )
    parser.add_argument(
        "-tuf",
        "--target_update_frequency",
        help="Number of training steps before updating the target Q-network.",
        type=int,
        default=200,
    )
    parser.add_argument(
        "-nis",
        "--n_initial_samples",
        help="Number of initial samples before the training starts.",
        type=int,
        default=1_000,
    )
    parser.add_argument(
        "-ee",
        "--epsilon_end",
        help="Ending value for the linear decaying epsilon used for exploration.",
        type=float,
        default=0.01,
    )
    parser.add_argument(
        "-ed",
        "--epsilon_duration",
        help="Duration of epsilon's linear decay used for exploration.",
        type=float,
        default=1_000,
    )


@output_added_arguments
def add_dqn_arguments(parser: argparse.ArgumentParser):
    parser.add_argument(
        "-gamma",
        "--gamma",
        help="Discounting factor.",
        type=float,
        default=0.99,
    )


@output_added_arguments
def add_metadqn_arguments(parser: argparse.ArgumentParser):
    parser.add_argument(
        "-gamma_i",
        "--gamma_init",
        help="Initial discounting factor.",
        type=float,
        default=0.99,
    )
    parser.add_argument(
        "-gamma_v",
        "--gamma_validation",
        help="Validation discounting factor.",
        type=float,
        default=0.995,
    )
    parser.add_argument(
        "-mlr",
        "--meta_learning_rate",
        help="Meta learning rate.",
        type=float,
        default=0.001,
    )


@output_added_arguments
def add_adadqn_arguments(parser: argparse.ArgumentParser):
    parser.add_argument(
        "-gammar",
        "--gamma_range",
        nargs=2,
        help="Gamma range.",
        type=float,
        default=[0.8, 0.999],
    )
    parser.add_argument(
        "-gamma_v",
        "--gamma_validation",
        help="Validation discounting factor.",
        type=float,
        default=0.995,
    )
    parser.add_argument(
        "-nn",
        "--n_networks",
        help="Number of networks trained in parralel.",
        type=int,
        default=5,
    )
    parser.add_argument(
        "-et",
        "--exploitation_type",
        help="Type of exploitation in the space of hyperparameters.",
        type=str,
        choices=["elitism", "truncation"],
        default="elitism",
    )
    parser.add_argument(
        "-huf",
        "--hp_update_frequency",
        help="Number of training steps before updating the hyperparameter.",
        type=int,
        default=10_000,
    )
