import os
import gym
import d4rl
import scipy
import tqdm
import functools

import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.optim import Adam
from torch.utils.data import DataLoader
from torch.utils.tensorboard import SummaryWriter

from diffusion_SDE.loss import loss_fn
from diffusion_SDE.schedule import marginal_prob_std
from diffusion_SDE.model import ScoreNet
from utils import get_args, pallaral_eval_policy
from dataset.dataset import D4RL_dataset


def support_action(args):
    for dir in ["./models_rl", "./logs"]:
        if not os.path.exists(dir):
            os.makedirs(dir)
    if not os.path.exists(os.path.join("./models_rl", str(args.expid))):
        os.makedirs(os.path.join("./models_rl", str(args.expid)))
    writer = SummaryWriter("./logs/" + str(args.expid))
    
    env = gym.make(args.env)
    env.seed(args.seed)
    env.action_space.seed(args.seed)
    torch.manual_seed(args.seed)
    np.random.seed(args.seed)
    args.eval_func = functools.partial(pallaral_eval_policy, env_name=args.env, seed=args.seed, eval_episodes=args.seed_per_evaluation, diffusion_steps=args.diffusion_steps)
    state_dim = env.observation_space.shape[0]
    action_dim = env.action_space.shape[0]
    max_action = float(env.action_space.high[0])
    args.writer = writer
    
    marginal_prob_std_fn = functools.partial(marginal_prob_std, device=args.device)
    args.marginal_prob_std_fn = marginal_prob_std_fn
    score_model= ScoreNet(input_dim=state_dim+action_dim, output_dim=action_dim, marginal_prob_std=marginal_prob_std_fn, args=args).to(args.device)
    score_model.q[0].to(args.device)

    print("loading actor...")
    ckpt = torch.load(args.actor_load_path, map_location=args.device)
    score_model.load_state_dict(ckpt)
    
    dataset = D4RL_dataset(args)
    data_loader = DataLoader(dataset, batch_size=256, shuffle=True)
    score_model.q[0].guidance_scale = 0.0
    
    allstates = dataset.states[:].cpu().numpy()
    actions = []
    for states in tqdm.tqdm(np.array_split(allstates, allstates.shape[0] // 256 + 1)):
        actions.append(score_model.sample(states, sample_per_state=args.M, diffusion_steps=args.diffusion_steps))
    actions = np.concatenate(actions)
    dataset.fake_actions = torch.Tensor(actions.astype(np.float32)).to(args.device)
    np.save(os.path.join("./models_rl", str(args.expid))+ "/actions{}_raw.npy".format(args.diffusion_steps), actions)

if __name__ == "__main__":
    args = get_args()
    if "antmaze" not in args.env:
        args.M = 16
    else:
        args.M = 32
    support_action(args)