import torch
import torch.nn as nn
import torch.nn.functional as F
from layers.Transformer_EncDec_change import Encoder, EncoderLayer
from layers.SelfAttention_Family import FullAttention, AttentionLayer
from layers.Embed import DataEmbedding_inverted, DataEmbedding_inverted_Linear, FrequencyMaskResample
import numpy as np


class Model(nn.Module):

    def __init__(self, configs):
        super(Model, self).__init__()
        self.seq_len = configs.seq_len
        self.pred_len = configs.pred_len
        self.output_attention = configs.output_attention
        self.use_norm = configs.use_norm
        # Embedding
        # self.enc_embedding = DataEmbedding_inverted(configs.seq_len, configs.d_model, configs.embed, configs.freq,
        #                                             configs.dropout)
        # self.enc_embedding = DataEmbedding_inverted_Linear(configs.seq_len, configs.d_model, configs.embed, configs.freq,
        #                                             configs.dropout)
        self.fmr = FrequencyMaskResample(num_variables=configs.enc_in, sequence_length=self.seq_len,
                                                    out_length=configs.d_model,
                                                    mask_activation='softplus')

        self.class_strategy = configs.class_strategy
        # Encoder-only architecture
        self.encoder = Encoder(
            [
                EncoderLayer(
                    AttentionLayer(
                        FullAttention(False, configs.factor, attention_dropout=configs.dropout,
                                      output_attention=configs.output_attention), configs.d_model, configs.n_heads), 
                    configs.d_model,  
                    configs.d_ff,
                    config=configs, 
                    dropout=configs.dropout,
                    activation=configs.activation
                ) for l in range(configs.e_layers)
            ],
            norm_layer=torch.nn.LayerNorm(configs.d_model),  
        )
        self.projector = nn.Linear(configs.d_model, configs.pred_len, bias=True)  

    def forecast(self, x_enc, x_mark_enc, x_dec, x_mark_dec):
        if self.use_norm:
            # Normalization from Non-stationary Transformer
            means = x_enc.mean(1, keepdim=True).detach()
            x_enc = x_enc - means
            stdev = torch.sqrt(torch.var(x_enc, dim=1, keepdim=True, unbiased=False) + 1e-5)
            x_enc /= stdev

        _, _, N = x_enc.shape  # B L N
        # enc_out = self.enc_embedding(x_enc, x_mark_enc)
        enc_out = self.fmr(x_enc.permute(0,2,1))  # remove timestamps information

        # B N E -> B N E    (B L E -> B L E in the vanilla Transformer)
        # enc_out, attns, adj = self.encoder(enc_out.permute(0,2,1), attn_mask=None)  # enc_out[B,N,D]
        enc_out, attns, adj = self.encoder(enc_out, attn_mask=None)  # enc_out[B,N,D]

        # B N E -> B N S -> B S N 
        # dec_out = self.projector(enc_out).permute(0, 2, 1)[:, :, :N] # filter the covariates
        dec_out = self.projector(enc_out).permute(0, 2, 1)

        if self.use_norm:
            # De-Normalization from Non-stationary Transformer
            dec_out = dec_out * (stdev[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))
            dec_out = dec_out + (means[:, 0, :].unsqueeze(1).repeat(1, self.pred_len, 1))

        return dec_out, attns,adj

    def forward(self, x_enc, x_mark_enc, x_dec, x_mark_dec, mask=None):
        dec_out, attns, adj = self.forecast(x_enc, x_mark_enc, x_dec, x_mark_dec)
        
        if self.output_attention:
            return dec_out[:, -self.pred_len:, :], attns
        else:
            return dec_out[:, -self.pred_len:, :], attns, adj  # [B, L, D]
