from functools import partial

import jax
import jax.numpy as jnp
import numpy as np
import optax
from flax.core import FrozenDict

from slimdqn.networks.architectures.dqn import DQNNet
from slimdqn.sample_collection.fixed_replay_buffer import FixedReplayBuffer
from slimdqn.sample_collection.replay_buffer import ReplayElement


class CQL:
    def __init__(
        self,
        key: jax.random.PRNGKey,
        observation_dim,
        n_actions,
        features: list,
        layer_norm: bool,
        architecture_type: str,
        learning_rate: float,
        gamma: float,
        update_horizon: int,
        target_update_frequency: int,
        alpha_cql: float,
        adam_eps: float = 0.0003125,
    ):
        self.network = DQNNet(features, architecture_type, n_actions, layer_norm)
        self.params = self.network.init(key, jnp.zeros(observation_dim, dtype=jnp.float32))

        self.optimizer = optax.adam(learning_rate, eps=adam_eps)
        self.optimizer_state = self.optimizer.init(self.params)
        self.target_params = self.params.copy()

        self.gamma = gamma
        self.update_horizon = update_horizon
        self.target_update_frequency = target_update_frequency
        self.cumulated_loss = np.zeros(2)  # one entry each for TD and BC component
        self.cumulated_variance = 0
        self.alpha_cql = alpha_cql

    @partial(jax.jit, static_argnames="self")
    def apply_multiple_updates(self, params, params_target, optimizer_state, batches):
        def apply_single_update(state, batch):
            params, optimizer_state, loss, variance = self.learn_on_batch(state[0], params_target, state[1], batch)
            return (params, optimizer_state), (loss, variance)

        # Convert the list of batch to a list single batch where each element
        # has the shape (n_batch, batch_size) + (element_shape,)
        batches = jax.tree.map(lambda *batch: jnp.stack(batch), *batches)
        (final_params, final_optimizer_state), (loss, variance) = jax.lax.scan(
            apply_single_update, (params, optimizer_state), batches
        )
        return final_params, final_optimizer_state, loss.sum(axis=0), variance.sum(axis=0)

    def n_updates_online_params(self, n_updates: int, replay_buffer: FixedReplayBuffer):
        batches = replay_buffer.sample(n_updates)
        self.params, self.optimizer_state, loss, variance = self.apply_multiple_updates(
            self.params, self.target_params, self.optimizer_state, batches
        )
        self.cumulated_loss += loss
        self.cumulated_variance += variance

    def update_target_params(self, **kwargs):
        self.target_params = self.params.copy()

        logs = {
            "td_loss": self.cumulated_loss[0] / self.target_update_frequency,
            "bc_loss": self.alpha_cql * self.cumulated_loss[1] / self.target_update_frequency,
            "variance": self.cumulated_variance / self.target_update_frequency,
        }
        self.cumulated_loss = np.zeros_like(self.cumulated_loss)
        self.cumulated_variance = 0

        return logs

    def learn_on_batch(self, params: FrozenDict, params_target: FrozenDict, optimizer_state, batch_samples):
        grad_loss, (losses, variance) = jax.grad(self.loss_on_batch, has_aux=True)(params, params_target, batch_samples)
        updates, optimizer_state = self.optimizer.update(grad_loss, optimizer_state)
        params = optax.apply_updates(params, updates)

        return params, optimizer_state, losses, variance

    def loss_on_batch(self, params: FrozenDict, params_target: FrozenDict, samples):
        losses, loss_terms, variance = jax.vmap(self.loss, in_axes=(None, None, 0))(params, params_target, samples)
        return losses.mean(axis=0), (loss_terms.mean(axis=0), variance.mean(axis=0))

    def loss(self, params: FrozenDict, params_target: FrozenDict, sample: ReplayElement):
        q_values = self.network.apply(params, sample.state)
        target = self.compute_target(params_target, sample)
        td_loss = jnp.square(target - q_values[sample.action])
        bc_loss = jax.scipy.special.logsumexp(q_values, axis=-1) - q_values[sample.action]
        return (
            td_loss + self.alpha_cql * bc_loss,
            jnp.array([td_loss, bc_loss]),
            target**2 - target * q_values[sample.action],
        )

    def compute_target(self, params: FrozenDict, sample: ReplayElement):
        # computes the target value for single sample
        return sample.reward + (1 - sample.is_terminal) * (self.gamma**self.update_horizon) * jnp.max(
            self.network.apply(params, sample.next_state)
        )

    @partial(jax.jit, static_argnames="self")
    def best_action(self, params: FrozenDict, state: jnp.ndarray, **kwargs):
        # computes the best action for a single state
        return jnp.argmax(self.network.apply(params, state))

    def get_model(self):
        return {"params": self.params}
