from functools import partial

import jax
import jax.numpy as jnp
import numpy as np
import optax
from flax.core import FrozenDict

from slimdqn.networks.architectures.dqn import DQNNet
from slimdqn.sample_collection.fixed_replay_buffer import FixedReplayBuffer
from slimdqn.sample_collection.replay_buffer import ReplayElement


class TFCQL:
    def __init__(
        self,
        key: jax.random.PRNGKey,
        observation_dim,
        n_actions,
        features: list,
        layer_norm: bool,
        batch_norm: bool,
        architecture_type: str,
        learning_rate: float,
        gamma: float,
        update_horizon: int,
        target_update_frequency: int,
        alpha_cql: float,
        adam_eps: float = 1e-8,
    ):
        self.network = DQNNet(features, architecture_type, n_actions, layer_norm, batch_norm)
        self.params = self.network.init(key, jnp.zeros(observation_dim, dtype=jnp.float32))

        self.optimizer = optax.adam(learning_rate, eps=adam_eps)
        self.optimizer_state = self.optimizer.init(self.params)

        self.gamma = gamma
        self.update_horizon = update_horizon
        self.target_update_frequency = target_update_frequency
        self.cumulated_loss = np.zeros(2)  # one entry each for TD and BC component
        self.alpha_cql = alpha_cql

    @partial(jax.jit, static_argnames="self")
    def apply_multiple_updates(self, params, optimizer_state, batches):
        def apply_single_update(state, batch):
            params, optimizer_state, loss = self.learn_on_batch(state[0], state[1], batch)
            return (params, optimizer_state), loss

        # Convert the list of batch to a list single batch where each element
        # has the shape (n_batch, batch_size) + (element_shape,)
        batches = jax.tree.map(lambda *batch: jnp.stack(batch), *batches)
        (final_params, final_optimizer_state), loss = jax.lax.scan(
            apply_single_update, (params, optimizer_state), batches
        )
        return final_params, final_optimizer_state, loss.sum(axis=0)

    def n_updates_online_params(self, n_updates: int, replay_buffer: FixedReplayBuffer):
        batches = replay_buffer.sample(n_updates)
        self.params, self.optimizer_state, loss = self.apply_multiple_updates(
            self.params, self.optimizer_state, batches
        )
        self.cumulated_loss += loss

    def update_target_params(self, **kwargs):
        logs = {
            "td_loss": self.cumulated_loss[0] / self.target_update_frequency,
            "bc_loss": self.alpha_cql * self.cumulated_loss[1] / self.target_update_frequency,
        }
        self.cumulated_loss = np.zeros_like(self.cumulated_loss)

        return logs

    @partial(jax.jit, static_argnames="self")
    def learn_on_batch(self, params: FrozenDict, optimizer_state, batch_samples):
        (loss, batch_stats), grad_loss = jax.value_and_grad(self.loss_on_batch, has_aux=True)(params, batch_samples)
        updates, optimizer_state = self.optimizer.update(grad_loss, optimizer_state)
        params = optax.apply_updates(params, updates)
        if self.network.batch_norm:
            params["batch_stats"] = batch_stats["batch_stats"]

        return params, optimizer_state, loss

    def loss_on_batch(self, params: FrozenDict, samples):
        batch_size = samples.state.shape[0]
        # shape (2 * batch_size, n_actions)
        all_q_values, batch_stats = self.network.apply(
            params, jnp.concatenate((samples.state, samples.next_state)), mutable=["batch_stats"]
        )
        q_values = jax.vmap(lambda q_value, action: q_value[action])(all_q_values[:batch_size], samples.action)
        targets = self.compute_target(samples, all_q_values[batch_size:])
        stop_grad_targets = jax.lax.stop_gradient(targets)
        # shape (batch_size, n_bellman_iterations)
        td_losses = jnp.square(q_values - stop_grad_targets)
        bc_losses = jax.scipy.special.logsumexp(all_q_values[:batch_size], axis=-1) - q_values

        return (td_losses + self.alpha_cql * bc_losses).mean(axis=0), (
            jnp.array([td_losses.mean(axis=0), bc_losses.mean(axis=0)]),
            batch_stats,
        )

    def compute_target(self, sample: ReplayElement, next_q_values: jax.Array):
        # shape of next_q_values (next_states, n_actions)
        return sample.reward + (1 - sample.is_terminal) * (self.gamma**self.update_horizon) * jnp.max(
            next_q_values, axis=-1
        )

    @partial(jax.jit, static_argnames="self")
    def best_action(self, params: FrozenDict, state: jnp.ndarray, **kwargs):
        # computes the best action for a single state
        return jnp.argmax(self.network.apply(params, state, use_running_average=True))

    def get_model(self):
        return {"params": self.params}
