DEBUG_MODE = False
USE_CUDA = False
CUDA_DEVICE_NUM = 0

# Path Config
import os
import numpy as np
from TSPTester import TSPTester as Tester

# Decode method: use RRC or not (greedy)
Use_RRC = False
RRC_budget = 0

model_load_path = 'checkpoints/'
model_load_epoch = 150

# Configuration for different problem sizes and modes
test_paras = {
    'train': {
        200: ['test_TSP200_n128.txt', 10, 10, 0],
        500: ['test_TSP500_n128.txt', 10, 10, 0],
        1000: ['test_TSP1000_n128.txt', 10, 10, 0],
    },
    'test': {
        200: ['test_TSP200_n128.txt', 64, 64, 64],
        500: ['test_TSP500_n128.txt', 64, 64, 64],
        1000: ['test_TSP1000_n128.txt', 64, 64, 64],
    }
}

def main():
    # Set working directory
    os.chdir(os.path.dirname(os.path.abspath(__file__)))
    
    # Check for required files
    basepath = os.path.dirname(__file__)
    if not os.path.isfile(os.path.join(basepath, "checkpoints/checkpoint-150.pt")):
        raise FileNotFoundError("No checkpoints found. Please see the readme.md and download the checkpoints.")
    for mode in ['train', 'test']:
        for problem_size in [200, 500, 1000]:
            if not os.path.isfile(os.path.join(basepath, f"data/test_TSP{problem_size}_n128.txt")):
                raise FileNotFoundError(f"No test data found for TSP{problem_size}. Please see the readme.md and download the data.")

    # Run training and testing for each problem size
    for mode in ['train', 'test']:
        print(f"\n[*] Running {mode.upper()} phase...")
        results = {}
        for problem_size in [200, 500, 1000]:
            print(f"\n[*] Processing problem size: {problem_size}")
            score_optimal, score_student, gap = main_test(problem_size, mode)
            results[problem_size] = {
                'optimal': score_optimal,
                'student': score_student,
                'gap': gap
            }
            print(f"Problem size: {problem_size}, Optimal: {score_optimal}, Student: {score_student}, Gap: {gap}")
        
        # Print average for training mode
        if mode == 'train':
            avg_student = np.mean([results[ps]['student'] for ps in results])
            print(f"\n[*] Average student score for {mode}: {avg_student}")

def main_test(problem_size, mode, use_RRC=None, cuda_device_num=None):
    env_params = {
        'mode': mode,
        'data_path': f"./data/{test_paras[mode][problem_size][0]}",
        'sub_path': False,
        'RRC_budget': RRC_budget if use_RRC is None else 0
    }

    model_params = {
        'mode': mode,
        'embedding_dim': 128,
        'sqrt_embedding_dim': 128 ** (1 / 2),
        'decoder_layer_num': 6,
        'qkv_dim': 16,
        'head_num': 8,
        'ff_hidden_dim': 512,
    }

    tester_params = {
        'use_cuda': USE_CUDA,
        'cuda_device_num': cuda_device_num if cuda_device_num is not None else CUDA_DEVICE_NUM,
        'test_episodes': test_paras[mode][problem_size][1],
        'test_batch_size': test_paras[mode][problem_size][2],
        'test_start_idx': test_paras[mode][problem_size][3],
        'model_load': {
            'path': model_load_path,
            'epoch': model_load_epoch,
        }
    }

    if DEBUG_MODE:
        _set_debug_mode(tester_params)
    
    tester = Tester(env_params=env_params,
                    model_params=model_params,
                    tester_params=tester_params)

    score_optimal, score_student, gap = tester.run()
    return score_optimal, score_student, gap

def _set_debug_mode(tester_params):
    tester_params['test_episodes'] = 100

if __name__ == "__main__":
    print("[*] Starting LEHD TSP evaluation...")
    main()
    print("[*] Evaluation completed.")