import torch
import torch.nn.functional as F

def heuristics(decoder, encoded_last_node, ninf_mask):
    """
    Heuristics function for solving the Traveling Salesman Problem (TSP) with advanced decision strategies.
    Args:
        decoder: TSP_Decoder instance providing necessary attributes.
        encoded_last_node: torch.Tensor of shape (batch, pomo, embedding).
        ninf_mask: torch.Tensor of shape (batch, pomo, problem).
    Returns:
        probs: torch.Tensor of shape (batch, pomo, problem).
    """
    head_num = decoder.model_params['head_num']

    # Multi-Head Attention
    q_last = decoder.Wq_last(encoded_last_node).view(-1, head_num, decoder.model_params['dim_head'])
    q = decoder.q_first.unsqueeze(1) + q_last
    out_concat = torch.matmul(F.softmax(torch.matmul(q, decoder.k.transpose(1, 2)) / (decoder.model_params['dim_head'] ** 0.5), dim=-1), decoder.v)
    mh_atten_out = decoder.multi_head_combine(out_concat)

    # Single-Head Attention
    score = torch.matmul(mh_atten_out, decoder.single_head_key)

    # Dynamic Bias Generation for Exploration
    problem_size = ninf_mask.size(-1)
    scale_factor = torch.log(torch.tensor(problem_size, dtype=torch.float, device=score.device))
    random_addition = torch.randn_like(score) * 0.1  # Introduce randomness
    dynamic_bias = torch.relu(torch.matmul(encoded_last_node, decoder.single_head_key)) * scale_factor + random_addition
    score = score + dynamic_bias

    # Logit Scaling and Masking
    sqrt_embedding_dim = decoder.model_params['sqrt_embedding_dim']
    logit_clipping = decoder.model_params['logit_clipping']
    score_scaled = score / sqrt_embedding_dim
    score_clipped = logit_clipping * torch.tanh(score_scaled)
    score_masked = score_clipped + ninf_mask
    probs = F.softmax(score_masked, dim=2)

    return probs
