# from typing import Optional
# import logging
# import subprocess
# import numpy as np
# import os
# from omegaconf import DictConfig

# from utils.utils import *
# from utils.llm_client.base import BaseClient


# class ReEvo:
#     def __init__(
#         self, 
#         cfg: DictConfig, 
#         root_dir: str, 
#         generator_llm: BaseClient, 
#         reflector_llm: Optional[BaseClient] = None,
        
#         # Support setting different LLMs for each of the four operators: 
#         # Short-term Reflection, Long-term Reflection, Crossover, Mutation
#         short_reflector_llm: Optional[BaseClient] = None,
#         long_reflector_llm: Optional[BaseClient] = None,
#         crossover_llm: Optional[BaseClient] = None,
#         mutation_llm: Optional[BaseClient] = None,
#         workspace_dir: Optional[str] = None,
#     ) -> None:
#         self.cfg = cfg
#         self.generator_llm = generator_llm
#         self.reflector_llm = reflector_llm or generator_llm

#         self.short_reflector_llm = short_reflector_llm or self.reflector_llm
#         self.long_reflector_llm = long_reflector_llm or self.reflector_llm
#         self.crossover_llm = crossover_llm or generator_llm
#         self.mutation_llm = mutation_llm or generator_llm

#         self.root_dir = root_dir
        
#         self.mutation_rate = cfg.mutation_rate
#         self.iteration = 0
#         self.function_evals = 0
#         self.elitist = None
#         self.long_term_reflection_str = ""
#         self.best_obj_overall = None
#         self.best_code_overall = None
#         self.best_code_path_overall = None
#         self.workspace_dir = workspace_dir
#         self.init_prompt()
#         self.init_population()


#     def init_prompt(self) -> None:
#         self.problem = self.cfg.problem.problem_name
#         self.problem_desc = self.cfg.problem.description
#         self.problem_size = self.cfg.problem.problem_size
#         self.func_name = self.cfg.problem.func_name
#         self.obj_type = self.cfg.problem.obj_type
#         self.problem_type = self.cfg.problem.problem_type
        
#         logging.info("Problem: " + self.problem)
#         logging.info("Problem description: " + self.problem_desc)
#         logging.info("Function name: " + self.func_name)
        
#         self.prompt_dir = f"{self.root_dir}/prompts"
#         self.output_file = f"{self.root_dir}/problems/{self.problem}/gpt.py"
        
#         # Loading all text prompts
#         # Problem-specific prompt components
#         prompt_path_suffix = "_black_box" if self.problem_type == "black_box" else ""
#         problem_prompt_path = f'{self.prompt_dir}/{self.problem}{prompt_path_suffix}'
#         self.seed_func = file_to_string(f'{problem_prompt_path}/seed_func.txt')
#         self.func_signature = file_to_string(f'{problem_prompt_path}/func_signature.txt')
#         self.func_desc = file_to_string(f'{problem_prompt_path}/func_desc.txt')
#         if os.path.exists(f'{problem_prompt_path}/external_knowledge.txt'):
#             self.external_knowledge = file_to_string(f'{problem_prompt_path}/external_knowledge.txt')
#             self.long_term_reflection_str = self.external_knowledge
#         else:
#             self.external_knowledge = ""
        
        
#         # Common prompts
#         self.system_generator_prompt = file_to_string(f'{self.prompt_dir}/common/system_generator.txt')
#         self.system_reflector_prompt = file_to_string(f'{self.prompt_dir}/common/system_reflector.txt')
#         self.user_reflector_st_prompt = file_to_string(f'{self.prompt_dir}/common/user_reflector_st.txt') if self.problem_type != "black_box" else file_to_string(f'{self.prompt_dir}/common/user_reflector_st_black_box.txt') # shrot-term reflection
#         self.user_reflector_lt_prompt = file_to_string(f'{self.prompt_dir}/common/user_reflector_lt.txt') # long-term reflection
#         self.crossover_prompt = file_to_string(f'{self.prompt_dir}/common/crossover.txt')
#         self.mutation_prompt = file_to_string(f'{self.prompt_dir}/common/mutation.txt')
#         self.user_generator_prompt = file_to_string(f'{self.prompt_dir}/common/user_generator.txt').format(
#             func_name=self.func_name, 
#             problem_desc=self.problem_desc,
#             func_desc=self.func_desc,
#             )
#         self.seed_prompt = file_to_string(f'{self.prompt_dir}/common/seed.txt').format(
#             seed_func=self.seed_func,
#             func_name=self.func_name,
#         )

#         # Flag to print prompts
#         self.print_crossover_prompt = True # Print crossover prompt for the first iteration
#         self.print_mutate_prompt = True # Print mutate prompt for the first iteration
#         self.print_short_term_reflection_prompt = True # Print short-term reflection prompt for the first iteration
#         self.print_long_term_reflection_prompt = True # Print long-term reflection prompt for the first iteration


#     def init_population(self) -> None:
#         # Evaluate the seed function, and set it as Elite
#         logging.info("Evaluating seed function...")
#         code = extract_code_from_generator(self.seed_func).replace("v1", "v2")
#         logging.info("Seed function code: \n" + code)
#         seed_ind = {
#             "stdout_filepath": f"problem_iter{self.iteration}_stdout0.txt",
#             "code_path": f"problem_iter{self.iteration}_code0.py",
#             "code": code,
#             "response_id": 0,
#         }
#         self.seed_ind = seed_ind
#         self.population = self.evaluate_population([seed_ind])

#         # If seed function is invalid, stop
#         if not self.seed_ind["exec_success"]:
#             raise RuntimeError(f"Seed function is invalid. Please check the stdout file in {os.getcwd()}.")

#         self.update_iter()
        
#         # Generate responses
#         system = self.system_generator_prompt
#         user = self.user_generator_prompt + "\n" + self.seed_prompt + "\n" + self.long_term_reflection_str
#         messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
#         logging.info("Initial Population Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)

#         responses = self.generator_llm.multi_chat_completion([messages], self.cfg.init_pop_size, temperature = self.generator_llm.temperature + 0.3) # Increase the temperature for diverse initial population
#         population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(responses)]

#         # Run code and evaluate population
#         population = self.evaluate_population(population)

#         # Update iteration
#         self.population = population
#         self.update_iter()

    
#     def response_to_individual(self, response: str, response_id: int, file_name: str=None) -> dict:
#         """
#         Convert response to individual
#         """
#         # Write response to file
#         file_name = f"problem_iter{self.iteration}_response{response_id}.txt" if file_name is None else file_name + ".txt"
#         with open(file_name, 'w') as file:
#             file.writelines(response + '\n')

#         code = extract_code_from_generator(response)

#         # Extract code and description from response
#         std_out_filepath = f"problem_iter{self.iteration}_stdout{response_id}.txt" if file_name is None else file_name + "_stdout.txt"
        
#         individual = {
#             "stdout_filepath": std_out_filepath,
#             "code_path": f"problem_iter{self.iteration}_code{response_id}.py",
#             "code": code,
#             "response_id": response_id,
#         }
#         return individual

#     def mark_invalid_individual(self, individual: dict, traceback_msg: str) -> dict:
#         """
#         Mark an individual as invalid.
#         """
#         individual["exec_success"] = False
#         individual["obj"] = float("inf")
#         individual["traceback_msg"] = traceback_msg
#         return individual


#     def evaluate_population(self, population: list[dict]) -> list[float]:
#         """
#         Evaluate population by running code in parallel and computing objective values.
#         """
#         inner_runs = []
#         # Run code to evaluate
#         for response_id in range(len(population)):
#             self.function_evals += 1
#             # Skip if response is invalid
#             if population[response_id]["code"] is None:
#                 population[response_id] = self.mark_invalid_individual(population[response_id], "Invalid response!")
#                 inner_runs.append(None)
#                 continue
            
#             logging.info(f"Iteration {self.iteration}: Running Code {response_id}")
            
#             try:
#                 process = self._run_code(population[response_id], response_id)
#                 inner_runs.append(process)
#             except Exception as e: # If code execution fails
#                 logging.info(f"Error for response_id {response_id}: {e}")
#                 population[response_id] = self.mark_invalid_individual(population[response_id], str(e))
#                 inner_runs.append(None)
        
#         # Update population with objective values
#         for response_id, inner_run in enumerate(inner_runs):
#             if inner_run is None: # If code execution fails, skip
#                 continue
#             try:
#                 inner_run.communicate(timeout=self.cfg.timeout) # Wait for code execution to finish
#             except subprocess.TimeoutExpired as e:
#                 logging.info(f"Error for response_id {response_id}: {e}")
#                 population[response_id] = self.mark_invalid_individual(population[response_id], str(e))
#                 inner_run.kill()
#                 continue

#             individual = population[response_id]
#             stdout_filepath = individual["stdout_filepath"]
#             with open(stdout_filepath, 'r') as f:  # read the stdout file
#                 stdout_str = f.read() 
#             traceback_msg = filter_traceback(stdout_str)
            
#             individual = population[response_id]
#             # Store objective value for each individual
#             if traceback_msg == '': # If execution has no error
#                 try:
#                     individual["obj"] = float(stdout_str.split('\n')[-2]) if self.obj_type == "min" else -float(stdout_str.split('\n')[-2])
#                     individual["exec_success"] = True
#                 except:
#                     population[response_id] = self.mark_invalid_individual(population[response_id], "Invalid std out / objective value!")
#             else: # Otherwise, also provide execution traceback error feedback
#                 population[response_id] = self.mark_invalid_individual(population[response_id], traceback_msg)

#             logging.info(f"Iteration {self.iteration}, response_id {response_id}: Objective value: {individual['obj']}")
#         return population


#     def _run_code(self, individual: dict, response_id) -> subprocess.Popen:
#         """
#         Write code into a file and run eval script.
#         """
#         logging.debug(f"Iteration {self.iteration}: Processing Code Run {response_id}")
        
#         with open(self.output_file, 'w') as file:
#             file.writelines(individual["code"] + '\n')

#         # Execute the python file with flags
#         with open(individual["stdout_filepath"], 'w') as f:
#             eval_file_path = f'{self.root_dir}/problems/{self.problem}/eval.py' if self.problem_type != "black_box" else f'{self.root_dir}/problems/{self.problem}/eval_black_box.py' 
#             process = subprocess.Popen(['python', '-u', eval_file_path, f'{self.problem_size}', self.root_dir, "train"],
#                                         stdout=f, stderr=f)

#         block_until_running(individual["stdout_filepath"], log_status=True, iter_num=self.iteration, response_id=response_id)
#         return process

    
#     def update_iter(self) -> None:
#         """
#         Update after each iteration
#         """
#         population = self.population
#         objs = [individual["obj"] for individual in population]
#         best_obj, best_sample_idx = min(objs), np.argmin(np.array(objs))
        
#         # update best overall
#         if self.best_obj_overall is None or best_obj < self.best_obj_overall:
#             self.best_obj_overall = best_obj
#             self.best_code_overall = population[best_sample_idx]["code"]
#             self.best_code_path_overall = population[best_sample_idx]["code_path"]
        
#         # update elitist
#         if self.elitist is None or best_obj < self.elitist["obj"]:
#             self.elitist = population[best_sample_idx]
#             logging.info(f"Iteration {self.iteration}: Elitist: {self.elitist['obj']}")
        
#         best_path = self.best_code_path_overall.replace(".py", ".txt").replace("code", "response")
#         logging.info(f"Best obj: {self.best_obj_overall}, Best Code Path: {best_path, self.best_code_path_overall}")
#         logging.info(f"Iteration {self.iteration} finished...")
#         logging.info(f"Function Evals: {self.function_evals}")
#         self.iteration += 1
        
#     def rank_select(self, population: list[dict]) -> list[dict]:
#         """
#         Rank-based selection, select individuals with probability proportional to their rank.
#         """
#         if self.problem_type == "black_box":
#             population = [individual for individual in population if individual["exec_success"] and individual["obj"] < self.seed_ind["obj"]]
#         else:
#             population = [individual for individual in population if individual["exec_success"]]
#         if len(population) < 2:
#             return None
#         # Sort population by objective value
#         population = sorted(population, key=lambda x: x["obj"])
#         ranks = [i for i in range(len(population))]
#         probs = [1 / (rank + 1 + len(population)) for rank in ranks]
#         # Normalize probabilities
#         probs = [prob / sum(probs) for prob in probs]
#         selected_population = []
#         trial = 0
#         while len(selected_population) < 2 * self.cfg.pop_size:
#             trial += 1
#             parents = np.random.choice(population, size=2, replace=False, p=probs)
#             if parents[0]["obj"] != parents[1]["obj"]:
#                 selected_population.extend(parents)
#             if trial > 1000:
#                 return None
#         return selected_population
    
    
#     def random_select(self, population: list[dict]) -> list[dict]:
#         """
#         Random selection, select individuals with equal probability.
#         """
#         selected_population = []
#         # Eliminate invalid individuals
#         if self.problem_type == "black_box":
#             population = [individual for individual in population if individual["exec_success"] and individual["obj"] < self.seed_ind["obj"]]
#         else:
#             population = [individual for individual in population if individual["exec_success"]]
#         if len(population) < 2:
#             return None
#         trial = 0
#         while len(selected_population) < 2 * self.cfg.pop_size:
#             trial += 1
#             parents = np.random.choice(population, size=2, replace=False)
#             # If two parents have the same objective value, consider them as identical; otherwise, add them to the selected population
#             if parents[0]["obj"] != parents[1]["obj"]:
#                 selected_population.extend(parents)
#             if trial > 1000:
#                 return None
#         return selected_population

#     def gen_short_term_reflection_prompt(self, ind1: dict, ind2: dict) -> tuple[list[dict], str, str]:
#         """
#         Short-term reflection before crossovering two individuals.
#         """
#         if ind1["obj"] == ind2["obj"]:
#             print(ind1["code"], ind2["code"])
#             raise ValueError("Two individuals to crossover have the same objective value!")
#         # Determine which individual is better or worse
#         if ind1["obj"] < ind2["obj"]:
#             better_ind, worse_ind = ind1, ind2
#         else: # robust in rare cases where two individuals have the same objective value
#             better_ind, worse_ind = ind2, ind1

#         worse_code = filter_code(worse_ind["code"])
#         better_code = filter_code(better_ind["code"])
        
#         system = self.system_reflector_prompt
#         user = self.user_reflector_st_prompt.format(
#             func_name = self.func_name,
#             func_desc = self.func_desc,
#             problem_desc = self.problem_desc,
#             worse_code=worse_code,
#             better_code=better_code
#             )
#         message = [{"role": "system", "content": system}, {"role": "user", "content": user}]
        
#         # Print reflection prompt for the first iteration
#         if self.print_short_term_reflection_prompt:
#                 logging.info("Short-term Reflection Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
#                 self.print_short_term_reflection_prompt = False
#         return message, worse_code, better_code


#     def short_term_reflection(self, population: list[dict]) -> tuple[list[list[dict]], list[str], list[str]]:
#         """
#         Short-term reflection before crossovering two individuals.
#         """
#         messages_lst = []
#         worse_code_lst = []
#         better_code_lst = []
#         for i in range(0, len(population), 2):
#             # Select two individuals
#             parent_1 = population[i]
#             parent_2 = population[i+1]
            
#             # Short-term reflection
#             messages, worse_code, better_code = self.gen_short_term_reflection_prompt(parent_1, parent_2)
#             messages_lst.append(messages)
#             worse_code_lst.append(worse_code)
#             better_code_lst.append(better_code)
        
#         # Asynchronously generate responses
#         response_lst = self.short_reflector_llm.multi_chat_completion(messages_lst)
#         return response_lst, worse_code_lst, better_code_lst
    
#     def long_term_reflection(self, short_term_reflections: list[str]) -> None:
#         """
#         Long-term reflection before mutation.
#         """
#         system = self.system_reflector_prompt
#         user = self.user_reflector_lt_prompt.format(
#             problem_desc = self.problem_desc,
#             prior_reflection = self.long_term_reflection_str,
#             new_reflection = "\n".join(short_term_reflections),
#             )
#         messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
        
#         if self.print_long_term_reflection_prompt:
#             logging.info("Long-term Reflection Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
#             self.print_long_term_reflection_prompt = False
        
#         self.long_term_reflection_str = self.long_reflector_llm.multi_chat_completion([messages])[0]
        
#         # Write reflections to file
#         file_name = f"problem_iter{self.iteration}_short_term_reflections.txt"
#         with open(file_name, 'w') as file:
#             file.writelines("\n".join(short_term_reflections) + '\n')
        
#         file_name = f"problem_iter{self.iteration}_long_term_reflection.txt"
#         with open(file_name, 'w') as file:
#             file.writelines(self.long_term_reflection_str + '\n')


#     def crossover(self, short_term_reflection_tuple: tuple[list[list[dict]], list[str], list[str]]) -> list[dict]:
#         reflection_content_lst, worse_code_lst, better_code_lst = short_term_reflection_tuple
#         messages_lst = []
#         for reflection, worse_code, better_code in zip(reflection_content_lst, worse_code_lst, better_code_lst):
#             # Crossover
#             system = self.system_generator_prompt
#             func_signature0 = self.func_signature.format(version=0)
#             func_signature1 = self.func_signature.format(version=1)
#             user = self.crossover_prompt.format(
#                 user_generator = self.user_generator_prompt,
#                 func_signature0 = func_signature0,
#                 func_signature1 = func_signature1,
#                 worse_code = worse_code,
#                 better_code = better_code,
#                 reflection = reflection,
#                 func_name = self.func_name,
#             )
#             messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
#             messages_lst.append(messages)
            
#             # Print crossover prompt for the first iteration
#             if self.print_crossover_prompt:
#                 logging.info("Crossover Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
#                 self.print_crossover_prompt = False
        
#         # Asynchronously generate responses
#         response_lst = self.crossover_llm.multi_chat_completion(messages_lst)
#         crossed_population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(response_lst)]

#         assert len(crossed_population) == self.cfg.pop_size
#         return crossed_population


#     def mutate(self) -> list[dict]:
#         """Elitist-based mutation. We only mutate the best individual to generate n_pop new individuals."""
#         system = self.system_generator_prompt
#         func_signature1 = self.func_signature.format(version=1) 
#         user = self.mutation_prompt.format(
#             user_generator = self.user_generator_prompt,
#             reflection = self.long_term_reflection_str + self.external_knowledge,
#             func_signature1 = func_signature1,
#             elitist_code = filter_code(self.elitist["code"]),
#             func_name = self.func_name,
#         )
#         messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
#         if self.print_mutate_prompt:
#             logging.info("Mutation Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
#             self.print_mutate_prompt = False
#         responses = self.mutation_llm.multi_chat_completion([messages], int(self.cfg.pop_size * self.mutation_rate))
#         population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(responses)]
#         return population


#     def evolve(self):
#         while self.function_evals < self.cfg.max_fe:
#             # If all individuals are invalid, stop
#             if all([not individual["exec_success"] for individual in self.population]):
#                 raise RuntimeError(f"All individuals are invalid. Please check the stdout files in {os.getcwd()}.")
#             # Select
#             population_to_select = self.population if (self.elitist is None or self.elitist in self.population) else [self.elitist] + self.population # add elitist to population for selection
#             selected_population = self.random_select(population_to_select)
#             if selected_population is None:
#                 raise RuntimeError("Selection failed. Please check the population.")
#             # Short-term reflection
#             short_term_reflection_tuple = self.short_term_reflection(selected_population) # (response_lst, worse_code_lst, better_code_lst)
#             # Crossover
#             crossed_population = self.crossover(short_term_reflection_tuple)
#             # Evaluate
#             self.population = self.evaluate_population(crossed_population)
#             # Update
#             self.update_iter()
#             # Long-term reflection
#             self.long_term_reflection([response for response in short_term_reflection_tuple[0]])
#             # Mutate
#             mutated_population = self.mutate()
#             # Evaluate
#             self.population.extend(self.evaluate_population(mutated_population))
#             # Update
#             self.update_iter()

#         return self.best_code_overall, self.best_code_path_overall

from typing import Optional
import logging
import subprocess
import numpy as np
import os
from omegaconf import DictConfig
from concurrent.futures import ThreadPoolExecutor, as_completed
from utils.utils import *
from utils.llm_client.base import BaseClient
import os
import json
import os
import json
import random
import numpy as np
import torch
from sklearn.metrics.pairwise import cosine_similarity
from transformers import AutoTokenizer, AutoModel
import random
from transformers import GPT2TokenizerFast
from sklearn.metrics.pairwise import cosine_similarity
from transformers import AutoTokenizer
from openai import OpenAI
from collections import Counter
from tqdm import tqdm
import tiktoken
import spacy
from transformers import T5Tokenizer, T5ForConditionalGeneration
from tqdm import tqdm  # 
import os
import concurrent.futures
class ReEvo:
    def __init__(
        self, 
        cfg: DictConfig, 
        root_dir: str, 
        generator_llm: BaseClient, 
        reflector_llm: Optional[BaseClient] = None,
        
        # Support setting different LLMs for each of the four operators: 
        # Short-term Reflection, Long-term Reflection, Crossover, Mutation
        short_reflector_llm: Optional[BaseClient] = None,
        long_reflector_llm: Optional[BaseClient] = None,
        crossover_llm: Optional[BaseClient] = None,
        mutation_llm: Optional[BaseClient] = None
    ) -> None:
        self.cfg = cfg
        self.generator_llm = generator_llm
        self.reflector_llm = reflector_llm or generator_llm

        self.short_reflector_llm = short_reflector_llm or self.reflector_llm
        self.long_reflector_llm = long_reflector_llm or self.reflector_llm
        self.crossover_llm = crossover_llm or generator_llm
        self.mutation_llm = mutation_llm or generator_llm

        self.root_dir = root_dir
        
        self.mutation_rate = cfg.mutation_rate
        self.iteration = 0
        self.function_evals = 0
        self.elitist = None
        self.long_term_reflection_str = ""
        self.best_obj_overall = None
        self.best_code_overall = None
        self.best_code_path_overall = None
        
        self.init_prompt()
        self.init_population()



    def init_prompt(self) -> None:
        self.problem = self.cfg.problem.problem_name
        self.problem_desc = self.cfg.problem.description
        self.problem_size = self.cfg.problem.problem_size
        self.func_name = self.cfg.problem.func_name
        self.obj_type = self.cfg.problem.obj_type
        self.problem_type = self.cfg.problem.problem_type
        
        logging.info("Problem: " + self.problem)
        logging.info("Problem description: " + self.problem_desc)
        logging.info("Function name: " + self.func_name)
        
        self.prompt_dir = f"{self.root_dir}/prompts"
        self.output_file = f"{self.root_dir}/problems/{self.problem}/gpt.py"
        
        # Loading all text prompts
        # Problem-specific prompt components
        prompt_path_suffix = "_black_box" if self.problem_type == "black_box" else ""
        problem_prompt_path = f'{self.prompt_dir}/{self.problem}{prompt_path_suffix}'
        self.seed_func = file_to_string(f'{problem_prompt_path}/seed_func_pomo.txt')
        self.func_signature = file_to_string(f'{problem_prompt_path}/func_signature_pomo.txt')
        self.func_desc = file_to_string(f'{problem_prompt_path}/func_desc_pomo.txt')
        if os.path.exists(f'{problem_prompt_path}/external_knowledge_pomo.txt'):
            self.external_knowledge = file_to_string(f'{problem_prompt_path}/external_knowledge_pomo.txt')
            self.long_term_reflection_str = self.external_knowledge
        else:
            self.external_knowledge = ""
        
        
        # Common prompts
        self.system_generator_prompt = file_to_string(f'{self.prompt_dir}/common/system_generator_pomo.txt')
        self.system_reflector_prompt = file_to_string(f'{self.prompt_dir}/common/system_reflector_pomo.txt')
        self.user_reflector_st_prompt = file_to_string(f'{self.prompt_dir}/common/user_reflector_st_pomo.txt') if self.problem_type != "black_box" else file_to_string(f'{self.prompt_dir}/common/user_reflector_st_black_box_pomo.txt') # shrot-term reflection
        self.user_reflector_lt_prompt = file_to_string(f'{self.prompt_dir}/common/user_reflector_lt_pomo.txt') # long-term reflection
        self.crossover_prompt = file_to_string(f'{self.prompt_dir}/common/crossover_pomo.txt')
        self.mutation_prompt = file_to_string(f'{self.prompt_dir}/common/mutation_pomo.txt')
        self.user_generator_prompt = file_to_string(f'{self.prompt_dir}/common/user_generator_pomo.txt').format(
            func_name=self.func_name, 
            problem_desc=self.problem_desc,
            func_desc=self.func_desc,
            )
        self.seed_prompt = file_to_string(f'{self.prompt_dir}/common/seed_pomo.txt').format(
            seed_func=self.seed_func,
            func_name=self.func_name,
        )

        # Flag to print prompts
        self.print_crossover_prompt = True # Print crossover prompt for the first iteration
        self.print_mutate_prompt = True # Print mutate prompt for the first iteration
        self.print_short_term_reflection_prompt = True # Print short-term reflection prompt for the first iteration
        self.print_long_term_reflection_prompt = True # Print long-term reflection prompt for the first iteration

    def generate_paper_summary(self, input_file_path, output_file_path, api_key, max_tokens=512, max_input_tokens=4096, progress_file="progress.json", num_workers=5):
       # If the summary file already exists, load it directly and return it
        if os.path.exists(output_file_path):
            with open(output_file_path, 'r', encoding='utf-8') as file:
                return file.read()

        # Step 1: Read the JSON file and extract the content field of all articles
        with open(input_file_path, 'r', encoding='utf-8') as file:
            data = json.load(file)

        content_list = []
        for paper_id, paper_info in data["cs_paper_info"].items():
            content_list.append(paper_info["content"])

        content = "\n\n".join(content_list)

        client = OpenAI(api_key=api_key, base_url="https://api.agicto.cn/v1")

        # Split the content into multiple segments for processing
        chunks = [content[i:i + max_input_tokens] for i in range(0, len(content), max_input_tokens)]

        if os.path.exists(progress_file):
            with open(progress_file, 'r', encoding='utf-8') as pf:
                progress_data = json.load(pf)
            start_idx = progress_data.get("last_processed_chunk", 0)
        else:
            start_idx = 0

        summary_parts = []

        with tqdm(total=len(chunks), desc="Generating summaries", position=0, ncols=100) as pbar:
            with ThreadPoolExecutor(max_workers=num_workers) as executor:
                futures = []
                for i, chunk in enumerate(chunks[start_idx:], start=start_idx):
                    input_tokens = len(chunk.split())  # Calculate the number of input tokens in the current segment
                    available_tokens = max_tokens  # Reserve the maximum number of tokens generated per request
                    
                    # If the input tokens plus the generated tokens exceed the maximum limit, the number of tokens generated will be reduced
                    if input_tokens + available_tokens > 4096:
                        available_tokens = 4096 - input_tokens

                    futures.append(executor.submit(self.generate_summary_for_chunk, client, chunk, available_tokens))

                for future in as_completed(futures):
                    summary_parts.append(future.result())
                    pbar.update(1)  

        # Step 3: Merge Summaries (Deduplicate, Transition)
        full_summary = self.merge_summaries(summary_parts)

        with open(output_file_path, 'w', encoding='utf-8') as file:
            file.write(full_summary)


        return full_summary    

    def generate_summary_for_chunk(self, client, chunk, available_tokens):
        """
        Generate a summary of a single block of text.  
        """
        chat_completion = client.chat.completions.create(
            messages=[{
                "role": "user",
                "content": f"As an academic research expert, you are very good at summarizing. Now please read the literature carefully and write an authoritative and concise summary with no more than {available_tokens}.At the same time, the shorter the better, under the premise of fully realizing the main idea. The literature is as follows:\n{chunk}",
            }],
            model="gpt-3.5-turbo",
        )
        # Get the generated summary content
        return chat_completion.choices[0].message.content.strip()

    def merge_summaries(self, summary_parts):
        """
        Merge multiple summary sections to deduplicate and ensure coherence.

        Parameter:
        summary_parts: A list of summary content generated by each section.

        Return:
        str: The final summary content after the merger.
        """
        # Remove duplicate content
        merged_summary = self.remove_duplicates(summary_parts)

        # Add transition sentences to ensure logical coherence between summaries
        final_summary = ""
        for i, part in enumerate(merged_summary):
            if i == 0:
                final_summary += part
            else:
                final_summary += f"\n\nIn addition, another part of the content is summarized as: {part}"
        
        return final_summary

    def remove_duplicates(self, parts):
        """
        Remove duplicate information from the summary.

        Parameter:
        parts (list): A list of summary content generated by each section.

        Return:
        list: A list of the deduplicated summary contents.
        """
        # Use Counter to count the digest and remove the parts that are frequently repeated
        counter = Counter(parts)
        unique_parts = list(counter.keys())
        
        return unique_parts

    def load_resources(self):
        nlp = spacy.load("en_core_web_sm")
        nlp.max_length = 2_500_0000  # Increase the maximum text length limit
        
        t5_model_path = f'{self.root_dir}/t5-base'  
        if not os.path.exists(t5_model_path):
            print(f"Error: The model path {t5_model_path} does not exist.")
            exit(1)
        
        # tokenizer = T5Tokenizer.from_pretrained(t5_model_path)
        tokenizer = T5Tokenizer.from_pretrained(str(t5_model_path))
        model = T5ForConditionalGeneration.from_pretrained(t5_model_path)
    
        return nlp, tokenizer, model

    def split_text_into_chunks(self,text, tokenizer, max_chunk_size=1024):
        """
        Split the text into smaller chunks, each up to max_chunk_size tokens
        """
        print("Splitting text into chunks...")
        chunks = []
        tokens = tokenizer.encode(text)
        start = 0
        while start < len(tokens):
            end = min(start + max_chunk_size, len(tokens))
            chunk = tokenizer.decode(tokens[start:end], skip_special_tokens=True)
            chunks.append(chunk)
            start = end
        print(f"Text split into {len(chunks)} chunks.")
        return chunks
    
    def summarize_text(self,text, tokenizer, model):
        print(f"Generating summary for chunk with length: {len(tokenizer.encode(text))} tokens...")
        
        inputs = tokenizer.encode("summarize: " + text, return_tensors="pt", max_length=1024, truncation=True)
        
        summary_ids = model.generate(inputs, max_length=300, num_beams=4, no_repeat_ngram_size=2, early_stopping=True)
        
        summary = tokenizer.decode(summary_ids[0], skip_special_tokens=True)
        print("Summary generated for the chunk.")
        return summary    

    def process_file(self, file_path, tokenizer, model, summary_file_path):
        # Check if the summary file already exists
        if os.path.exists(summary_file_path):
            with open(summary_file_path, 'r', encoding='utf-8') as summary_file:
                summary_content = summary_file.read()
            return summary_content

        # If the summary file doesn't exist, process the text
        with open(file_path, 'r', encoding='utf-8') as file:
            content = file.read()
        
        # Split text into manageable chunks
        chunks = self.split_text_into_chunks(content, tokenizer)

        # Process the chunks in parallel
        summaries = []
        with concurrent.futures.ProcessPoolExecutor() as executor:
            futures = [executor.submit(self.summarize_text, chunk, tokenizer, model) for chunk in chunks]
            
            # Wait for all tasks to finish and collect results
            for future in tqdm(concurrent.futures.as_completed(futures), total=len(futures), desc="Processing text chunks"):
                summary = future.result()
                summaries.append(summary)

        # Merge all summaries into a single final summary
        final_summary = " ".join(summaries)
        
        # Encode the final summary and check its token length
        final_summary_tokens = tokenizer.encode(final_summary)

        
        # Save the final summary to the file for future use
        with open(summary_file_path, 'w', encoding='utf-8') as summary_file:
            summary_file.write(final_summary)
        
        return final_summary

    def init_population(self) -> None:
        # Generate paper summary if needed
        full_summary = self.generate_paper_summary(
            input_file_path=f'{self.root_dir}/TSP_database/TSP_similar_papers_pro.json',  # Input JSON file
            output_file_path=f'{self.root_dir}/TSP_database/summary.txt',  # Output summary file
            api_key= self.cfg.llm_client.api_key  # API key for OpenAI
        )

        nlp, tokenizer, model = self.load_resources()
        final_summary = self.process_file(file_path = f'{self.root_dir}/TSP_database/summary.txt',tokenizer = tokenizer, model = model,summary_file_path=f"{self.root_dir}/TSP_database/final_summary.txt")
        # Evaluate the seed function, and set it as Elite
        logging.info("Evaluating seed function...")
        code = extract_code_from_generator(self.seed_func).replace("v1", "v2")
        seed_ind = {
            "stdout_filepath": f"problem_iter{self.iteration}_stdout0.txt",
            "code_path": f"problem_iter{self.iteration}_code0.py",
            "code": code,
            "response_id": 0,
        }
        self.seed_ind = seed_ind
        self.population = self.evaluate_population([seed_ind])

        # If seed function is invalid, stop
        if not self.seed_ind["exec_success"]:
            raise RuntimeError(f"Seed function is invalid. Please check the stdout file in {os.getcwd()}.")

        self.update_iter()
        
        # Generate responses using the paper summary as additional information
        system = self.system_generator_prompt
        user = self.user_generator_prompt + "\n" + self.seed_prompt + "\n" + self.long_term_reflection_str + "Here is a summary of relevant papers:\n"+ final_summary       
        messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
        logging.info("Initial Population Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
        
        # Increase temperature for more diversity in the initial population
        responses = self.generator_llm.multi_chat_completion([messages], self.cfg.init_pop_size, temperature = self.generator_llm.temperature + 0.3) 
        
        # Convert responses to population individuals
        population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(responses)]
        
        # Run code and evaluate population
        population = self.evaluate_population(population)

        # Update iteration
        self.population = population
        self.update_iter()        

    
    def response_to_individual(self, response: str, response_id: int, file_name: str=None) -> dict:
        """
        Convert response to individual
        """
        # Write response to file
        file_name = f"problem_iter{self.iteration}_response{response_id}.txt" if file_name is None else file_name + ".txt"
        with open(file_name, 'w') as file:
            file.writelines(response + '\n')

        code = extract_code_from_generator(response)

        # Extract code and description from response
        std_out_filepath = f"problem_iter{self.iteration}_stdout{response_id}.txt" if file_name is None else file_name + "_stdout.txt"
        individual = {
            "stdout_filepath": std_out_filepath,
            "code_path": f"problem_iter{self.iteration}_code{response_id}.py",
            "code": code,
            "response_id": response_id,
        }
        return individual

    def mark_invalid_individual(self, individual: dict, traceback_msg: str) -> dict:
        """
        Mark an individual as invalid.
        """
        individual["exec_success"] = False
        individual["obj"] = float("inf")
        individual["traceback_msg"] = traceback_msg
        return individual


    def evaluate_population(self, population: list[dict]) -> list[float]:
        """
        Evaluate population by running code in parallel and computing objective values.
        """
        if population == []:
            return []        
        inner_runs = []
        # Run code to evaluate
        for response_id in range(len(population)):
            self.function_evals += 1
            # Skip if response is invalid
            if population[response_id]["code"] is None:
                population[response_id] = self.mark_invalid_individual(population[response_id], "Invalid response!")
                inner_runs.append(None)
                continue
            
            logging.info(f"Iteration {self.iteration}: Running Code {response_id}")
            
            try:
                process = self._run_code(population[response_id], response_id)
                inner_runs.append(process)
            except Exception as e: # If code execution fails
                logging.info(f"Error for response_id {response_id}: {e}")
                population[response_id] = self.mark_invalid_individual(population[response_id], str(e))
                inner_runs.append(None)
        # Update population with objective values
        for response_id, inner_run in enumerate(inner_runs):
            if inner_run is None: # If code execution fails, skip
                continue
            try:
                inner_run.communicate(timeout=self.cfg.timeout) # Wait for code execution to finish
            except subprocess.TimeoutExpired as e:
                logging.info(f"Error for response_id {response_id}: {e}")
                population[response_id] = self.mark_invalid_individual(population[response_id], str(e))
                inner_run.kill()
                continue
            individual = population[response_id]
            stdout_filepath = individual["stdout_filepath"]
            with open(stdout_filepath, 'r') as f:  # read the stdout file
                stdout_str = f.read() 
            traceback_msg = filter_traceback(stdout_str)
            individual = population[response_id]
            # Store objective value for each individual
            if traceback_msg == '': # If execution has no error
                try:
                    individual["obj"] = float(stdout_str.split('\n')[-2]) if self.obj_type == "min" else -float(stdout_str.split('\n')[-2])
                    individual["exec_success"] = True
                except:
                    population[response_id] = self.mark_invalid_individual(population[response_id], "Invalid std out / objective value!")
            else: # Otherwise, also provide execution traceback error feedback
                population[response_id] = self.mark_invalid_individual(population[response_id], traceback_msg)
            logging.info(f"Iteration {self.iteration}, response_id {response_id}: Objective value: {individual['obj']}")
        return population


    def _run_code(self, individual: dict, response_id) -> subprocess.Popen:
        """
        Write code into a file and run eval script.
        """
        logging.debug(f"Iteration {self.iteration}: Processing Code Run {response_id}")
        
        with open(self.output_file, 'w') as file:
            file.writelines(individual["code"] + '\n')

        # Execute the python file with flags
        with open(individual["stdout_filepath"], 'w') as f:
            eval_file_path = f'{self.root_dir}/problems/{self.problem}/eval.py' if self.problem_type != "black_box" else f'{self.root_dir}/problems/{self.problem}/eval_black_box.py' 
            process = subprocess.Popen(['python', '-u', eval_file_path, f'{self.problem_size}', self.root_dir, "train"],
                                        stdout=f, stderr=f)

        block_until_running(individual["stdout_filepath"], log_status=True, iter_num=self.iteration, response_id=response_id)
        return process

    
    def update_iter(self) -> None:
        """
        Update after each iteration
        """
        population = self.population
        if population == []:
            return
        objs = [individual["obj"] for individual in population]
        best_obj, best_sample_idx = min(objs), np.argmin(np.array(objs))
        
        # update best overall
        if self.best_obj_overall is None or best_obj < self.best_obj_overall:
            self.best_obj_overall = best_obj
            self.best_code_overall = population[best_sample_idx]["code"]
            self.best_code_path_overall = population[best_sample_idx]["code_path"]
        
        # update elitist
        if self.elitist is None or best_obj < self.elitist["obj"]:
            self.elitist = population[best_sample_idx]
            logging.info(f"Iteration {self.iteration}: Elitist: {self.elitist['obj']}")
        
        logging.info(f"Iteration {self.iteration} finished...")
        logging.info(f"Best obj: {self.best_obj_overall}, Best Code Path: {self.best_code_path_overall}")
        logging.info(f"Function Evals: {self.function_evals}")
        self.iteration += 1
        

    def split_into_chunks(self, text: str, chunk_token_size: int = 512) -> list[str]:
        """
        Use the tokenizer of MiniLM (reevo/all-MiniLM-L6-v2) to split the text into multiple chunks of a specified size.

        Args:
        text (str): The text to be split
        chunk_token_size (int): Maximum number of tokens per block (default 512)

        Returns:
        list[str]: A list of split blocks of text
        """
        tokenizer = AutoTokenizer.from_pretrained("all-MiniLM-L6-v2")
        
        tokens = tokenizer.encode(text)
        chunks = []
        
        # Split the list of tokens by chunk_token_size
        for i in range(0, len(tokens), chunk_token_size):
            chunk_tokens = tokens[i:i + chunk_token_size]
            chunk_text = tokenizer.decode(chunk_tokens, skip_special_tokens=True)
            chunks.append(chunk_text)
        
        return chunks

    def split_papers_contents(self, chunk_token_size: int = 512) -> dict:
        """
        Read each paper under cs_paper_info in TSP_similar_papers_pro.json file,
        Split the content of each paper (512 tokens each) and save it as a new JSON file.

        Parameter:
        papers_file_path: The path to the original paper data JSON file
        output_file_path: Path to the output file of the split data (JSON format)
        chunk_token_size: The number of tokens each block contains

        Return:
        dict: A new dictionary containing the information of the split paper, with the key being the paper ID and the value being the paper information dictionary,
        Include the title, the original content (optional), and the split chunks
        """
    # If the output file already exists, it will be loaded back directly
        papers_file_path: str = f"{self.root_dir}/TSP_database/TSP_similar_papers_pro.json"
        output_file_path: str = f"{self.root_dir}/TSP_database/TSP_papers_chunks.json"
        if os.path.exists(output_file_path):
            print(f"Loading existing paper chunks from {output_file_path}")
            with open(output_file_path, "r", encoding="utf-8") as f:
                papers_chunks = json.load(f)
            return papers_chunks

        os.makedirs(os.path.dirname(output_file_path), exist_ok=True)
        
        with open(papers_file_path, "r", encoding="utf-8") as f:
            data = json.load(f)
        
        cs_paper_info = data.get("cs_paper_info", {})
        papers_chunks = {}
        for paper_id, paper in cs_paper_info.items():
            content = paper.get("content", "")
            if content:
                chunks = self.split_into_chunks(content, chunk_token_size)
            else:
                chunks = []
            
            papers_chunks[paper_id] = {
                "id": paper.get("id"),
                "title": paper.get("title"),
                "chunks": chunks
            }
        
        with open(output_file_path, "w", encoding="utf-8") as f:
            json.dump(papers_chunks, f, ensure_ascii=False, indent=2)
        
        print(f"Saved paper chunks for {len(papers_chunks)} papers to {output_file_path}")
        return papers_chunks
        
    def rank_select(self, population: list[dict]) -> list[dict]:
        """
        Rank-based selection, select individuals with probability proportional to their rank.
        """
        if self.problem_type == "black_box":
            population = [individual for individual in population if individual["exec_success"] and individual["obj"] < self.seed_ind["obj"]]
        else:
            population = [individual for individual in population if individual["exec_success"]]
        if len(population) < 2:
            return None
        # Sort population by objective value
        population = sorted(population, key=lambda x: x["obj"])
        ranks = [i for i in range(len(population))]
        probs = [1 / (rank + 1 + len(population)) for rank in ranks]
        # Normalize probabilities
        probs = [prob / sum(probs) for prob in probs]
        selected_population = []
        trial = 0
        while len(selected_population) < 2 * self.cfg.pop_size:
            trial += 1
            parents = np.random.choice(population, size=2, replace=False, p=probs)
            if parents[0]["obj"] != parents[1]["obj"]:
                selected_population.extend(parents)
            if trial > 1000:
                return None
        return selected_population
    
    
    def random_select(self, population: list[dict]) -> list[dict]:
        """
        Random selection, select individuals with equal probability.
        """
        selected_population = []
        # Eliminate invalid individuals
        if self.problem_type == "black_box":
            population = [individual for individual in population if individual["exec_success"] and individual["obj"] < self.seed_ind["obj"]]
        else:
            population = [individual for individual in population if individual["exec_success"]]
        # if len(population) < 2:
        #     return None
        trial = 0
        while len(selected_population) < 2 * self.cfg.pop_size:
            trial += 1
            parents = np.random.choice(population, size=2, replace=False)
            # If two parents have the same objective value, consider them as identical; otherwise, add them to the selected population
            if parents[0]["obj"] != parents[1]["obj"]:
                selected_population.extend(parents)
            if trial > 1000:
                return None
        return selected_population

    def gen_short_term_reflection_prompt(self, ind1: dict, ind2: dict) -> tuple[list[dict], str, str]:
        """
        Short-term reflection before crossovering two individuals.
        """
        if ind1["obj"] == ind2["obj"]:
            print(ind1["code"], ind2["code"])
            raise ValueError("Two individuals to crossover have the same objective value!")
        # Determine which individual is better or worse
        if ind1["obj"] < ind2["obj"]:
            better_ind, worse_ind = ind1, ind2
        elif ind1["obj"] > ind2["obj"]:
            better_ind, worse_ind = ind2, ind1

        worse_code = filter_code(worse_ind["code"])
        better_code = filter_code(better_ind["code"])
        
        system = self.system_reflector_prompt
        user = self.user_reflector_st_prompt.format(
            func_name = self.func_name,
            func_desc = self.func_desc,
            problem_desc = self.problem_desc,
            worse_code=worse_code,
            better_code=better_code
            )
        message = [{"role": "system", "content": system}, {"role": "user", "content": user}]
        
        # Print reflection prompt for the first iteration
        if self.print_short_term_reflection_prompt:
                logging.info("Short-term Reflection Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
                self.print_short_term_reflection_prompt = False
        return message, worse_code, better_code


    def short_term_reflection(self, population: list[dict]) -> tuple[list[list[dict]], list[str], list[str]]:
        """
        Short-term reflection before crossovering two individuals.
        """
        messages_lst = []
        worse_code_lst = []
        better_code_lst = []
        for i in range(0, len(population), 2):
            # Select two individuals
            parent_1 = population[i]
            parent_2 = population[i+1]
            
            # Short-term reflection
            messages, worse_code, better_code = self.gen_short_term_reflection_prompt(parent_1, parent_2)
            messages_lst.append(messages)
            worse_code_lst.append(worse_code)
            better_code_lst.append(better_code)
        
        # Asynchronously generate responses
        response_lst = self.short_reflector_llm.multi_chat_completion(messages_lst)
        return response_lst, worse_code_lst, better_code_lst
    
    def long_term_reflection(self, short_term_reflections: list[str]) -> None:
        """
        Long-term reflection before mutation.
        """
        system = self.system_reflector_prompt
        user = self.user_reflector_lt_prompt.format(
            problem_desc = self.problem_desc,
            prior_reflection = self.long_term_reflection_str,
            new_reflection = "\n".join(short_term_reflections),
            )
        messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
        
        if self.print_long_term_reflection_prompt:
            logging.info("Long-term Reflection Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
            self.print_long_term_reflection_prompt = False
        
        self.long_term_reflection_str = self.long_reflector_llm.multi_chat_completion([messages])[0]
        
        # Write reflections to file
        file_name = f"problem_iter{self.iteration}_short_term_reflections.txt"
        with open(file_name, 'w') as file:
            file.writelines("\n".join(short_term_reflections) + '\n')
        
        file_name = f"problem_iter{self.iteration}_long_term_reflection.txt"
        with open(file_name, 'w') as file:
            file.writelines(self.long_term_reflection_str + '\n')


    def crossover(self, short_term_reflection_tuple: tuple[list[list[dict]], list[str], list[str]]) -> list[dict]:
        reflection_content_lst, worse_code_lst, better_code_lst = short_term_reflection_tuple
        messages_lst = []
        for reflection, worse_code, better_code in zip(reflection_content_lst, worse_code_lst, better_code_lst):
            # Crossover
            system = self.system_generator_prompt
            func_signature0 = self.func_signature.format(version=0)
            func_signature1 = self.func_signature.format(version=1)
            user = self.crossover_prompt.format(
                user_generator = self.user_generator_prompt,
                func_signature0 = func_signature0,
                func_signature1 = func_signature1,
                worse_code = worse_code,
                better_code = better_code,
                reflection = reflection,
                func_name = self.func_name,
            )
            messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
            messages_lst.append(messages)
            
            # Print crossover prompt for the first iteration
            if self.print_crossover_prompt:
                logging.info("Crossover Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
                self.print_crossover_prompt = False
        
        # Asynchronously generate responses
        response_lst = self.crossover_llm.multi_chat_completion(messages_lst)
        crossed_population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(response_lst)]

        assert len(crossed_population) == self.cfg.pop_size
        return crossed_population


    def mutate(self) -> list[dict]:
        """Elitist-based mutation. We only mutate the best individual to generate n_pop new individuals."""
        system = self.system_generator_prompt
        func_signature1 = self.func_signature.format(version=1) 
        user = self.mutation_prompt.format(
            user_generator = self.user_generator_prompt,
            reflection = self.long_term_reflection_str + self.external_knowledge,
            func_signature1 = func_signature1,
            elitist_code = filter_code(self.elitist["code"]),
            func_name = self.func_name,
        )
        messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
        if self.print_mutate_prompt:
            logging.info("Mutation Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
            self.print_mutate_prompt = False
        responses = self.mutation_llm.multi_chat_completion([messages], int(self.cfg.pop_size * self.mutation_rate))
        population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(responses)]
        return population


#     def rag_operator(self, short_term_reflection_tuple: tuple[list[list[dict]], list[str], list[str]], 
#                     long_term_reflection_str: str) -> list[dict]:

#         chunks_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks.json"
#         embeddings_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks_embeddings.npy"
#         index_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks_index.npy"

#         if not os.path.exists(chunks_file):
#             raise FileNotFoundError(f"Knowledge chunks file not found: {chunks_file}")
        
#         with open(chunks_file, "r", encoding="utf-8") as f:
#             papers_dict = json.load(f)

#         all_chunks = []
#         for paper in papers_dict.values():
#             if "chunks" in paper:
#                 all_chunks.extend(paper["chunks"])

#         tokenizer = AutoTokenizer.from_pretrained(f"{self.root_dir}/all-MiniLM-L6-v2")
#         model = AutoModel.from_pretrained(f"{self.root_dir}/all-MiniLM-L6-v2")
#         model.eval()  

#         expected_D = 384  

#         # Load or compute the embedding of each chunk
#         if os.path.exists(embeddings_file):
#             chunks_embeddings = np.load(embeddings_file, allow_pickle=True).item()
#         else:
#             chunks_embeddings = {}
#             for chunk in all_chunks:
#                 # Use the model to calculate the embedding of each chunk
#                 inputs = tokenizer(chunk, return_tensors="pt", padding=True, truncation=True)
#                 with torch.no_grad():
#                     output = model(**inputs)
#                 chunk_embedding = output.last_hidden_state.mean(dim=1).cpu().numpy()  
#                 chunks_embeddings[chunk] = chunk_embedding.tolist()  
#             os.makedirs(os.path.dirname(embeddings_file), exist_ok=True)
#             np.save(embeddings_file, chunks_embeddings)

#         # Load or build an index matrix
#         index_matrix = None
#         if os.path.exists(index_file):
#             loaded_index = np.load(index_file, allow_pickle=True)
#             index_data = loaded_index.item()
#             all_chunks_ordered, index_matrix_loaded = index_data.get("all_chunks"), index_data.get("index_matrix")
#             # Check whether the number of chunks is the same as that of the embedding dimension
#             if len(all_chunks_ordered) == len(all_chunks) and index_matrix_loaded.shape[1] == expected_D:
#                 index_matrix = index_matrix_loaded

#         if index_matrix is None:
#             embeddings_list = []
#             for chunk in all_chunks:
#                 emb = np.array(chunks_embeddings[chunk]).reshape(1, -1)
#                 # If the dimension is incorrect, the embedding is recalculated
#                 if emb.shape[1] != expected_D:
#                     inputs = tokenizer(chunk, return_tensors="pt", padding=True, truncation=True)
#                     with torch.no_grad():
#                         output = model(**inputs)
#                     emb = output.last_hidden_state.mean(dim=1).cpu().numpy()  
#                 embeddings_list.append(emb)
#             # Use vstack to stack all embeddings into a 2D matrix with the shape (N, D)
#             index_matrix = np.vstack(embeddings_list)
#             index_data = {"all_chunks": all_chunks, "index_matrix": index_matrix}
#             os.makedirs(os.path.dirname(index_file), exist_ok=True)
#             np.save(index_file, index_data)

#         client = OpenAI(
#             api_key=self.cfg.llm_client.api_key,  
#             base_url="https://api.agicto.cn/v1"  
#         )

#         reflection_content_lst, worse_code_lst, better_code_lst = short_term_reflection_tuple
#         population = []

#         for reflection, worse_code, better_code in zip(reflection_content_lst, worse_code_lst, better_code_lst):
            
#             if isinstance(reflection, list):
#                 updated_reflection = reflection.copy()
#             else:
#                 updated_reflection = [{"original": reflection}]
            
#             # Combine all Reflection content to generate 4O-Mini prompts
#             combined_reflection = " ".join([str(item) for item in reflection]) if isinstance(reflection, list) else str(reflection)

#             # Generate prompts for improvement suggestions to ensure that they are not duplicated with the Reflection
#             prompt = (
#                 "I am designing a heuristic for solving the TSP problem. Please generate a possible improvement for me."
#                 "Please be careful not to duplicate or be similar to the known reflections below."
#                 "The following are known reflections:\n"
#                 f"{long_term_reflection_str}\n"  # Long-term reflection
#                 f"{combined_reflection}"  # Current Reflection Content
#                 "Please provide suggestions for improvement other than the above.Please give your comments directly, no more than 20 words and without any extra words."
#             )
            
#             chat_completion = client.chat.completions.create(
#                 messages=[{"role": "user", "content": prompt}],
#                 model="gpt-3.5-turbo"
#             )
            
#             suggestion = chat_completion.choices[0].message.content
            

#             # Calculate the embedding of the generated recommendations
#             suggestion_inputs = tokenizer(suggestion, return_tensors="pt", padding=True, truncation=True)
#             with torch.no_grad():
#                 suggestion_output = model(**suggestion_inputs)
#             suggestion_embedding = suggestion_output.last_hidden_state.mean(dim=1).cpu().numpy()  # (1, D)
#             suggestion_embedding /= np.linalg.norm(suggestion_embedding)

#             # Calculate the cosine similarity to each chunk
#             similarities = cosine_similarity(index_matrix, suggestion_embedding).flatten()

#             # Select the two blocks with the highest similarity
#             top_indices = np.argsort(similarities)[-2:][::-1]
#             top_chunks = [(all_chunks[i], similarities[i]) for i in top_indices]


#             # Generate a population using a crossover-like structure
#             # Generate a new population based on the two most relevant chunks and their suggestions
#             population_prompt = (
#                 """You are an expert in the domain of optimization heuristics. Your task is to design heuristics that can effectively solve optimization problems.
# Your response outputs Python code and nothing else. Format your code as a Python code string: "```python ... ```".\n"""
#                 """Write a heuristics function for Solving Traveling Salesman Problem (TSP) via stochastic solution sampling following "heuristics". TSP requires finding the shortest path that visits all given nodes and returns to the starting node.
# The `heuristics` function takes as input a distance matrix, and returns prior indicators of how promising it is to include each edge in a solution. The return is of the same shape as the input.\n"""
#                 f"Here are some resources you can refer to:\n{top_chunks}\n"
#                 "Please write an improved function `heuristics_v2`, according to the resources. Output code only and enclose your code with Python code block: ```python ... ```.."
#                 "Please give your answer directly according to the example, without any nonsense"
#             )

#             population_response = client.chat.completions.create(
#                 messages=[{"role": "user", "content": population_prompt}],
#                 model="gpt-3.5-turbo"
#             )

#             population_content = population_response.choices[0].message.content

#             rag_population = [self.response_to_individual(population_content, 0)]


#         return rag_population

    def rag_operator(self, short_term_reflection_tuple: tuple[list[list[dict]], list[str], list[str]], 
                    long_term_reflection_str: str) -> list[dict]:

        chunks_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks.json"
        embeddings_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks_embeddings.npy"
        index_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks_index.npy"

        if not os.path.exists(chunks_file):
            raise FileNotFoundError(f"Knowledge chunks file not found: {chunks_file}")
        
        with open(chunks_file, "r", encoding="utf-8") as f:
            papers_dict = json.load(f)

        all_chunks = []
        for paper in papers_dict.values():
            if "chunks" in paper:
                all_chunks.extend(paper["chunks"])

        tokenizer = AutoTokenizer.from_pretrained(f"{self.root_dir}/all-MiniLM-L6-v2")
        model = AutoModel.from_pretrained(f"{self.root_dir}/all-MiniLM-L6-v2")
        model.eval()  

        expected_D = 384  

        # Load or compute the embedding of each chunk
        if os.path.exists(embeddings_file):
            chunks_embeddings = np.load(embeddings_file, allow_pickle=True).item()
        else:
            chunks_embeddings = {}
            for chunk in all_chunks:
                # Use the model to calculate the embedding of each chunk
                inputs = tokenizer(chunk, return_tensors="pt", padding=True, truncation=True)
                with torch.no_grad():
                    output = model(**inputs)
                chunk_embedding = output.last_hidden_state.mean(dim=1).cpu().numpy()  
                chunks_embeddings[chunk] = chunk_embedding.tolist()  
            os.makedirs(os.path.dirname(embeddings_file), exist_ok=True)
            np.save(embeddings_file, chunks_embeddings)

        # Load or build an index matrix
        index_matrix = None
        if os.path.exists(index_file):
            loaded_index = np.load(index_file, allow_pickle=True)
            index_data = loaded_index.item()
            all_chunks_ordered, index_matrix_loaded = index_data.get("all_chunks"), index_data.get("index_matrix")
            # Check whether the number of chunks is the same as that of the embedding dimension
            if len(all_chunks_ordered) == len(all_chunks) and index_matrix_loaded.shape[1] == expected_D:
                index_matrix = index_matrix_loaded

        if index_matrix is None:
            embeddings_list = []
            for chunk in all_chunks:
                emb = np.array(chunks_embeddings[chunk]).reshape(1, -1)
                # If the dimension is incorrect, the embedding is recalculated
                if emb.shape[1] != expected_D:
                    inputs = tokenizer(chunk, return_tensors="pt", padding=True, truncation=True)
                    with torch.no_grad():
                        output = model(**inputs)
                    emb = output.last_hidden_state.mean(dim=1).cpu().numpy()  
                embeddings_list.append(emb)
            # Use vstack to stack all embeddings into a 2D matrix with the shape (N, D)
            index_matrix = np.vstack(embeddings_list)
            index_data = {"all_chunks": all_chunks, "index_matrix": index_matrix}
            os.makedirs(os.path.dirname(index_file), exist_ok=True)
            np.save(index_file, index_data)

        client = OpenAI(
            api_key=self.cfg.llm_client.api_key,  
            base_url="https://api.agicto.cn/v1"  
        )

        reflection_content_lst, worse_code_lst, better_code_lst = short_term_reflection_tuple
        population = []

        for reflection, worse_code, better_code in zip(reflection_content_lst, worse_code_lst, better_code_lst):
            
            if isinstance(reflection, list):
                updated_reflection = reflection.copy()
            else:
                updated_reflection = [{"original": reflection}]
            
            # Combine all Reflection content to generate 4O-Mini prompts
            combined_reflection = " ".join([str(item) for item in reflection]) if isinstance(reflection, list) else str(reflection)

            # Generate prompts for improvement suggestions to ensure that they are not duplicated with the Reflection
            prompt = (
                "I am designing a heuristic for solving the TSP problem. Please generate a possible improvement for me."
                "Please be careful not to duplicate or be similar to the known reflections below."
                "The following are known reflections:\n"
                f"{long_term_reflection_str}\n"  # Long-term reflection
                f"{combined_reflection}"  # Current Reflection Content
                "Please provide suggestions for improvement other than the above.Please give your comments directly, no more than 20 words and without any extra words."
            )
            
            chat_completion = client.chat.completions.create(
                messages=[{"role": "user", "content": prompt}],
                model="gpt-3.5-turbo"
            )
            
            suggestion = chat_completion.choices[0].message.content
            

            # Calculate the embedding of the generated recommendations
            # 选择设备
            device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
            model.to(device)  # 把模型放到 GPU

            # Tokenize suggestion 并转移到 GPU
            suggestion_inputs = tokenizer(suggestion, return_tensors="pt", padding=True, truncation=True)
            suggestion_inputs = {k: v.to(device) for k, v in suggestion_inputs.items()}

            # suggestion_inputs = tokenizer(suggestion, return_tensors="pt", padding=True, truncation=True)
            with torch.no_grad():
                suggestion_output = model(**suggestion_inputs)
            suggestion_embedding = suggestion_output.last_hidden_state.mean(dim=1).cpu().numpy()  # (1, D)
            suggestion_embedding /= np.linalg.norm(suggestion_embedding)

            # Calculate the cosine similarity to each chunk
            similarities = cosine_similarity(index_matrix, suggestion_embedding).flatten()

            # Select the two blocks with the highest similarity
            top_indices = np.argsort(similarities)[-2:][::-1]
            top_chunks = [(all_chunks[i], similarities[i]) for i in top_indices]

            # Generate a population using a crossover-like structure
            # Generate a new population based on the two most relevant chunks and their suggestions
            population_prompt = (
                """You are an expert in the domain of optimization heuristics. Your task is to design heuristics that can effectively solve optimization problems.
            Your response outputs Python code and nothing else. Format your code as a Python code string: "```python ... ```".\n"""
                """Write a decision function for solving the Traveling Salesman Problem (TSP) via the Random Repair Cycle (RRC) framework, specifically for deciding whether to accept a repaired sub-solution. The function, `decide_whether_to_repair_solution`, determines whether to incorporate a repaired sub-solution into the complete TSP solution to minimize the path length gap with the optimal solution.
            The `decide_whether_to_repair_solution` function takes as input:
            - `after_repair_sub_solution`: A tensor of shape `(batch_size, sub_length)` representing the repaired sub-solution.
            - `before_reward`: A tensor of shape `(batch_size,)` representing the reward (negative path length) before repair.
            - `after_reward`: A tensor of shape `(batch_size,)` representing the reward (negative path length) after repair.
            - `first_node_index`: An integer indicating the starting index of the sub-solution in `double_solution`.
            - `length_of_subpath`: An integer indicating the length of the sub-solution.
            - `double_solution`: A tensor of shape `(batch_size, 2 * problem_size)` representing the doubled solution.
            It returns a tensor of shape `(batch_size, problem_size)` representing the complete TSP solution after the repair decision.\n"""
                f"Here are some resources you can refer to:\n{top_chunks}\n"
                """
            Example of handling PyTorch tensor input:\n
            ```python
            import torch

            def decide_whether_to_repair_solution(after_repair_sub_solution, before_reward, after_reward,
                                                first_node_index, length_of_subpath, double_solution):

                the_whole_problem_size = int(double_solution.shape[1] / 2)
                other_part_1 = double_solution[:, :first_node_index]
                other_part_2 = double_solution[:, first_node_index + length_of_subpath:]
                origin_sub_solution = double_solution[:, first_node_index : first_node_index + length_of_subpath]

                # Sort original sub-solution for mapping
                jjj, _ = torch.sort(origin_sub_solution, dim=1, descending=False)
                index = torch.arange(jjj.shape[0])[:, None].repeat(1, jjj.shape[1])
                kkk_2 = jjj[index, after_repair_sub_solution]

                # Simulated annealing-inspired acceptance
                temperature = 0.1 * the_whole_problem_size * (1.0 - length_of_subpath / the_whole_problem_size)
                reward_diff = after_reward - before_reward  # Positive if after_reward is better (less negative)
                prob_accept = torch.exp(reward_diff / temperature)
                
                # Accept if after_reward is better or with probability based on simulated annealing
                if_repair = (after_reward > before_reward) | (torch.rand_like(before_reward) < prob_accept)
                
                # Additional condition: favor shorter sub-paths to reduce disruption
                max_subpath_ratio = 0.3  # Only accept long sub-path repairs if significantly better
                if_repair = if_repair & (
                    (length_of_subpath / the_whole_problem_size < max_subpath_ratio) |
                    (after_reward > before_reward + 0.05 * the_whole_problem_size)
                )

                # Update double_solution where repair is accepted
                double_solution[if_repair] = torch.cat(
                    (other_part_1[if_repair], kkk_2[if_repair], other_part_2[if_repair]), dim=1)

                # Extract the complete solution
                after_repair_complete_solution = double_solution[:, first_node_index:first_node_index + the_whole_problem_size]
                return after_repair_complete_solution
            ```"""
                "Please write an improved function `decide_whether_to_repair_solution`, according to the resources. Output code only and enclose your code with Python code block: ```python ... ```.")
            population_response = client.chat.completions.create(
                messages=[{"role": "user", "content": population_prompt}],
                model="gpt-3.5-turbo"
            )

            population_content = population_response.choices[0].message.content

            rag_population = [self.response_to_individual(population_content, 0)]

        return rag_population

    def evolve(self):
        print("running。。。。")
        chunks = self.split_papers_contents()
        while self.function_evals < self.cfg.max_fe:
            # If all individuals are invalid, stop
            if all([not individual["exec_success"] for individual in self.population]):
                raise RuntimeError(f"All individuals are invalid. Please check the stdout files in {os.getcwd()}.")
            # Select
            population_to_select = self.population if (self.elitist is None or self.elitist in self.population) else [self.elitist] + self.population # add elitist to population for selection
            selected_population = self.random_select(population_to_select)
            # if selected_population is None:
            #     raise RuntimeError("Selection failed. Please check the population.")
            # Short-term reflection
            short_term_reflection_tuple = self.short_term_reflection(selected_population) # (response_lst, worse_code_lst, better_code_lst)
            
            # Crossover
            crossed_population = self.crossover(short_term_reflection_tuple)
            # Evaluate
            self.population = self.evaluate_population(crossed_population)
            # Update
            self.update_iter()
            # Long-term reflection
            self.long_term_reflection([response for response in short_term_reflection_tuple[0]])
            # Mutate
            mutated_population = self.mutate()
            # Evaluate
            self.population.extend(self.evaluate_population(mutated_population))
            # Update
            self.update_iter()
            # Call the RAG operator to get the updated triplet
            if random.random() < 1.0:    
                rag_population = self.rag_operator(short_term_reflection_tuple,self.long_term_reflection_str)
                self.population = self.evaluate_population(rag_population)
                self.update_iter() 
        return self.best_code_overall, self.best_code_path_overall


# from typing import Optional
# import logging
# import subprocess
# import numpy as np
# import os
# from omegaconf import DictConfig
# from concurrent.futures import ThreadPoolExecutor, as_completed
# from utils.utils import *
# from utils.llm_client.base import BaseClient
# import os
# import json
# import os
# import json
# import random
# import numpy as np
# import torch
# from sklearn.metrics.pairwise import cosine_similarity
# from transformers import AutoTokenizer, AutoModel
# import random
# from transformers import GPT2TokenizerFast
# from sklearn.metrics.pairwise import cosine_similarity
# from transformers import AutoTokenizer
# from openai import OpenAI
# from collections import Counter
# from tqdm import tqdm
# import tiktoken
# import spacy
# from transformers import T5Tokenizer, T5ForConditionalGeneration
# from tqdm import tqdm  # 
# import os
# import concurrent.futures
# class ReEvo:
#     def __init__(
#         self, 
#         cfg: DictConfig, 
#         root_dir: str, 
#         generator_llm: BaseClient, 
#         reflector_llm: Optional[BaseClient] = None,
        
#         # Support setting different LLMs for each of the four operators: 
#         # Short-term Reflection, Long-term Reflection, Crossover, Mutation
#         short_reflector_llm: Optional[BaseClient] = None,
#         long_reflector_llm: Optional[BaseClient] = None,
#         crossover_llm: Optional[BaseClient] = None,
#         mutation_llm: Optional[BaseClient] = None
#     ) -> None:
#         self.cfg = cfg
#         self.generator_llm = generator_llm
#         self.reflector_llm = reflector_llm or generator_llm

#         self.short_reflector_llm = short_reflector_llm or self.reflector_llm
#         self.long_reflector_llm = long_reflector_llm or self.reflector_llm
#         self.crossover_llm = crossover_llm or generator_llm
#         self.mutation_llm = mutation_llm or generator_llm

#         self.root_dir = root_dir
        
#         self.mutation_rate = cfg.mutation_rate
#         self.iteration = 0
#         self.function_evals = 0
#         self.elitist = None
#         self.long_term_reflection_str = ""
#         self.best_obj_overall = None
#         self.best_code_overall = None
#         self.best_code_path_overall = None
        
#         self.init_prompt()
#         self.init_population()



#     def init_prompt(self) -> None:
#         self.problem = self.cfg.problem.problem_name
#         self.problem_desc = self.cfg.problem.description
#         self.problem_size = self.cfg.problem.problem_size
#         self.func_name = self.cfg.problem.func_name
#         self.obj_type = self.cfg.problem.obj_type
#         self.problem_type = self.cfg.problem.problem_type
        
#         logging.info("Problem: " + self.problem)
#         logging.info("Problem description: " + self.problem_desc)
#         logging.info("Function name: " + self.func_name)
        
#         self.prompt_dir = f"{self.root_dir}/prompts"
#         self.output_file = f"{self.root_dir}/problems/{self.problem}/gpt.py"
        
#         # Loading all text prompts
#         # Problem-specific prompt components
#         prompt_path_suffix = "_black_box" if self.problem_type == "black_box" else ""
#         problem_prompt_path = f'{self.prompt_dir}/{self.problem}{prompt_path_suffix}'
#         self.seed_func = file_to_string(f'{problem_prompt_path}/seed_func_pomo.txt')
#         self.func_signature = file_to_string(f'{problem_prompt_path}/func_signature_pomo.txt')
#         self.func_desc = file_to_string(f'{problem_prompt_path}/func_desc_pomo.txt')
#         if os.path.exists(f'{problem_prompt_path}/external_knowledge_pomo.txt'):
#             self.external_knowledge = file_to_string(f'{problem_prompt_path}/external_knowledge_pomo.txt')
#             self.long_term_reflection_str = self.external_knowledge
#         else:
#             self.external_knowledge = ""
        
        
#         # Common prompts
#         self.system_generator_prompt = file_to_string(f'{self.prompt_dir}/common/system_generator_pomo.txt')
#         self.system_reflector_prompt = file_to_string(f'{self.prompt_dir}/common/system_reflector_pomo.txt')
#         self.user_reflector_st_prompt = file_to_string(f'{self.prompt_dir}/common/user_reflector_st_pomo.txt') if self.problem_type != "black_box" else file_to_string(f'{self.prompt_dir}/common/user_reflector_st_black_box_pomo.txt') # shrot-term reflection
#         self.user_reflector_lt_prompt = file_to_string(f'{self.prompt_dir}/common/user_reflector_lt_pomo.txt') # long-term reflection
#         self.crossover_prompt = file_to_string(f'{self.prompt_dir}/common/crossover_pomo.txt')
#         self.mutation_prompt = file_to_string(f'{self.prompt_dir}/common/mutation_pomo.txt')
#         self.user_generator_prompt = file_to_string(f'{self.prompt_dir}/common/user_generator_pomo.txt').format(
#             func_name=self.func_name, 
#             problem_desc=self.problem_desc,
#             func_desc=self.func_desc,
#             )
#         self.seed_prompt = file_to_string(f'{self.prompt_dir}/common/seed_pomo.txt').format(
#             seed_func=self.seed_func,
#             func_name=self.func_name,
#         )

#         # Flag to print prompts
#         self.print_crossover_prompt = True # Print crossover prompt for the first iteration
#         self.print_mutate_prompt = True # Print mutate prompt for the first iteration
#         self.print_short_term_reflection_prompt = True # Print short-term reflection prompt for the first iteration
#         self.print_long_term_reflection_prompt = True # Print long-term reflection prompt for the first iteration

#     def generate_paper_summary(self, input_file_path, output_file_path, api_key, max_tokens=512, max_input_tokens=4096, progress_file="progress.json", num_workers=5):
#        # If the summary file already exists, load it directly and return it
#         if os.path.exists(output_file_path):
#             with open(output_file_path, 'r', encoding='utf-8') as file:
#                 return file.read()

#         # Step 1: Read the JSON file and extract the content field of all articles
#         with open(input_file_path, 'r', encoding='utf-8') as file:
#             data = json.load(file)

#         content_list = []
#         for paper_id, paper_info in data["cs_paper_info"].items():
#             content_list.append(paper_info["content"])

#         content = "\n\n".join(content_list)

#         client = OpenAI(api_key=api_key, base_url="https://api.agicto.cn/v1")

#         # Split the content into multiple segments for processing
#         chunks = [content[i:i + max_input_tokens] for i in range(0, len(content), max_input_tokens)]

#         if os.path.exists(progress_file):
#             with open(progress_file, 'r', encoding='utf-8') as pf:
#                 progress_data = json.load(pf)
#             start_idx = progress_data.get("last_processed_chunk", 0)
#         else:
#             start_idx = 0

#         summary_parts = []

#         with tqdm(total=len(chunks), desc="Generating summaries", position=0, ncols=100) as pbar:
#             with ThreadPoolExecutor(max_workers=num_workers) as executor:
#                 futures = []
#                 for i, chunk in enumerate(chunks[start_idx:], start=start_idx):
#                     input_tokens = len(chunk.split())  # Calculate the number of input tokens in the current segment
#                     available_tokens = max_tokens  # Reserve the maximum number of tokens generated per request
                    
#                     # If the input tokens plus the generated tokens exceed the maximum limit, the number of tokens generated will be reduced
#                     if input_tokens + available_tokens > 4096:
#                         available_tokens = 4096 - input_tokens

#                     futures.append(executor.submit(self.generate_summary_for_chunk, client, chunk, available_tokens))

#                 for future in as_completed(futures):
#                     summary_parts.append(future.result())
#                     pbar.update(1)  

#         # Step 3: Merge Summaries (Deduplicate, Transition)
#         full_summary = self.merge_summaries(summary_parts)

#         with open(output_file_path, 'w', encoding='utf-8') as file:
#             file.write(full_summary)


#         return full_summary    

#     def generate_summary_for_chunk(self, client, chunk, available_tokens):
#         """
#         Generate a summary of a single block of text.  
#         """
#         chat_completion = client.chat.completions.create(
#             messages=[{
#                 "role": "user",
#                 "content": f"As an academic research expert, you are very good at summarizing. Now please read the literature carefully and write an authoritative and concise summary with no more than {available_tokens}.At the same time, the shorter the better, under the premise of fully realizing the main idea. The literature is as follows:\n{chunk}",
#             }],
#             model="gpt-3.5-turbo",
#         )
#         # Get the generated summary content
#         return chat_completion.choices[0].message.content.strip()

#     def merge_summaries(self, summary_parts):
#         """
#         Merge multiple summary sections to deduplicate and ensure coherence.

#         Parameter:
#         summary_parts: A list of summary content generated by each section.

#         Return:
#         str: The final summary content after the merger.
#         """
#         # Remove duplicate content
#         merged_summary = self.remove_duplicates(summary_parts)

#         # Add transition sentences to ensure logical coherence between summaries
#         final_summary = ""
#         for i, part in enumerate(merged_summary):
#             if i == 0:
#                 final_summary += part
#             else:
#                 final_summary += f"\n\nIn addition, another part of the content is summarized as: {part}"
        
#         return final_summary

#     def remove_duplicates(self, parts):
#         """
#         Remove duplicate information from the summary.

#         Parameter:
#         parts (list): A list of summary content generated by each section.

#         Return:
#         list: A list of the deduplicated summary contents.
#         """
#         # Use Counter to count the digest and remove the parts that are frequently repeated
#         counter = Counter(parts)
#         unique_parts = list(counter.keys())
        
#         return unique_parts

#     def load_resources(self):
#         nlp = spacy.load("en_core_web_sm")
#         nlp.max_length = 2_500_0000  # Increase the maximum text length limit
        
#         t5_model_path = f'{self.root_dir}/t5-base'  
#         if not os.path.exists(t5_model_path):
#             print(f"Error: The model path {t5_model_path} does not exist.")
#             exit(1)
        
#         tokenizer = T5Tokenizer.from_pretrained(t5_model_path)
#         model = T5ForConditionalGeneration.from_pretrained(t5_model_path)
    
#         return nlp, tokenizer, model

#     def split_text_into_chunks(self,text, tokenizer, max_chunk_size=1024):
#         """
#         Split the text into smaller chunks, each up to max_chunk_size tokens
#         """
#         print("Splitting text into chunks...")
#         chunks = []
#         tokens = tokenizer.encode(text)
#         start = 0
#         while start < len(tokens):
#             end = min(start + max_chunk_size, len(tokens))
#             chunk = tokenizer.decode(tokens[start:end], skip_special_tokens=True)
#             chunks.append(chunk)
#             start = end
#         print(f"Text split into {len(chunks)} chunks.")
#         return chunks
    
#     def summarize_text(self,text, tokenizer, model):
#         print(f"Generating summary for chunk with length: {len(tokenizer.encode(text))} tokens...")
        
#         inputs = tokenizer.encode("summarize: " + text, return_tensors="pt", max_length=1024, truncation=True)
        
#         summary_ids = model.generate(inputs, max_length=300, num_beams=4, no_repeat_ngram_size=2, early_stopping=True)
        
#         summary = tokenizer.decode(summary_ids[0], skip_special_tokens=True)
#         print("Summary generated for the chunk.")
#         return summary    

#     def process_file(self, file_path, tokenizer, model, summary_file_path):
#         # Check if the summary file already exists
#         if os.path.exists(summary_file_path):
#             with open(summary_file_path, 'r', encoding='utf-8') as summary_file:
#                 summary_content = summary_file.read()
#             return summary_content

#         # If the summary file doesn't exist, process the text
#         with open(file_path, 'r', encoding='utf-8') as file:
#             content = file.read()
        
#         # Split text into manageable chunks
#         chunks = self.split_text_into_chunks(content, tokenizer)

#         # Process the chunks in parallel
#         summaries = []
#         with concurrent.futures.ProcessPoolExecutor() as executor:
#             futures = [executor.submit(self.summarize_text, chunk, tokenizer, model) for chunk in chunks]
            
#             # Wait for all tasks to finish and collect results
#             for future in tqdm(concurrent.futures.as_completed(futures), total=len(futures), desc="Processing text chunks"):
#                 summary = future.result()
#                 summaries.append(summary)

#         # Merge all summaries into a single final summary
#         final_summary = " ".join(summaries)
        
#         # Encode the final summary and check its token length
#         final_summary_tokens = tokenizer.encode(final_summary)

        
#         # Save the final summary to the file for future use
#         with open(summary_file_path, 'w', encoding='utf-8') as summary_file:
#             summary_file.write(final_summary)
        
#         return final_summary

#     def init_population(self) -> None:
#         # Generate paper summary if needed
#         full_summary = self.generate_paper_summary(
#             input_file_path=f'{self.root_dir}/TSP_database/TSP_similar_papers_pro.json',  # Input JSON file
#             output_file_path=f'{self.root_dir}/TSP_database/summary.txt',  # Output summary file
#             api_key= self.cfg.llm_client.api_key  # API key for OpenAI
#         )

#         nlp, tokenizer, model = self.load_resources()
#         final_summary = self.process_file(file_path = f'{self.root_dir}/TSP_database/summary.txt',tokenizer = tokenizer, model = model,summary_file_path=f"{self.root_dir}/TSP_database/final_summary.txt")
#         # Evaluate the seed function, and set it as Elite
#         logging.info("Evaluating seed function...")
#         code = extract_code_from_generator(self.seed_func).replace("v1", "v2")
#         seed_ind = {
#             "stdout_filepath": f"problem_iter{self.iteration}_stdout0.txt",
#             "code_path": f"problem_iter{self.iteration}_code0.py",
#             "code": code,
#             "response_id": 0,
#         }
#         self.seed_ind = seed_ind
#         self.population = self.evaluate_population([seed_ind])

#         # If seed function is invalid, stop
#         if not self.seed_ind["exec_success"]:
#             raise RuntimeError(f"Seed function is invalid. Please check the stdout file in {os.getcwd()}.")

#         self.update_iter()
        
#         # Generate responses using the paper summary as additional information
#         system = self.system_generator_prompt
#         user = self.user_generator_prompt + "\n" + self.seed_prompt + "\n" + self.long_term_reflection_str + "Here is a summary of relevant papers:\n"+ final_summary       
#         messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
#         logging.info("Initial Population Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
        
#         # Increase temperature for more diversity in the initial population
#         responses = self.generator_llm.multi_chat_completion([messages], self.cfg.init_pop_size, temperature = self.generator_llm.temperature + 0.3) 
        
#         # Convert responses to population individuals
#         population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(responses)]
        
#         # Run code and evaluate population
#         population = self.evaluate_population(population)

#         # Update iteration
#         self.population = population
#         self.update_iter()        

    
#     def response_to_individual(self, response: str, response_id: int, file_name: str=None) -> dict:
#         """
#         Convert response to individual
#         """
#         # Write response to file
#         file_name = f"problem_iter{self.iteration}_response{response_id}.txt" if file_name is None else file_name + ".txt"
#         with open(file_name, 'w') as file:
#             file.writelines(response + '\n')

#         code = extract_code_from_generator(response)

#         # Extract code and description from response
#         std_out_filepath = f"problem_iter{self.iteration}_stdout{response_id}.txt" if file_name is None else file_name + "_stdout.txt"
#         individual = {
#             "stdout_filepath": std_out_filepath,
#             "code_path": f"problem_iter{self.iteration}_code{response_id}.py",
#             "code": code,
#             "response_id": response_id,
#         }
#         return individual

#     def mark_invalid_individual(self, individual: dict, traceback_msg: str) -> dict:
#         """
#         Mark an individual as invalid.
#         """
#         individual["exec_success"] = False
#         individual["obj"] = float("inf")
#         individual["traceback_msg"] = traceback_msg
#         return individual


#     def evaluate_population(self, population: list[dict]) -> list[float]:
#         """
#         Evaluate population by running code in parallel and computing objective values.
#         """
#         if population == []:
#             return []        
#         inner_runs = []
#         # Run code to evaluate
#         for response_id in range(len(population)):
#             self.function_evals += 1
#             # Skip if response is invalid
#             if population[response_id]["code"] is None:
#                 population[response_id] = self.mark_invalid_individual(population[response_id], "Invalid response!")
#                 inner_runs.append(None)
#                 continue
            
#             logging.info(f"Iteration {self.iteration}: Running Code {response_id}")
            
#             try:
#                 process = self._run_code(population[response_id], response_id)
#                 inner_runs.append(process)
#             except Exception as e: # If code execution fails
#                 logging.info(f"Error for response_id {response_id}: {e}")
#                 population[response_id] = self.mark_invalid_individual(population[response_id], str(e))
#                 inner_runs.append(None)
#         # Update population with objective values
#         for response_id, inner_run in enumerate(inner_runs):
#             if inner_run is None: # If code execution fails, skip
#                 continue
#             try:
#                 inner_run.communicate(timeout=self.cfg.timeout) # Wait for code execution to finish
#             except subprocess.TimeoutExpired as e:
#                 logging.info(f"Error for response_id {response_id}: {e}")
#                 population[response_id] = self.mark_invalid_individual(population[response_id], str(e))
#                 inner_run.kill()
#                 continue
#             individual = population[response_id]
#             stdout_filepath = individual["stdout_filepath"]
#             with open(stdout_filepath, 'r') as f:  # read the stdout file
#                 stdout_str = f.read() 
#             traceback_msg = filter_traceback(stdout_str)
#             individual = population[response_id]
#             # Store objective value for each individual
#             if traceback_msg == '': # If execution has no error
#                 try:
#                     individual["obj"] = float(stdout_str.split('\n')[-2]) if self.obj_type == "min" else -float(stdout_str.split('\n')[-2])
#                     individual["exec_success"] = True
#                 except:
#                     population[response_id] = self.mark_invalid_individual(population[response_id], "Invalid std out / objective value!")
#             else: # Otherwise, also provide execution traceback error feedback
#                 population[response_id] = self.mark_invalid_individual(population[response_id], traceback_msg)
#             logging.info(f"Iteration {self.iteration}, response_id {response_id}: Objective value: {individual['obj']}")
#         return population


#     def _run_code(self, individual: dict, response_id) -> subprocess.Popen:
#         """
#         Write code into a file and run eval script.
#         """
#         logging.debug(f"Iteration {self.iteration}: Processing Code Run {response_id}")
        
#         with open(self.output_file, 'w') as file:
#             file.writelines(individual["code"] + '\n')

#         # Execute the python file with flags
#         with open(individual["stdout_filepath"], 'w') as f:
#             eval_file_path = f'{self.root_dir}/problems/{self.problem}/eval.py' if self.problem_type != "black_box" else f'{self.root_dir}/problems/{self.problem}/eval_black_box.py' 
#             process = subprocess.Popen(['python', '-u', eval_file_path, f'{self.problem_size}', self.root_dir, "train"],
#                                         stdout=f, stderr=f)

#         block_until_running(individual["stdout_filepath"], log_status=True, iter_num=self.iteration, response_id=response_id)
#         return process

    
#     def update_iter(self) -> None:
#         """
#         Update after each iteration
#         """
#         population = self.population
#         if population == []:
#             return
#         objs = [individual["obj"] for individual in population]
#         best_obj, best_sample_idx = min(objs), np.argmin(np.array(objs))
        
#         # update best overall
#         if self.best_obj_overall is None or best_obj < self.best_obj_overall:
#             self.best_obj_overall = best_obj
#             self.best_code_overall = population[best_sample_idx]["code"]
#             self.best_code_path_overall = population[best_sample_idx]["code_path"]
        
#         # update elitist
#         if self.elitist is None or best_obj < self.elitist["obj"]:
#             self.elitist = population[best_sample_idx]
#             logging.info(f"Iteration {self.iteration}: Elitist: {self.elitist['obj']}")
        
#         logging.info(f"Iteration {self.iteration} finished...")
#         logging.info(f"Best obj: {self.best_obj_overall}, Best Code Path: {self.best_code_path_overall}")
#         logging.info(f"Function Evals: {self.function_evals}")
#         self.iteration += 1
        

#     def split_into_chunks(self, text: str, chunk_token_size: int = 512) -> list[str]:
#         """
#         Use the tokenizer of MiniLM (reevo/all-MiniLM-L6-v2) to split the text into multiple chunks of a specified size.

#         Args:
#         text (str): The text to be split
#         chunk_token_size (int): Maximum number of tokens per block (default 512)

#         Returns:
#         list[str]: A list of split blocks of text
#         """
#         tokenizer = AutoTokenizer.from_pretrained("all-MiniLM-L6-v2")
        
#         tokens = tokenizer.encode(text)
#         chunks = []
        
#         # Split the list of tokens by chunk_token_size
#         for i in range(0, len(tokens), chunk_token_size):
#             chunk_tokens = tokens[i:i + chunk_token_size]
#             chunk_text = tokenizer.decode(chunk_tokens, skip_special_tokens=True)
#             chunks.append(chunk_text)
        
#         return chunks

#     def split_papers_contents(self, chunk_token_size: int = 512) -> dict:
#         """
#         Read each paper under cs_paper_info in TSP_similar_papers_pro.json file,
#         Split the content of each paper (512 tokens each) and save it as a new JSON file.

#         Parameter:
#         papers_file_path: The path to the original paper data JSON file
#         output_file_path: Path to the output file of the split data (JSON format)
#         chunk_token_size: The number of tokens each block contains

#         Return:
#         dict: A new dictionary containing the information of the split paper, with the key being the paper ID and the value being the paper information dictionary,
#         Include the title, the original content (optional), and the split chunks
#         """
#     # If the output file already exists, it will be loaded back directly
#         papers_file_path: str = f"{self.root_dir}/TSP_database/TSP_similar_papers_pro.json"
#         output_file_path: str = f"{self.root_dir}/TSP_database/TSP_papers_chunks.json"
#         if os.path.exists(output_file_path):
#             print(f"Loading existing paper chunks from {output_file_path}")
#             with open(output_file_path, "r", encoding="utf-8") as f:
#                 papers_chunks = json.load(f)
#             return papers_chunks

#         os.makedirs(os.path.dirname(output_file_path), exist_ok=True)
        
#         with open(papers_file_path, "r", encoding="utf-8") as f:
#             data = json.load(f)
        
#         cs_paper_info = data.get("cs_paper_info", {})
#         papers_chunks = {}
#         for paper_id, paper in cs_paper_info.items():
#             content = paper.get("content", "")
#             if content:
#                 chunks = self.split_into_chunks(content, chunk_token_size)
#             else:
#                 chunks = []
            
#             papers_chunks[paper_id] = {
#                 "id": paper.get("id"),
#                 "title": paper.get("title"),
#                 "chunks": chunks
#             }
        
#         with open(output_file_path, "w", encoding="utf-8") as f:
#             json.dump(papers_chunks, f, ensure_ascii=False, indent=2)
        
#         print(f"Saved paper chunks for {len(papers_chunks)} papers to {output_file_path}")
#         return papers_chunks
        
#     def rank_select(self, population: list[dict]) -> list[dict]:
#         """
#         Rank-based selection, select individuals with probability proportional to their rank.
#         """
#         if self.problem_type == "black_box":
#             population = [individual for individual in population if individual["exec_success"] and individual["obj"] < self.seed_ind["obj"]]
#         else:
#             population = [individual for individual in population if individual["exec_success"]]
#         if len(population) < 2:
#             return None
#         # Sort population by objective value
#         population = sorted(population, key=lambda x: x["obj"])
#         ranks = [i for i in range(len(population))]
#         probs = [1 / (rank + 1 + len(population)) for rank in ranks]
#         # Normalize probabilities
#         probs = [prob / sum(probs) for prob in probs]
#         selected_population = []
#         trial = 0
#         while len(selected_population) < 2 * self.cfg.pop_size:
#             trial += 1
#             parents = np.random.choice(population, size=2, replace=False, p=probs)
#             if parents[0]["obj"] != parents[1]["obj"]:
#                 selected_population.extend(parents)
#             if trial > 1000:
#                 return None
#         return selected_population
    
    
#     def random_select(self, population: list[dict]) -> list[dict]:
#         """
#         Random selection, select individuals with equal probability.
#         """
#         selected_population = []
#         # Eliminate invalid individuals
#         if self.problem_type == "black_box":
#             population = [individual for individual in population if individual["exec_success"] and individual["obj"] < self.seed_ind["obj"]]
#         else:
#             population = [individual for individual in population if individual["exec_success"]]
#         if len(population) < 2:
#             return None
#         trial = 0
#         while len(selected_population) < 2 * self.cfg.pop_size:
#             trial += 1
#             parents = np.random.choice(population, size=2, replace=False)
#             # If two parents have the same objective value, consider them as identical; otherwise, add them to the selected population
#             if parents[0]["obj"] != parents[1]["obj"]:
#                 selected_population.extend(parents)
#             if trial > 1000:
#                 return None
#         return selected_population

#     def gen_short_term_reflection_prompt(self, ind1: dict, ind2: dict) -> tuple[list[dict], str, str]:
#         """
#         Short-term reflection before crossovering two individuals.
#         """
#         if ind1["obj"] == ind2["obj"]:
#             print(ind1["code"], ind2["code"])
#             raise ValueError("Two individuals to crossover have the same objective value!")
#         # Determine which individual is better or worse
#         if ind1["obj"] < ind2["obj"]:
#             better_ind, worse_ind = ind1, ind2
#         elif ind1["obj"] > ind2["obj"]:
#             better_ind, worse_ind = ind2, ind1

#         worse_code = filter_code(worse_ind["code"])
#         better_code = filter_code(better_ind["code"])
        
#         system = self.system_reflector_prompt
#         user = self.user_reflector_st_prompt.format(
#             func_name = self.func_name,
#             func_desc = self.func_desc,
#             problem_desc = self.problem_desc,
#             worse_code=worse_code,
#             better_code=better_code
#             )
#         message = [{"role": "system", "content": system}, {"role": "user", "content": user}]
        
#         # Print reflection prompt for the first iteration
#         if self.print_short_term_reflection_prompt:
#                 logging.info("Short-term Reflection Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
#                 self.print_short_term_reflection_prompt = False
#         return message, worse_code, better_code


#     def short_term_reflection(self, population: list[dict]) -> tuple[list[list[dict]], list[str], list[str]]:
#         """
#         Short-term reflection before crossovering two individuals.
#         """
#         messages_lst = []
#         worse_code_lst = []
#         better_code_lst = []
#         for i in range(0, len(population), 2):
#             # Select two individuals
#             parent_1 = population[i]
#             parent_2 = population[i+1]
            
#             # Short-term reflection
#             messages, worse_code, better_code = self.gen_short_term_reflection_prompt(parent_1, parent_2)
#             messages_lst.append(messages)
#             worse_code_lst.append(worse_code)
#             better_code_lst.append(better_code)
        
#         # Asynchronously generate responses
#         response_lst = self.short_reflector_llm.multi_chat_completion(messages_lst)
#         return response_lst, worse_code_lst, better_code_lst
    
#     def long_term_reflection(self, short_term_reflections: list[str]) -> None:
#         """
#         Long-term reflection before mutation.
#         """
#         system = self.system_reflector_prompt
#         user = self.user_reflector_lt_prompt.format(
#             problem_desc = self.problem_desc,
#             prior_reflection = self.long_term_reflection_str,
#             new_reflection = "\n".join(short_term_reflections),
#             )
#         messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
        
#         if self.print_long_term_reflection_prompt:
#             logging.info("Long-term Reflection Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
#             self.print_long_term_reflection_prompt = False
        
#         self.long_term_reflection_str = self.long_reflector_llm.multi_chat_completion([messages])[0]
        
#         # Write reflections to file
#         file_name = f"problem_iter{self.iteration}_short_term_reflections.txt"
#         with open(file_name, 'w') as file:
#             file.writelines("\n".join(short_term_reflections) + '\n')
        
#         file_name = f"problem_iter{self.iteration}_long_term_reflection.txt"
#         with open(file_name, 'w') as file:
#             file.writelines(self.long_term_reflection_str + '\n')


#     def crossover(self, short_term_reflection_tuple: tuple[list[list[dict]], list[str], list[str]]) -> list[dict]:
#         reflection_content_lst, worse_code_lst, better_code_lst = short_term_reflection_tuple
#         messages_lst = []
#         for reflection, worse_code, better_code in zip(reflection_content_lst, worse_code_lst, better_code_lst):
#             # Crossover
#             system = self.system_generator_prompt
#             func_signature0 = self.func_signature
#             func_signature1 = self.func_signature
#             user = self.crossover_prompt.format(
#                 user_generator = self.user_generator_prompt,
#                 func_signature0 = func_signature0,
#                 func_signature1 = func_signature1,
#                 worse_code = worse_code,
#                 better_code = better_code,
#                 reflection = reflection,
#                 func_name = self.func_name,
#             )
#             messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
#             messages_lst.append(messages)
            
#             # Print crossover prompt for the first iteration
#             if self.print_crossover_prompt:
#                 logging.info("Crossover Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
#                 self.print_crossover_prompt = False
        
#         # Asynchronously generate responses
#         response_lst = self.crossover_llm.multi_chat_completion(messages_lst)
#         crossed_population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(response_lst)]

#         assert len(crossed_population) == self.cfg.pop_size
#         return crossed_population


#     def mutate(self) -> list[dict]:
#         """Elitist-based mutation. We only mutate the best individual to generate n_pop new individuals."""
#         system = self.system_generator_prompt
#         func_signature1 = self.func_signature 
#         user = self.mutation_prompt.format(
#             user_generator = self.user_generator_prompt,
#             reflection = self.long_term_reflection_str + self.external_knowledge,
#             func_signature1 = func_signature1,
#             elitist_code = filter_code(self.elitist["code"]),
#             func_name = self.func_name,
#         )
#         messages = [{"role": "system", "content": system}, {"role": "user", "content": user}]
#         if self.print_mutate_prompt:
#             logging.info("Mutation Prompt: \nSystem Prompt: \n" + system + "\nUser Prompt: \n" + user)
#             self.print_mutate_prompt = False
#         responses = self.mutation_llm.multi_chat_completion([messages], int(self.cfg.pop_size * self.mutation_rate))
#         population = [self.response_to_individual(response, response_id) for response_id, response in enumerate(responses)]
#         return population


#     def rag_operator(self, short_term_reflection_tuple: tuple[list[list[dict]], list[str], list[str]], 
#                     long_term_reflection_str: str) -> list[dict]:

#         chunks_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks.json"
#         embeddings_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks_embeddings.npy"
#         index_file = f"{self.root_dir}/TSP_database/TSP_papers_chunks_index.npy"

#         if not os.path.exists(chunks_file):
#             raise FileNotFoundError(f"Knowledge chunks file not found: {chunks_file}")
        
#         with open(chunks_file, "r", encoding="utf-8") as f:
#             papers_dict = json.load(f)

#         all_chunks = []
#         for paper in papers_dict.values():
#             if "chunks" in paper:
#                 all_chunks.extend(paper["chunks"])

#         tokenizer = AutoTokenizer.from_pretrained(f"{self.root_dir}/all-MiniLM-L6-v2")
#         model = AutoModel.from_pretrained(f"{self.root_dir}/all-MiniLM-L6-v2")
#         model.eval()  

#         expected_D = 384  

#         # Load or compute the embedding of each chunk
#         if os.path.exists(embeddings_file):
#             chunks_embeddings = np.load(embeddings_file, allow_pickle=True).item()
#         else:
#             chunks_embeddings = {}
#             for chunk in all_chunks:
#                 # Use the model to calculate the embedding of each chunk
#                 inputs = tokenizer(chunk, return_tensors="pt", padding=True, truncation=True)
#                 with torch.no_grad():
#                     output = model(**inputs)
#                 chunk_embedding = output.last_hidden_state.mean(dim=1).cpu().numpy()  
#                 chunks_embeddings[chunk] = chunk_embedding.tolist()  
#             os.makedirs(os.path.dirname(embeddings_file), exist_ok=True)
#             np.save(embeddings_file, chunks_embeddings)

#         # Load or build an index matrix
#         index_matrix = None
#         if os.path.exists(index_file):
#             loaded_index = np.load(index_file, allow_pickle=True)
#             index_data = loaded_index.item()
#             all_chunks_ordered, index_matrix_loaded = index_data.get("all_chunks"), index_data.get("index_matrix")
#             # Check whether the number of chunks is the same as that of the embedding dimension
#             if len(all_chunks_ordered) == len(all_chunks) and index_matrix_loaded.shape[1] == expected_D:
#                 index_matrix = index_matrix_loaded

#         if index_matrix is None:
#             embeddings_list = []
#             for chunk in all_chunks:
#                 emb = np.array(chunks_embeddings[chunk]).reshape(1, -1)
#                 # If the dimension is incorrect, the embedding is recalculated
#                 if emb.shape[1] != expected_D:
#                     inputs = tokenizer(chunk, return_tensors="pt", padding=True, truncation=True)
#                     with torch.no_grad():
#                         output = model(**inputs)
#                     emb = output.last_hidden_state.mean(dim=1).cpu().numpy()  
#                 embeddings_list.append(emb)
#             # Use vstack to stack all embeddings into a 2D matrix with the shape (N, D)
#             index_matrix = np.vstack(embeddings_list)
#             index_data = {"all_chunks": all_chunks, "index_matrix": index_matrix}
#             os.makedirs(os.path.dirname(index_file), exist_ok=True)
#             np.save(index_file, index_data)

#         client = OpenAI(
#             api_key=self.cfg.llm_client.api_key,  
#             base_url="https://api.agicto.cn/v1"  
#         )

#         reflection_content_lst, worse_code_lst, better_code_lst = short_term_reflection_tuple
#         population = []

#         for reflection, worse_code, better_code in zip(reflection_content_lst, worse_code_lst, better_code_lst):
            
#             if isinstance(reflection, list):
#                 updated_reflection = reflection.copy()
#             else:
#                 updated_reflection = [{"original": reflection}]
            
#             # Combine all Reflection content to generate 4O-Mini prompts
#             combined_reflection = " ".join([str(item) for item in reflection]) if isinstance(reflection, list) else str(reflection)

#             # Generate prompts for improvement suggestions to ensure that they are not duplicated with the Reflection
#             prompt = (
#                 """I am designing a "multi_head_attention" function in the POMO model to solve the TSP problem. Please generate a possible improvement for me."""
#                 "Please be careful not to duplicate or be similar to the known reflections below."
#                 "The following are known reflections:\n"
#                 f"{long_term_reflection_str}\n"  # Long-term reflection
#                 f"{combined_reflection}"  # Current Reflection Content
#                 "Please provide suggestions for improvement other than the above. Please give your comments directly, no more than 20 words and without any extra words."
#             )
            
#             chat_completion = client.chat.completions.create(
#                 messages=[{"role": "user", "content": prompt}],
#                 model="gpt-3.5-turbo"
#             )
            
#             suggestion = chat_completion.choices[0].message.content
            

#             # Calculate the embedding of the generated recommendations
#             suggestion_inputs = tokenizer(suggestion, return_tensors="pt", padding=True, truncation=True)
#             with torch.no_grad():
#                 suggestion_output = model(**suggestion_inputs)
#             suggestion_embedding = suggestion_output.last_hidden_state.mean(dim=1).cpu().numpy()  # (1, D)
#             suggestion_embedding /= np.linalg.norm(suggestion_embedding)

#             # Calculate the cosine similarity to each chunk
#             similarities = cosine_similarity(index_matrix, suggestion_embedding).flatten()

#             # Select the two blocks with the highest similarity
#             top_indices = np.argsort(similarities)[-2:][::-1]
#             top_chunks = [(all_chunks[i], similarities[i]) for i in top_indices]

#             # # 明确指定函数签名
#             func_signature = self.func_signature

#             # # Generate a population using a crossover-like structure
#             # # Generate a new population based on the two most relevant chunks and their suggestions
#             # population_prompt = (
#             # """You are an expert in the domain of designing function. Your task is to design the pick_move function in the aco algorithm that can effectively solve the optimization TSP problem.
#             # Your answer only outputs code, and Always include `from torch.distributions import Categorical` when using the Categorical distribution for sampling. This is critical - failure to include this import will cause runtime errors. And format all code as a Python code string: "```python ... ```".\n"""
#             #         f"""Write a pick_move function for Solving Traveling Salesman Problem (TSP) with EXACTLY this function signature:
#             # {func_signature}

#             # IMPORTANT TENSOR DIMENSIONS:
#             # - prev: tensor with shape (n_ants,) - Previous nodes for all ants
#             # - mask: tensor with shape (n_ants, problem_size) - Masks (0) for visited cities 
#             # - pheromone: tensor with shape (problem_size, problem_size) - Pheromone matrix
#             # - heuristic: tensor with shape (problem_size, problem_size) - Heuristic information matrix
#             # - alpha: float - Importance of pheromone
#             # - beta: float - Importance of heuristic

#             # PAY CAREFUL ATTENTION TO TENSOR DIMENSIONS:
#             # 1. When extracting values from pheromone or heuristic using prev as indices, the result will have shape (n_ants, problem_size)
#             # 2. The return statement MUST be exactly: return actions, log_probs
#             # 3. This result MUST be multiplied with mask which has shape (n_ants, problem_size)
#             # 4. NEVER perform operations between tensors with incompatible shapes
#             # 5. ALWAYS check that tensor dimensions match before operations like multiplication

#             # The pick_move function should handle individualized decision strategies for each ant, implementing exploration and exploitation balance.

#             # MANDATORY ERROR HANDLING:
#             # - Include shape verification at the start of your function to catch potential dimension mismatches
#             # - Make sure all tensor operations respect the exact dimensions of the inputs
#             # - log_probs: If require_prob is True, a tensor with shape (n_ants,) of log probabilities, else None


#             # Here are some resources you can refer to:
#             # {top_chunks}

#             # The function MUST be implemented WITHOUT shape errors and MUST exactly follow the specified signature.
#             # """
#             #         "Please write an improved function `pick_move` with EXACTLY the function signature provided above. The function must accept these 7 parameters: prev, mask, require_prob, pheromone, heuristic, alpha, beta. Output code only and enclose your code with Python code block: ```python ... ```."
#             #     )

#             # population_response = client.chat.completions.create(
#             #     messages=[{"role": "user", "content": population_prompt}],
#             #     model="gpt-3.5-turbo"
#             # )

#             # population_content = population_response.choices[0].message.content

#             # if "def pick_move(prev, mask, require_prob, pheromone, heuristic, alpha, beta):" not in population_content or "return actions, log_probs" not in population_content:
#             #     # 如果不符合要求，尝试修复
#             #     fix_prompt = (
#             #         "The generated code does not meet the requirements. Please rewrite the function with EXACTLY this signature and return statement:\n"
#             #         f"Signature: {func_signature}\n"
#             #         f"Return statement:return actions, log_probs\n"
#             #         "The function MUST return EXACTLY two values: actions and log_probs. No more, no fewer.\n\n"
#             #         "Here is the code you generated:\n\n"
#             #         f"{population_content}\n\n"
#             #         "Please fix it to ensure it returns EXACTLY two values and output only the corrected code enclosed in ```python ... ```."
#             #     )
                
#             #     fix_response = client.chat.completions.create(
#             #         messages=[{"role": "user", "content": fix_prompt}],
#             #         model="gpt-3.5-turbo"
#             #     )

#             # if "RuntimeError: The size of tensor a" in population_content or "combined_info * mask" in population_content:
#             #     # 如果发现潜在的张量形状问题，进行特定修复
#             #     shape_fix_prompt = (
#             #         "The code you generated may have tensor shape compatibility issues. Please fix the following code to ensure all tensor operations respect dimensions:\n\n"
#             #         f"{population_content}\n\n"
#             #         "Specifically, ensure that when applying mask to any computed values, both tensors have compatible shapes. "
#             #         "prev has shape (n_ants,), mask has shape (n_ants, problem_size), pheromone and heuristic have shape (problem_size, problem_size).\n\n"
#             #         "When you index pheromone[prev], the result has shape (n_ants, problem_size). Make sure all subsequent operations maintain dimension compatibility.\n\n"
#             #         "Please output only the corrected code enclosed in ```python ... ```."
#             #     )
                
#             #     shape_fix_response = client.chat.completions.create(
#             #         messages=[{"role": "user", "content": shape_fix_prompt}],
#             #         model="gpt-3.5-turbo"
#             #     )
                
#             #     population_content = shape_fix_response.choices[0].message.content
#             # Prompt for optimizing initial_heuristics function in ReEvo framework
#             population_prompt = (
#                 f"""You are an expert in designing "multi_head_attention" function in the POMO model to solve the Traveling Salesman Problem (TSP).

#             Your task is to improve or refactor the multi_head_attention function that generates attention scores for the Learned Heuristic Decoder (LEHD) in POMO to guide next-city selection.

#             Write an attention function for solving TSP with EXACTLY this function signature:{func_signature}

#             IMPORTANT REQUIREMENTS:

#             1.Input: Query (q), key (k), value (v) tensors of shapes (batch, head_num, n, key_dim) and (batch, head_num, problem, key_dim), and optional masks.\n

#             2.Output: Attention output tensor of shape (batch, n, head_num * key_dim).\n

#             3.Handle potential numerical instabilities in attention score computation.\n

#             4.Preserve the core logic of multi-head attention with scaled dot-product attention.\n

#             5.Ensure the function works with different batch sizes and problem sizes (TSP20 to TSP1000).\n

#             SPECIFIC CONSTRAINTS:

#             1.Avoid numerical overflow in score scaling.\n

#             2.Use stable softmax transformations (e.g., nn.Softmax(dim=3)).\n

#             3.Implement efficient tensor operations for attention computation.\n

#             4.Return a tensor that supports downstream attention bias generation in POMO's decoder.\n

#             ADDITIONAL CONSIDERATIONS:\n
#             1.The function should handle both single-node and full-graph attention contexts.\n
#             2.Maintain computational efficiency using tensor operations.\n
#             3.Provide robust attention scores that capture node relationships effectively.\n
#             4.Support flexible mask handling for visited nodes or group-specific constraints.\n

#             Here are some resources you can refer to:\n {top_chunks}\n

#             The function MUST be implemented WITHOUT shape errors and MUST exactly follow the specified signature.

#             Please write an improved function multi_head_attention with EXACTLY the function signature provided above. The function must compute attention scores using efficient tensor operations and stable transformations, supporting POMO's path selection. Output code only and enclose your code with Python code block: ```python ... ```.""")
                
#             population_response = client.chat.completions.create(
#                 messages=[{"role": "user", "content": population_prompt}],
#                 model="gpt-3.5-turbo"
#             )

#             population_content = population_response.choices[0].message.content           


#             rag_population = [self.response_to_individual(population_content, 0)]
#         return rag_population


#     def evolve(self):
#         chunks = self.split_papers_contents()
#         while self.function_evals < self.cfg.max_fe:
#             # If all individuals are invalid, stop
#             if all([not individual["exec_success"] for individual in self.population]):
#                 raise RuntimeError(f"All individuals are invalid. Please check the stdout files in {os.getcwd()}.")
#             # Select
#             population_to_select = self.population if (self.elitist is None or self.elitist in self.population) else [self.elitist] + self.population # add elitist to population for selection
#             selected_population = self.random_select(population_to_select)
#             if selected_population is None:
#                 raise RuntimeError("Selection failed. Please check the population.")
#             # Short-term reflection
#             short_term_reflection_tuple = self.short_term_reflection(selected_population) # (response_lst, worse_code_lst, better_code_lst)
            
#             # Crossover
#             crossed_population = self.crossover(short_term_reflection_tuple)
#             # Evaluate
#             self.population = self.evaluate_population(crossed_population)
#             # Update
#             self.update_iter()
#             # Long-term reflection
#             self.long_term_reflection([response for response in short_term_reflection_tuple[0]])
#             # Mutate
#             mutated_population = self.mutate()
#             # Evaluate
#             self.population.extend(self.evaluate_population(mutated_population))
#             # Update
#             self.update_iter()
#             # Call the RAG operator to get the updated triplet
#             if random.random() < 1.0:     
#                 rag_population = self.rag_operator(short_term_reflection_tuple,self.long_term_reflection_str)
#                 self.population = self.evaluate_population(rag_population)
#                 self.update_iter() 
#         return self.best_code_overall, self.best_code_path_overall
