"""
Configuration for Lorenz system experiments.
"""
from dataclasses import dataclass
from typing import List
from .base_config import BaseConfig


@dataclass
class LorenzConfig(BaseConfig):
    """Configuration for stochastic Lorenz system experiments."""
    
    # Experiment settings
    experiment_name: str = "lorenz_rpit"
    
    # Lorenz system parameters
    sigma: float = 10.0      # Prandtl number
    rho: float = 28.0        # Rayleigh number
    beta: float = 8.0/3.0    # Geometric factor
    
    # Stochastic parameters
    noise_std: float = 0.5   # System noise standard deviation
    initial_noise_std: float = 0.1  # Initial condition noise
    
    # Time domain
    t_start: float = 0.0
    t_end: float = 10.0
    dt: float = 0.01
    
    # Training data
    num_trajectories: int = 10      # Number of different trajectories
    num_time_points: int = 1000     # Points per trajectory
    corruption_level: float = 0.1   # Data corruption percentage
    
    # Model architecture (for 3D Lorenz system)
    output_dim: int = 3
    hidden_layers: List[int] = None
    
    def __post_init__(self):
        if self.hidden_layers is None:
            self.hidden_layers = [128, 128, 128, 128]
        super().__post_init__()
    
    def get_lorenz_params(self) -> dict:
        """Get Lorenz system parameters."""
        return {
            'sigma': self.sigma,
            'rho': self.rho,
            'beta': self.beta,
        }
    
    def get_time_config(self) -> dict:
        """Get time domain configuration."""
        return {
            't_start': self.t_start,
            't_end': self.t_end,
            'dt': self.dt,
        }
