'''ResNet in PyTorch.
For Pre-activation ResNet, see 'preact_resnet.py'.
Reference:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385
'''
import torch
import torch.nn as nn
import torch.nn.functional as F


def snnl_func(x, y, t, metric='cosine'):
    x = F.relu(x)
    same_label_mask = (y == y.unsqueeze(1)).squeeze().to(torch.float32).cuda()
    if metric == 'cosine':
        norm = F.normalize(x.flatten(1), p=2, dim=1)
        dist = 1 - torch.matmul(norm, norm.transpose(0, 1))
    else:
        raise Exception(f'unimplemented metric: {metric}')
    exp = torch.clamp(torch.exp(-(dist/t)) - torch.eye(x.shape[0]).cuda(), min=0, max=1)
    prob = (exp / (1e-5+exp.sum(1).unsqueeze(1)))*same_label_mask
    loss = -torch.log(1e-5+prob.sum(1)).mean()
    return loss


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class EWEResNet(nn.Module):
    def __init__(self, block, num_blocks, n_outputs=10):
        super(EWEResNet, self).__init__()
        self.num_blocks = num_blocks
        self.in_planes = 64

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)

        # planes = 64
        # layer_stride = 1
        # strides = [layer_stride] + [1] * (num_blocks[0] - 1)
        # self.layers1 = []
        # for stride in strides:
        #     self.layers1.append(block(self.in_planes, planes, stride))
        #     self.in_planes = planes * block.expansion
        # self.layers1 = nn.Sequential(*self.layers1)
        #
        # planes = 128
        # layer_stride = 2
        # strides = [layer_stride] + [1] * (num_blocks[1] - 1)
        # self.layers2 = []
        # for stride in strides:
        #     self.layers2.append(block(self.in_planes, planes, stride))
        #     self.in_planes = planes * block.expansion
        #
        # planes = 256
        # layer_stride = 2
        # strides = [layer_stride] + [1] * (num_blocks[2] - 1)
        # self.layers3 = []
        # for stride in strides:
        #     self.layers3.append(block(self.in_planes, planes, stride))
        #     self.in_planes = planes * block.expansion
        #
        # planes = 512
        # layer_stride = 2
        # strides = [layer_stride] + [1] * (num_blocks[3] - 1)
        # self.layers4 = []
        # for stride in strides:
        #     self.layers4.append(block(self.in_planes, planes, stride))
        #     self.in_planes = planes * block.expansion

        self.fc = nn.Linear(512*block.expansion, n_outputs)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        res = []
        out = F.relu(self.bn1(self.conv1(x)))
        for i in range(self.num_blocks[0]):
            out = self.layer1[i](out)
        for i in range(self.num_blocks[1]):
            out = self.layer2[i](out)
        for i in range(self.num_blocks[2]):
            out = self.layer3[i](out)
        for i in range(self.num_blocks[3]):
            res.append(out)
            out = self.layer4[i](out)
        # out = self.layer1(out)
        # out = self.layer2(out)
        # out = self.layer3(out)
        # out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        res.append(out)
        out = out.view(out.size(0), -1)
        self.features = out
        out = self.fc(out)
        res.append(out)
        return res

    def snnl(self, predictions, w, temp):
        x1 = predictions[-2]
        x2 = predictions[-3]
        x3 = predictions[-4]
        inv_temp_1 = 100./temp[0]
        inv_temp_2 = 100./temp[1]
        inv_temp_3 = 100./temp[2]
        loss1 = snnl_func(x1, w, inv_temp_1)
        loss2 = snnl_func(x2, w, inv_temp_2)
        loss3 = snnl_func(x3, w, inv_temp_3)
        return [loss1, loss2, loss3]

    def total_loss(self, predictions, targets, w, factors, temp, snnl_factor=1):
        ce_loss = F.cross_entropy(predictions[-1], targets)
        snnl = self.snnl(predictions, w, temp)
        soft_nearest_neighbor = factors[0] * snnl[0] + factors[1] * snnl[1] + factors[2] * snnl[2]
        soft_nearest_neighbor = torch.gt(w.mean(), 0).to(torch.float32) * soft_nearest_neighbor * snnl_factor
        # print(soft_nearest_neighbor)
        return ce_loss - soft_nearest_neighbor


class PlainResNet(nn.Module):
    def __init__(self, block, num_blocks, n_outputs=10):
        super(PlainResNet, self).__init__()
        self.num_blocks = num_blocks
        self.in_planes = 64

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        # self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        # self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        # self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        # self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)

        planes = 64
        layer_stride = 1
        strides = [layer_stride] + [1] * (num_blocks[0] - 1)
        self.layers1 = []
        for stride in strides:
            self.layers1.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        planes = 128
        layer_stride = 2
        strides = [layer_stride] + [1] * (num_blocks[1] - 1)
        self.layers2 = []
        for stride in strides:
            self.layers2.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        planes = 256
        layer_stride = 2
        strides = [layer_stride] + [1] * (num_blocks[2] - 1)
        self.layers3 = []
        for stride in strides:
            self.layers3.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        planes = 512
        layer_stride = 2
        strides = [layer_stride] + [1] * (num_blocks[3] - 1)
        self.layers4 = []
        for stride in strides:
            self.layers4.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        self.fc = nn.Linear(512*block.expansion, n_outputs)

    # def _make_layer(self, block, planes, num_blocks, stride):
    #     strides = [stride] + [1]*(num_blocks-1)
    #     layers = []
    #     for stride in strides:
    #         layers.append(block(self.in_planes, planes, stride))
    #         self.in_planes = planes * block.expansion
    #     return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        for i in range(self.num_blocks[0]):
            out = self.layers1[i]
        for i in range(self.num_blocks[1]):
            out = self.layers2[i]
        for i in range(self.num_blocks[2]):
            out = self.layers3[i]
        for i in range(self.num_blocks[3]):
            out = self.layers4[i]
        # out = self.layer1(out)
        # out = self.layer2(out)
        # out = self.layer3(out)
        # out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        self.features = out
        out = self.fc(out)
        return out

# call EWE resnet

def EWEResNet18(n_outputs):
    return EWEResNet(BasicBlock, [2, 2, 2, 2], n_outputs=n_outputs)


def EWEResNet34(n_outputs):
    return EWEResNet(BasicBlock, [3, 4, 6, 3], n_outputs=n_outputs)


def EWEResNet50(n_outputs):
    return EWEResNet(Bottleneck, [3, 4, 6, 3], n_outputs=n_outputs)


def EWEResNet101(n_outputs):
    return EWEResNet(Bottleneck, [3, 4, 23, 3], n_outputs=n_outputs)


def EWEResNet152(n_outputs):
    return EWEResNet(Bottleneck, [3, 8, 36, 3], n_outputs=n_outputs)


# call plain resnet

def PlainResNet18(n_outputs):
    return PlainResNet(BasicBlock, [2, 2, 2, 2], n_outputs=n_outputs)


def PlainResNet34(n_outputs):
    return PlainResNet(BasicBlock, [3, 4, 6, 3], n_outputs=n_outputs)


def PlainResNet50(n_outputs):
    return PlainResNet(Bottleneck, [3, 4, 6, 3], n_outputs=n_outputs)


def PlainResNet101(n_outputs):
    return PlainResNet(Bottleneck, [3, 4, 23, 3], n_outputs=n_outputs)


def PlainResNet152(n_outputs):
    return PlainResNet(Bottleneck, [3, 8, 36, 3], n_outputs=n_outputs)