# AgentGym: Benchmarking the Capabilities and Adversarial Robustness of LLM Agents

## Running the benchmark

The benchmark can be run with the [benchmark](scripts/benchmark.py) script. Documentation on how to use the script can be obtained with the `--help` flag.

For example, to run the `workspace` suite on the tasks 0 and 1, with `gpt-4o-2024-05-13` as the LLM, the tool filter as a defense, and the attacker with tool knowlege, run the following command:

```bash
python scripts/benchmark.py -s workspace -ut user_task_0 -ut user_task_1 --model gpt-4o-2024-05-13 --defense tool_filter --attacker tool_knowledge
```

To run the above, but on all suites and tasks, run the following:

```bash
python scripts/benchmark.py --model gpt-4o-2024-05-13 --defense tool_filter --attacker tool_knowledge
```

## Tasks definition

The (benign) user tasks are defined by subclassing `BaseUserTask`, while the (malicious) injection tasks are defined by subclassing `BaseInjectionTask`. All tasks need to be registered with the task suite they belong to.

### Task suite

Each group of tasks is defined by a `TaskSuite` instantiation. The task suite is resposnible for defining:

- The name of the suite
- The tasks that belong to the suite
- The tools avaliable for the suite tasks
- The type for the environment used by the suite

An example is availabe in the [`workspace`](src/agentgym/task_suites/workspace/task_suite.py) task suite. Each suite is a generic parametrized with the environment used by the tasks. This is to ease type checking when writing the tasks.

The task suite has also the ability to run tasks with the `run_task` method. This method takes as argument the user task to run and the injection task to check for. If no injection task is passed, then the method will only run the user task. The method returns booleans indicating if the task and/or the injection were successful. The method also takes as an optional argument the environment to use for the tasks. If no environment is passed, then the method will use the default environment for the suite. Moreover, the method takes as argument the injections to inject within the (default environment).

The environment for the task is defined in `src/agentgym/data/suites/{suite_name}/environment.yaml` and the injection vectors for the environment are defined in `src/function_calling_pi/data/suites/{suite_name}/injection_vectors.yaml`.

### User tasks

Each suite has a set of (benign) user tasks. Each user task is an instantiation of the [`BaseUserTask`](src/agentgym/task_suites/base_tasks.py) class. Also each user task is a generic parametrized with the environment used by the tasks. This is to ease type checking when writing the tasks. Each user task **must** define:

- A `PROMPT`: which is the prompt that will be passed to the LLM
- A `COMMENT`: which is a comment useful for humans that need to understand the task
- A `GROUND_TRUTH_OUTPUT`: which is a portion of the expected output of the task.
- A `ground_truth` method: a list of function calls with corresponding arguments which, when executed, should solve the task. The arguments can be the `Ellipsis` object (i.e., `...`) if the argument is not known and not relevant for solving the task.
- A `utility` method, used to calculate the utility of the task. It takes as argument the original environment and the environment after the LLM executed the task. Returns a boolean indicating if the task was executed successfully.

Additionally, each user task _may_ define an `init_environment` static method, which is used to initialize the environment for the task. This is useful for tasks that need to set up the environment before the task is executed. If the original environment is enough for the task, then this method is not needed.

Each user task should be decorated with the `@task_suite.register_user_task`, where `task_suite` is the task suite that the task belongs to.

An example implementation of a user task is available in the [`workspace`](src/agentgym/task_suites/workspace/user_tasks.py) module.

### Injection tasks

Each suite has a set of (malicious) injection tasks. Each injection task is an instantiation of the [`BaseInjectionTask`](src/agentgym/task_suites/base_tasks.py) class. Also each injection task is a generic parametrized with the environment used by the tasks. This is to ease type checking when writing the tasks. Each injection task **must** define:

- A `GOAL`: which is the goal that needs to be achieved via the injection
- A `COMMENT`: which is a comment useful for humans that need to understand the injection goal
- A `ground_truth` method: a list of function calls with corresponding arguments which, when executed, should execute the injection. The arguments can be the `Ellipsis` object (i.e., `...`) if the argument is not known and not relevant for solving the task.
- A `security` method, used to calculate if the injection was successful. It takes as argument the original environment and the environment after the LLM executed the task. It returns a boolean indicating if the injection was successful.

### Checking the correctness of the task suite

The correctness of the task suite can be checked with the [check_suite.py](scripts/check_suites.py) script. By default it checks all the suites, but, optionally takes as arguments the name of the selected suites to check, each after the argument `-s`, and checks if:

- The `ground_truth` method can be correctly executed.
- The `ground_truth` and `utility`/`security` methods are consistent with each other.
- If all the user tasks are injectable, i.e., if at least an injection placeholder is part of what is return by a tool and seen by the agent.

The script can be run with the following command (e.g., for the `workspace` suite):

```bash
python scripts/check_suites.py -s workspace
```

Or for all suites:

```bash
python scripts/check_suites.py
```

## Adding Tools

Tools are functions that can be used by the tasks to perform some operations. Tools can be stateless or stateful.

Each tool must have type hints for both the input arguments and the return value. The type hints are used to generate the documentation for the tools. Moreover, tools **must** contain documentation for all the input argument (with the exception of the state argument for statefult tools, which **must not** be documented).

All tools can be tested as regular functions.

### Stateless tools

Tools that do not need to access any state, can be registered with the `@register_function` decorator. Each tool can be defined as a function that takes any number of arguments and returns the result of the tool.

An example implementation of stateless tool is

```python
from agentgym import register_function


@register_function
def sum(a: int, b: int) -> int:
    """Sums to numbers.

    :param a: First number
    :param b: Second number
    """
    return a + b
```

### Stateful tools

Tools that need to access some state, must also be registered with the `@register_function` decorator. Each tool can be defined as a function that takes an arbitrary number of arguments.

Arguments that need to be injected from the environment are called _dependencies_ and work similarly to [FastAPI dependencies](https://fastapi.tiangolo.com/tutorial/dependencies/). Assuming a dependency as type `ExampleType`, and the dependency is inside of benchmark environments with the name `"example_name"`, then the argument needs to be type hinted using the type `Annotated[ExampleType, Depends("example_name")]`.

> [!NOTE]
> The tool is responsible for updating the state (and need to make sure that the update happens in-place!)

An example implementation of stateful tool is

```python
from typing import Annotated
from agentgym import Depends, register_stateful_function

from pydantic import BaseModel


class Environment(BaseModel):
    counter: Counter


class Counter():
    def __init__(self):
        self.counter = 0


@register_function
def add_to_counter(
        counter: Annotated[Counter, Depends("counter")], n: int = 1,
) -> int:
    """Adds an arbitrary number `n` to the counter and returns the new value.

    :param n: The number to add to the counter. Default is 1.
    """
    counter.counter += 1
    return counter.counter

```

> [!NOTE]
> Note that the `counter` argument is a dependency and is updated in-place, and is **not** documented in the docstring. This is because the agent does not need to know about this argument as it is injected by the engine executing the function.

## Development set-up

We use `rye`. You can follow the instructions on the [website](https://rye-up.com) to set it up.

### Install the dev packages, lint, test, typecheck

- Install all the packages with `rye sync --features transformers` (include `--features transformers` if you want to work on the PI detector)
- Lint with `rye lint --fix`
- Test with `rye test`
- Run the static type checker with `rye run pyright`

### What if I don't want to install `rye`?

Then you can set up the dev environment via `uv` or `pip` by running `[uv] pip install -r requirements-dev.lock`.

Then, you can lint with `ruff check . --fix`, test with `pytest`, and typecheck with `pyright`.
