import datetime
from typing import Annotated

from pydantic import BaseModel, EmailStr, model_validator
from typing_extensions import Self

from agentgym.functions_engine import Depends, register_function
from agentgym.tools.types import (
    CloudDriveFile,
    CloudDriveFileID,
    SharingPermission,
)


class CloudDrive(BaseModel):
    initial_files: list[CloudDriveFile]
    account_email: EmailStr
    files: dict[CloudDriveFileID, CloudDriveFile] = {}

    @model_validator(mode="after")
    def _create_files(self) -> Self:
        all_ids = [file.id_ for file in self.initial_files]
        if len(all_ids) != len(set(all_ids)):
            duplicate_ids = [id_ for id_ in all_ids if all_ids.count(id_) > 1]
            raise ValueError(f"File IDs must be unique. Duplicates: {duplicate_ids}")
        self.files = {file.id_: file for file in self.initial_files}
        return self

    def _get_next_id(self) -> CloudDriveFileID:
        largest_id = max((int(key) for key in self.files.keys()), default=0)
        return CloudDriveFileID(largest_id + 1)

    def get_file_by_id(self, file_id: CloudDriveFileID) -> CloudDriveFile:
        if file_id not in self.files:
            raise ValueError(f"File with ID '{file_id}' not found.")
        return self.files[file_id]

    def create_file(self, filename: str, content: str) -> CloudDriveFile:
        new_file = CloudDriveFile(
            id_=self._get_next_id(),
            filename=filename,
            content=content,
            size=len(content),
            owner=self.account_email,
            last_modified=datetime.datetime.now(),
        )
        self.files[new_file.id_] = new_file
        return new_file

    def delete_file(self, file_id: CloudDriveFileID) -> CloudDriveFile:
        if file_id not in self.files:
            raise ValueError(f"File with ID '{file_id}' not found.")
        file = self.files.pop(file_id)
        return file

    def search_files_by_filename(self, filename: str) -> list[CloudDriveFile]:
        return [file for file in self.files.values() if filename.lower() in file.filename.lower()]

    def search_files(self, query: str) -> list[CloudDriveFile]:
        return [file for file in self.files.values() if query.lower() in file.content.lower()]

    def append_to_file(self, file_id: CloudDriveFileID, content: str) -> CloudDriveFile:
        file = self.get_file_by_id(file_id)
        file.content += content
        file.size = len(file.content)
        file.last_modified = datetime.datetime.now()
        return file

    def get_file_sizes(self) -> dict[str, int]:
        return {file.id_: file.size for file in self.files.values()}


@register_function
def search_files_by_filename(
    cloud_drive: Annotated[CloudDrive, Depends("cloud_drive")], filename: str
) -> list[CloudDriveFile]:
    """Get a file from a cloud drive by its filename. It returns a list of files.
    Each file contains the file id, the content, the file type, and the filename.

    :param filename: The name of the file to retrieve.
    """
    return cloud_drive.search_files_by_filename(filename)


@register_function
def create_file(
    cloud_drive: Annotated[CloudDrive, Depends("cloud_drive")],
    filename: str,
    content: str,
) -> CloudDriveFile:
    """Create a new file in the cloud drive.

    :param filename: The name of the file to create.
    :param content: The content of the file to create.
    """
    return cloud_drive.create_file(filename, content)


@register_function
def delete_file(cloud_drive: Annotated[CloudDrive, Depends("cloud_drive")], file_id: str) -> CloudDriveFile:
    """Delete a file from the cloud drive by its filename.
    It returns the file that was deleted.

    :param file_id: The name of the file to delete.
    """
    return cloud_drive.delete_file(CloudDriveFileID(file_id))


@register_function
def get_file_by_id(cloud_drive: Annotated[CloudDrive, Depends("cloud_drive")], file_id: str) -> CloudDriveFile:
    """Get a file from a cloud drive by its ID.

    :param file_id: The ID of the file to retrieve.
    """
    return cloud_drive.get_file_by_id(file_id)


@register_function
def list_files(
    cloud_drive: Annotated[CloudDrive, Depends("cloud_drive")],
) -> list[CloudDriveFile]:
    """Retrieve all files in the cloud drive."""
    return list(cloud_drive.files.values())


@register_function
def share_file(
    cloud_drive: Annotated[CloudDrive, Depends("cloud_drive")],
    file_id: str,
    email: str,
    permission: SharingPermission,
) -> CloudDriveFile:
    """Share a file with a user.

    :param file_id: The ID of the file to share.
    :param email: The email of the user to share the file with.
    :param permission: The permission level to grant the user.
    """
    file = cloud_drive.get_file_by_id(CloudDriveFileID(file_id))
    file.shared_with[email] = permission
    return file


@register_function
def append_to_file(
    cloud_drive: Annotated[CloudDrive, Depends("cloud_drive")],
    file_id: str,
    content: str,
) -> CloudDriveFile:
    """Append content to a file in the cloud drive.

    :param file_id: The ID of the file to append content to.
    :param content: The content to append to the file.
    """
    return cloud_drive.append_to_file(CloudDriveFileID(file_id), content)


@register_function
def search_files(cloud_drive: Annotated[CloudDrive, Depends("cloud_drive")], query: str) -> list[CloudDriveFile]:
    """Search for files in the cloud drive by content.

    :param query: The string to search for in the files.
    """
    matches = cloud_drive.search_files(query)
    if len(matches) == 0:
        raise ValueError(f"No files found with the given query ('{query}'). Try a different search term.")
    return matches
