import sys
import json
from pathlib import Path

import numpy as np

import matplotlib.pyplot as plt


def main():
    for file_name in sys.argv[1:]:
        path = Path(file_name)
        with open(path, 'r') as f:
            data = json.load(f)
        n_trees = data['n_trees']
        metric = data['metric']
        scratch_classifier = np.array(data['scratch_classifier'])
        scratch_regressor = np.array(data['scratch_regressor'])
        kd_mse = np.array(data['kd_mse'])
        kd_mixed = np.array(data['kd_mixed']) if data['kd_mixed'] is not None else None
        kd_relerr = np.array(data['kd_relerr'])
        kd_abserr = np.array(data['kd_abserr'])
        kd_power = np.array(data['kd_power'])
        teacher_result = np.array(data['teacher_result'])

        plt.figure()
        plt.errorbar(n_trees, np.mean(scratch_classifier, 0), np.std(scratch_classifier, 0),
                    label='From scratch, classifier')
        plt.errorbar(n_trees, np.mean(scratch_regressor, 0), np.std(scratch_regressor, 0),
                    label='From scratch, regressor')
        plt.errorbar(n_trees, np.mean(kd_mse, 0), np.std(kd_mse, 0), label='KD MSE')
        if kd_mixed is not None:
            plt.errorbar(n_trees, np.mean(kd_mixed, 0), np.std(kd_mixed, 0), label='KD, tuned alpha')
        plt.errorbar(n_trees, np.mean(kd_relerr, 0), np.std(kd_relerr, 0), label='KD, minimax, relative error')
        plt.errorbar(n_trees, np.mean(kd_abserr, 0), np.std(kd_abserr, 0), label='KD, minimax, absolute error')
        plt.errorbar(n_trees, np.mean(kd_power, 0), np.std(kd_power, 0), label='KD, minimax, power constraint')
        plt.errorbar(n_trees, [np.mean(teacher_result)] * len(n_trees),
                    [np.std(teacher_result)] * len(n_trees), linestyle='--', label='Teacher')
        plt.xticks(n_trees)
        plt.ylabel(f'{metric}', fontsize=14)
        plt.xlabel('number of trees', fontsize=14)
        plt.legend()
        plot_path = path.with_suffix('.pdf')
        plt.savefig(str(plot_path), bbox_inches='tight')
        plt.close()

if __name__ == '__main__':
    main()
