Abstract Reasoning
==================

Code for running the experiments in the paper
Logical Activation Functions: Logit-space equivalents of Probabilistic Boolean Operators
https://openreview.net/forum?id=m6HNNpQO8dc
on the I-RAVEN dataset using an SRAN-based architecture.

This repository was forked from [husheng12345/SRAN](https://github.com/husheng12345/SRAN) under the GPLv3 license.
The source repository which contains the official implementation of the model from the AAAI 2021 paper:
[Stratified Rule-Aware Network for Abstract Visual Reasoning](https://arxiv.org/abs/2002.06838)  
Sheng Hu\*, Yuqing Ma\*, Xianglong Liu†, Yanlu Wei, Shihao Bai  
*Proceedings of the AAAI Conference on Artificial Intelligence (AAAI)*, 2021  
(\* equal contribution, † corresponding author)


I-RAVEN dataset generation
--------------------------

Using an Python 2.7 environment, generate the I-RAVEN dataset.

```bash
# Pick a name for the new environment
ENVNAME=raven2.7

# Create the environment
conda create -p "$ENVNAME" -q python=2.7 pip

# Activate the environment
conda activate "$ENVNAME"

# Install dependencies
pip install -r I-RAVEN/requirements.txt

# Generate the dataset into a data subdirectory within the repository
mkdir -p data
python I-RAVEN/main.py --save-dir data/I-RAVEN

# Deactivate env (we are done with the Python 2.7 environment now)
conda deactivate
```


Installation
------------

Set up a new Python 3 environment and install the SRAN dependinces.

```bash
# Pick a name for the new environment
ENVNAME=SRAN

# Create the python3.x conda environment, with pip installed
conda create -p "$ENVNAME" -q python=3.7 pip

# Activate the environment
conda activate "$ENVNAME"

# Install dependencies
pip install -r SRAN/requirements.txt
```


Train experiments
-----------------

To train the models, run the following bash code on a machine with 4 GPUs and 32 CPU cores.

Each model takes around 90 hours to train on a machine with four RTX-6000 GPUs.

```bash
DATASET="I-RAVEN"
DATAPATH="data/I-RAVEN"

for ACTFUN in relu max ail_xnor ail_or ail_and_or_dup ail_or_xnor_part ail_or_xnor_dup ail_and_or_xnor_part; do
    python SRAN/main.py \
        --seed 0 \
        --load_workers 32 \
        --dataset "$DATASET" \
        --dataset_path "$DATAPATH" \
        --save "checkpoints/sran-$ACTFUN" \
        --resume "checkpoints/sran-$ACTFUN/checkpoint_latest.pth" \
        --actfun "$ACTFUN"
done
```


Evaluate performance
--------------------

To run all evaluation permutations and generate the evaluation YAML files, run the following code.
Note that you need a machine with at least 1 GPU. More GPUs are not necessary for evaluation, but will be utilised if they are available.

```bash
DATASET="I-RAVEN"
DATAPATH="data/I-RAVEN"

for ACTFUN in relu max ail_xnor ail_or ail_and_or_dup ail_or_xnor_part ail_or_xnor_dup ail_and_or_xnor_part; do
    # Measure the performance for each of the 7 figure configurations
    for FIGURE_CONFIG in {0..6};
    do
        python SRAN/test.py \
            --dataset "$DATASET" \
            --dataset_path "$DATAPATH" \
            --test_figure_configurations="$FIGURE_CONFIG" \
            --resume "checkpoints/sran-$ACTFUN/checkpoint_latest.pth" \
            --batch_size 16 \
            --save "test_results/sran-$ACTFUN" \
            --print_freq 100
    done

    # Measure the overall performance
    python SRAN/test.py \
        --dataset "$DATASET" \
        --dataset_path "$DATAPATH" \
        --resume "checkpoints/sran-$ACTFUN/checkpoint_latest.pth" \
        --batch_size 16 \
        --save "test_results/sran-$ACTFUN" \
        --print_freq 100
done
```


Generate tables
---------------

To generate the tables shown in the paper, run the notebook `results.ipynb`.

```bash
python -m pip install jupyterlab pandas pyyaml
python -m jupyterlab "results.ipynb"
```

