Transfer learning experiments
=============================

Code for running the experiments in the paper
Logical Activation Functions: Logit-space equivalents of Probabilistic Boolean Operators
https://openreview.net/forum?id=m6HNNpQO8dc
for transfer learning from a network pre-trained on ImageNet.

The `train.py` and `validate.py` scripts were based on [rwightman/pytorch-image-models (v0.4.12)](https://github.com/rwightman/pytorch-image-models/tree/v0.4.12), which is distributed under the Apache License 2.0.


Installation
------------

Set up environment and install dependencies.

```
## Pick a name for the new environment
ENVNAME="exp-timm-ail-transfer"

# Create the python3.x conda environment, with pip installed
conda create -p "$ENVNAME" -q python=3.9.7 pip

# Activate the environment
conda activate "$ENVNAME"

## install reqs
pip install -r requirements.txt
```


Download datasets
-----------------

References to the papers which introduced the datasets used can be found in the Appendix of the paper.

We downloaded the CIFAR-10, CIFAR-100, and STL-10 datasets using `torchvision.datasets`.
This can be done with the following python code.

```python
import os
import torchvision.datasets

root_dir = "~/Datasets"

torchvision.datasets.CIFAR10(os.path.join(root_dir, "cifar10"), train=True, download=True)
torchvision.datasets.CIFAR10(os.path.join(root_dir, "cifar10"), train=False, download=True)
torchvision.datasets.CIFAR100(os.path.join(root_dir, "cifar100"), train=True, download=True)
torchvision.datasets.CIFAR100(os.path.join(root_dir, "cifar100"), train=False, download=True)
torchvision.datasets.STL10(os.path.join(root_dir, "cifar100"), split="train", download=True)
torchvision.datasets.STL10(os.path.join(root_dir, "cifar100"), split="test", download=True)
```

We downloaded the Caltech101, Oxford Flowers, and Standford Cars datasets from Kaggle, using the following sources:
- https://www.kaggle.com/huangruichu/caltech101/version/2?select=Caltech101
- https://www.kaggle.com/c/oxford-102-flower-pytorch/data
- https://www.kaggle.com/jutrera/stanford-car-dataset-by-classes-folder/version/2

Our code expects each dataset to be found in the directory `~/Datasets/<dataset-name>`,, but this can be customised as necessary by modifying the path declarations in the module `datasets.py`.


Run experiments
---------------

The experimental results shown in the paper can be replicated with the following bash code.

The code is set up to run on a machine with 4 GPUs, and a total batch size of 128.
Each experiment trains for a total of 25 epochs, which takes around 20 minutes on a machine with four NVIDIA Tesla T4 GPUs.

```
NLAYER=0
WIDTH=1024  # Dummy value; width is irrelevant with 0 layers but must be set for sensible output filenames
for SEED in {0..4}; do
    launch.sh caltech101 "$(( 170 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}"
    launch.sh cifar10 "$(( 270 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}"
    launch.sh cifar100 "$(( 370 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}"
    launch.sh oxfordflowers102 "$(( 470 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}"
    launch.sh stanfordcars "$(( 570 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}"
    launch.sh svhn "$(( 670 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}"
    launch.sh stl10 "$(( 770 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}"
done

NLAYER=2
WIDTH=512
for ACTFUN in "relu" "max" "ail_or" "ail_xnor" "ail_xnor" "ail_or_xnor_part" "ail_and_or_xnor_part" "max_min_dup" "ail_and_or_dup" "ail_or_xnor_dup" "ail_and_or_xnor_dup";
do
    for SEED in {0..4}; do
        launch.sh caltech101 "$(( 170 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh cifar10 "$(( 270 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh cifar100 "$(( 370 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh oxfordflowers102 "$(( 470 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh stanfordcars "$(( 570 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh svhn "$(( 670 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh stl10 "$(( 770 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
    done
done

NLAYER=2
WIDTH=720
for ACTFUN in "max" "ail_or" "ail_xnor" "ail_xnor" "ail_or_xnor_part" "ail_and_or_xnor_part"
do
    for SEED in {0..4}; do
        launch.sh caltech101 "$(( 170 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh cifar10 "$(( 270 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh cifar100 "$(( 370 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh oxfordflowers102 "$(( 470 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh stanfordcars "$(( 570 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh svhn "$(( 670 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh stl10 "$(( 770 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
    done
done

ACTFUN=ail_and_or_xnor_dup
for WIDTH in 438 352
do
    for SEED in {0..4}; do
        launch.sh caltech101 "$(( 170 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh cifar10 "$(( 270 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh cifar100 "$(( 370 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh oxfordflowers102 "$(( 470 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh stanfordcars "$(( 570 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh svhn "$(( 670 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh stl10 "$(( 770 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
    done
done


NLAYER=2
WIDTH=410
for ACTFUN in "relu" "max_min_dup" "ail_and_or_dup" "ail_or_xnor_dup" "ail_and_or_xnor_dup"
do
    for SEED in {0..4}; do
        launch.sh caltech101 "$(( 170 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh cifar10 "$(( 270 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh cifar100 "$(( 370 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh oxfordflowers102 "$(( 470 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh stanfordcars "$(( 570 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh svhn "$(( 670 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
        launch.sh stl10 "$(( 770 + SEED))" --mlp-layers="$NLAYER" --mlp-width="${WIDTH}" --mlp-actfun="$ACTFUN"
    done
done
```

Generate results table
----------------------

After running the experiments, the tables shown in the paper can be generated by running the notebook `results.ipynb`.

```
python -m pip install jupyterlab pandas
python -m jupyterlab results.ipynb
```
