# copied from lightly
from typing import Any, Callable, Dict, Iterable, List, Optional, Tuple, Type

import torch
from torch.nn import Module
from torch.nn.modules import (CrossMapLRN2d, GroupNorm, LayerNorm,
                              LocalResponseNorm)
from torch.nn.modules.batchnorm import _NormBase
from torch.nn.parameter import Parameter
from torch.optim.optimizer import Optimizer  # type: ignore[attr-defined]
from torch.optim.optimizer import required

_NORM_LAYERS = (_NormBase, LayerNorm, CrossMapLRN2d, LocalResponseNorm, GroupNorm)


def get_weight_decay_parameters(
    modules: Iterable[Module],
    decay_norm: bool = False,
    decay_bias: bool = False,
    norm_layers: Tuple[Type[Module], ...] = _NORM_LAYERS,
) -> Tuple[List[Parameter], List[Parameter]]:
    """Returns all parameters of the modules that should be decayed and not decayed.

    Args:
        modules:
            List of modules to get the parameters from.
        decay_norm:
            If True, normalization parameters are decayed.
        decay_bias:
            If True, bias parameters are decayed.
        norm_layers:
            Tuple of normalization classes to decay if decay_norm is True.

    Returns:
        (params, params_no_weight_decay) tuple.
    """
    params = []
    params_no_weight_decay = []
    for module in modules:
        for mod in module.modules():
            if isinstance(mod, norm_layers):
                if not decay_norm:
                    params_no_weight_decay.extend(mod.parameters(recurse=False))
                else:
                    params.extend(mod.parameters(recurse=False))
            else:
                for name, param in mod.named_parameters(recurse=False):
                    if not decay_bias and name.endswith("bias"):
                        params_no_weight_decay.append(param)
                    else:
                        params.append(param)
    return params, params_no_weight_decay

# copied from lightly lars.py but fixed
class LARS(Optimizer):
    """Extends SGD in PyTorch with LARS scaling from the paper "Large batch training of
    Convolutional Networks" [0].

    Implementation from PyTorch Lightning Bolts [1].

    - [0]: https://arxiv.org/pdf/1708.03888.pdf
    - [1]: https://github.com/Lightning-Universe/lightning-bolts/blob/2dfe45a4cf050f120d10981c45cfa2c785a1d5e6/pl_bolts/optimizers/lars.py#L1

    Args:
        params:
            Iterable of parameters to optimize or dicts defining parameter groups.
        lr:
            Learning rate
        momentum:
            Momentum factor.
        weight_decay:
            Weight decay (L2 penalty).
        dampening:
            Dampening for momentum.
        nesterov:
            Enables Nesterov momentum.
        trust_coefficient:
            Trust coefficient for computing learning rate.
        eps:
            Eps for division denominator.

    Example:
        >>> model = torch.nn.Linear(10, 1)
        >>> input = torch.Tensor(10)
        >>> target = torch.Tensor([1.])
        >>> loss_fn = lambda input, target: (input - target) ** 2
        >>> #
        >>> optimizer = LARS(model.parameters(), lr=0.1, momentum=0.9)
        >>> optimizer.zero_grad()
        >>> loss_fn(model(input), target).backward()
        >>> optimizer.step()

    .. note::
        The application of momentum in the SGD part is modified according to
        the PyTorch standards. LARS scaling fits into the equation in the
        following fashion.

        .. math::
            \begin{aligned}
                g_{t+1} & = \text{lars_lr} * (\beta * p_{t} + g_{t+1}), \\
                v_{t+1} & = \\mu * v_{t} + g_{t+1}, \\
                p_{t+1} & = p_{t} - \text{lr} * v_{t+1},
            \\end{aligned}

        where :math:`p`, :math:`g`, :math:`v`, :math:`\\mu` and :math:`\beta` denote the
        parameters, gradient, velocity, momentum, and weight decay respectively.
        The :math:`lars_lr` is defined by Eq. 6 in the paper.
        The Nesterov version is analogously modified.

    .. warning::
        Parameters with weight decay set to 0 will automatically be excluded from
        layer-wise LR scaling. This is to ensure consistency with papers like SimCLR
        and BYOL.
    """

    def __init__(
        self,
        params: Any,
        lr: float = required,
        momentum: float = 0,
        dampening: float = 0,
        weight_decay: float = 0,
        nesterov: bool = False,
        trust_coefficient: float = 0.001,
        eps: float = 1e-8,
        classic_momentum: bool = True,
    ):
        if lr is not required and lr < 0.0:
            raise ValueError(f"Invalid learning rate: {lr}")
        if momentum < 0.0:
            raise ValueError(f"Invalid momentum value: {momentum}")
        if weight_decay < 0.0:
            raise ValueError(f"Invalid weight_decay value: {weight_decay}")

        self.classic_momentum = classic_momentum
        defaults = dict(
            lr=lr,
            momentum=momentum,
            dampening=dampening,
            weight_decay=weight_decay,
            nesterov=nesterov,
            trust_coefficient=trust_coefficient,
            eps=eps,
        )
        if nesterov and (momentum <= 0 or dampening != 0):
            raise ValueError("Nesterov momentum requires a momentum and zero dampening")

        super().__init__(params, defaults)

    def __setstate__(self, state: Dict[str, Any]) -> None:
        super().__setstate__(state)

        for group in self.param_groups:
            group.setdefault("nesterov", False)

    @torch.no_grad()
    def step(self, closure: Optional[Callable[[], float]] = None) -> Optional[float]:
        """Performs a single optimization step.

        Args:
            closure (callable, optional): A closure that reevaluates the model
                and returns the loss.
        """
        loss = None
        if closure is not None:
            with torch.enable_grad():
                loss = closure()

        # exclude scaling for params with 0 weight decay
        for group in self.param_groups:
            weight_decay = group["weight_decay"]
            momentum = group["momentum"]
            dampening = group["dampening"]
            nesterov = group["nesterov"]
            eta = group["trust_coefficient"]
            eps = group["eps"]

            for p in group["params"]:
                if p.grad is None:
                    continue
                grad = p.grad + weight_decay * p
                if self.classic_momentum:
                    lars_lr = 1.0
                    if weight_decay > 0:  # apply layer adaptation only on params with weight_decay - not general but okay for now
                        p_norm = torch.linalg.norm(p)
                        g_norm = torch.linalg.norm(grad)
                        lars_lr = torch.where(
                            torch.greater(p_norm, 0),  # if p norm > 0, then yield
                            torch.where(torch.greater(g_norm, 0),  # if grad norm > 0, then yield
                                        eta * p_norm / (g_norm + eps),
                                        1),  # otherwise 1
                            1) # otherwise 1
                    scaled_lr = group['lr'] * lars_lr  # now scale by global lr
                    upd = scaled_lr * grad
                    if momentum != 0:
                        param_state = self.state[p]
                        if "momentum_buffer" not in param_state:
                            param_state["momentum_buffer"] = torch.zeros_like(p)
                        v = param_state["momentum_buffer"]
                        v.mul_(momentum).add_(upd, alpha=1 - dampening)
                        if nesterov:
                            upd = upd.add(v, alpha=momentum)
                        else:
                            upd = v
                    p.add_(upd, alpha=-1)
                else:
                    # popular momentum:
                    # update momentum buffer before scaling grad with local lr
                    # momentum also gets local lr
                    param_state = self.state[p]
                    if "momentum_buffer" not in param_state:
                        param_state["momentum_buffer"] = torch.zeros_like(p)
                    v = param_state["momentum_buffer"]
                    v.mul_(momentum).add_(grad, alpha=1 - dampening)
                    if nesterov:
                        upd = upd.add(v, alpha=momentum)
                    else:
                        upd = v
                    lars_lr = 1.0
                    if weight_decay > 0:
                        p_norm = torch.linalg.norm(p)
                        v_norm = torch.linalg.norm(upd)
                        lars_lr = torch.where(
                            torch.greater(p_norm, 0),  # if p norm > 0, then yield
                            torch.where(torch.greater(v_norm, 0),  # if grad norm > 0, then yield
                                        eta * p_norm / (v_norm + eps),
                                        1),  # otherwise 1
                            1) # otherwise 1
                    scaled_lr = group['lr'] * lars_lr  # now scale by global lr
                    p.add_(upd, alpha=-scaled_lr)
        return loss