'''Train CIFAR10 with PyTorch.'''
import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import torch.backends.cudnn as cudnn

import torchvision
import torchvision.transforms as transforms

import os
import argparse

from models import *
try:
    from utils import progress_bar
    PROGRESS_BAR_AVAILABLE = True
except (ValueError, OSError):
    # Fallback progress bar if utils.py fails
    PROGRESS_BAR_AVAILABLE = False
    print("Warning: Progress bar not available, using simple progress display")
    
    def progress_bar(current, total, msg=None):
        if current % 10 == 0 or current == total - 1:
            print(f"Progress: {current+1}/{total} {msg or ''}")


parser = argparse.ArgumentParser(description='PyTorch CIFAR10 Training')
parser.add_argument('--lr', default=0.1, type=float, help='learning rate')
parser.add_argument('--resume', '-r', action='store_true',
                    help='resume from checkpoint')
parser.add_argument('--patience', type=int, default=20, help='number of epochs to wait for validation accuracy improvement before early stopping')
args = parser.parse_args()

device = 'cuda' if torch.cuda.is_available() else 'cpu'
best_acc = 0  # best test accuracy
start_epoch = 0  # start from epoch 0 or last checkpoint epoch

print(f'Using device: {device}')
print(f'Early stopping patience: {args.patience} epochs')

# Data
print('==> Preparing data..')
transform_train = transforms.Compose([
    transforms.RandomCrop(32, padding=4),
    transforms.RandomHorizontalFlip(),
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
])

transform_test = transforms.Compose([
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
])

trainset = torchvision.datasets.CIFAR10(
    root='./data', train=True, download=True, transform=transform_train)
trainloader = torch.utils.data.DataLoader(
    trainset, batch_size=128, shuffle=True, num_workers=2)

# Split test set into validation and test sets
testset_full = torchvision.datasets.CIFAR10(
    root='./data', train=False, download=True, transform=transform_test)

# Set random seed for reproducible split
torch.manual_seed(42)
test_size = len(testset_full)
val_size = test_size // 2  # 5000 images
test_size = test_size - val_size  # 5000 images

valset, testset = torch.utils.data.random_split(testset_full, [val_size, test_size])

valloader = torch.utils.data.DataLoader(
    valset, batch_size=100, shuffle=False, num_workers=2)
testloader = torch.utils.data.DataLoader(
    testset, batch_size=100, shuffle=False, num_workers=2)

print(f'Training set: {len(trainset)} images')
print(f'Validation set: {len(valset)} images')
print(f'Test set: {len(testset)} images')

classes = ('plane', 'car', 'bird', 'cat', 'deer',
           'dog', 'frog', 'horse', 'ship', 'truck')

# Model
print('==> Building model..')
net = ResNet18()
net = net.to(device)
if device == 'cuda':
    net = torch.nn.DataParallel(net)
    cudnn.benchmark = True

if args.resume:
    # Load checkpoint.
    print('==> Resuming from checkpoint..')
    assert os.path.isdir('checkpoint'), 'Error: no checkpoint directory found!'
    checkpoint = torch.load('./checkpoint/ckpt.pth')
    net.load_state_dict(checkpoint['net'])
    best_acc = checkpoint['acc']
    start_epoch = checkpoint['epoch']

criterion = nn.CrossEntropyLoss()
optimizer = optim.SGD(net.parameters(), lr=args.lr,
                      momentum=0.9, weight_decay=5e-4)
scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(optimizer, T_max=200)


# Training
def train(epoch):
    print('\nEpoch: %d' % epoch)
    net.train()
    train_loss = 0
    correct = 0
    total = 0
    for batch_idx, (inputs, targets) in enumerate(trainloader):
        inputs, targets = inputs.to(device), targets.to(device)
        optimizer.zero_grad()
        outputs = net(inputs)
        loss = criterion(outputs, targets)
        loss.backward()
        optimizer.step()

        train_loss += loss.item()
        _, predicted = outputs.max(1)
        total += targets.size(0)
        correct += predicted.eq(targets).sum().item()

        progress_bar(batch_idx, len(trainloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
                     % (train_loss/(batch_idx+1), 100.*correct/total, correct, total))


def test(epoch, dataloader=None, is_validation=False):
    if dataloader is None:
        dataloader = testloader
        set_name = "Test"
    else:
        set_name = "Validation" if is_validation else "Test"
    
    print(f'==> {set_name}ing..')
    global best_acc
    net.eval()
    test_loss = 0
    correct = 0
    total = 0
    with torch.no_grad():
        for batch_idx, (inputs, targets) in enumerate(dataloader):
            inputs, targets = inputs.to(device), targets.to(device)
            outputs = net(inputs)
            loss = criterion(outputs, targets)

            test_loss += loss.item()
            _, predicted = outputs.max(1)
            total += targets.size(0)
            correct += predicted.eq(targets).sum().item()

            progress_bar(batch_idx, len(dataloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
                         % (test_loss/(batch_idx+1), 100.*correct/total, correct, total))

    # Calculate accuracy
    acc = 100.*correct/total
    avg_loss = test_loss / len(dataloader)
    
    print(f'{set_name} Results - Loss: {avg_loss:.3f} | Accuracy: {acc:.2f}% ({correct}/{total})')
    
    # Save checkpoint if validation accuracy improved (only during training)
    if is_validation and acc > best_acc:
        print('Saving..')
        state = {
            'net': net.state_dict(),
            'acc': acc,
            'epoch': epoch,
        }
        if not os.path.isdir('checkpoint'):
            os.mkdir('checkpoint')
        torch.save(state, './checkpoint/ckpt.pth')
        best_acc = acc
        print(f'New best validation accuracy: {best_acc:.2f}%')
    
    return acc


# Training loop
print('==> Starting training..')

# Early stopping variables
best_val_acc = 0
patience_counter = 0

for epoch in range(start_epoch, start_epoch+200):
    train(epoch)
    # Use validation set during training for model selection
    val_acc = test(epoch, valloader, is_validation=True)
    
    # Early stopping check
    if val_acc > best_val_acc:
        best_val_acc = val_acc
        patience_counter = 0
        print(f'Epoch {epoch}: Validation accuracy improved to {best_val_acc:.2f}%')
    else:
        patience_counter += 1
        print(f'Epoch {epoch}: No improvement for {patience_counter} epochs (best: {best_val_acc:.2f}%)')
        
        if patience_counter >= args.patience:
            print(f'Early stopping triggered after {patience_counter} epochs without improvement')
            print(f'Best validation accuracy: {best_val_acc:.2f}%')
            break
    
    scheduler.step()

# Final evaluation on test set (unseen during training)
print('==> Final evaluation on test set..')
final_test_acc = test(epoch, testloader, is_validation=False)
print(f'Final test accuracy: {final_test_acc:.2f}%')
