from typing import List, Tuple, Dict, Any, Optional, Union, Callable
from functools import partial

import torch
import torch.nn as nn
from transformers.cache_utils import (
    Cache,
    DynamicCache,
)
from transformers.modeling_outputs import (
    BaseModelOutputWithPast,
    CausalLMOutputWithPast,
)
from transformers.processing_utils import Unpack
from transformers.modeling_flash_attention_utils import FlashAttentionKwargs
from transformers.models.llama.modeling_llama import (
    LlamaConfig,
    LlamaMLP,
    LlamaAttention,
    apply_rotary_pos_emb,
    eager_attention_forward,
    ALL_ATTENTION_FUNCTIONS,
    LlamaDecoderLayer,
    LlamaModel,
    LlamaForCausalLM,
    KwargsForCausalLM,
)

class LoraxLlamaMLP(LlamaMLP):
    def __init__(self, config, layer_idx):
        super().__init__(config)
        self.layer_idx = layer_idx
    
    def forward(
        self,
        x: torch.Tensor,
        lora_B_idx: torch.LongTensor = None,
        contrastive_targets: List[str] = None,
        representations: List[torch.Tensor] = None,
        representations_for_probe: Optional[Dict] = None,
    ):
        
        if f"layers.{self.layer_idx}.mlp.gate_proj" in contrastive_targets:
            gate_proj = self.gate_proj(
                x,
                lora_B_idx,
                should_store=True,
                representations=representations,
            )
        else:
            gate_proj = self.gate_proj(
                x,
                lora_B_idx,
                should_store=False,
            )
        if f"layers.{self.layer_idx}.mlp.up_proj" in contrastive_targets:
            up_proj = self.up_proj(
                x,
                lora_B_idx,
                should_store=True,
                representations=representations,
            )
        else:
            up_proj = self.up_proj(
                x,
                lora_B_idx,
                should_store=False,
            )
        if f"layers.{self.layer_idx}.mlp.down_proj" in contrastive_targets:
            down_proj = self.down_proj(
                self.act_fn(gate_proj) * up_proj,
                lora_B_idx,
                should_store=True,
                representations=representations,
            )
        else:
            down_proj = self.down_proj(
                self.act_fn(gate_proj) * up_proj,
                lora_B_idx,
                should_store=False,
            )
        if representations_for_probe is not None:
            representations_for_probe[f"layers.{self.layer_idx}.mlp.gate_proj"].append(gate_proj.squeeze(0)[-1].detach().cpu().float().numpy())
            representations_for_probe[f"layers.{self.layer_idx}.mlp.up_proj"].append(up_proj.squeeze(0)[-1].detach().cpu().float().numpy())
            representations_for_probe[f"layers.{self.layer_idx}.mlp.down_proj"].append(down_proj.squeeze(0)[-1].detach().cpu().float().numpy())
        return down_proj
    

class LoraxLlamaAttention(LlamaAttention):
    def __init__(self, config: LlamaConfig, layer_idx: int):
       super().__init__(config, layer_idx)
    
    def forward(
        self,
        hidden_states: torch.Tensor,
        position_embeddings: Tuple[torch.Tensor, torch.Tensor],
        attention_mask: Optional[torch.Tensor],
        past_key_value: Optional[Cache] = None,
        cache_position: Optional[torch.LongTensor] = None,
        lora_B_idx: torch.LongTensor = None,
        contrastive_targets: List[str] = None,
        representations: List[torch.Tensor] = None,
        representations_for_probe: Optional[Dict] = None,
        **kwargs: Unpack[FlashAttentionKwargs],
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        input_shape = hidden_states.shape[:-1]
        hidden_shape = (*input_shape, -1, self.head_dim)

        if f"layers.{self.layer_idx}.self_attn.q_proj" in contrastive_targets:
            query_states = self.q_proj(
                hidden_states,
                lora_B_idx,
                should_store=True,
                representations=representations,
            )
        else:
            query_states = self.q_proj(
                hidden_states,
                lora_B_idx,
                should_store=False,
            )
        if f"layers.{self.layer_idx}.self_attn.k_proj" in contrastive_targets:
            key_states = self.k_proj(
                hidden_states,
                lora_B_idx,
                should_store=True,
                representations=representations,
            )
        else:
            key_states = self.k_proj(
                hidden_states,
                lora_B_idx,
                should_store=False,
            )
        if f"layers.{self.layer_idx}.self_attn.v_proj" in contrastive_targets:
            value_states = self.v_proj(
                hidden_states,
                lora_B_idx,
                should_store=True,
                representations=representations,
            )
        else:
            value_states = self.v_proj(
                hidden_states,
                lora_B_idx,
                should_store=False,
            )

        if representations_for_probe is not None:
            representations_for_probe[f"layers.{self.layer_idx}.self_attn.q_proj"].append(query_states.squeeze(0)[-1].detach().cpu().float().numpy())
            representations_for_probe[f"layers.{self.layer_idx}.self_attn.k_proj"].append(key_states.squeeze(0)[-1].detach().cpu().float().numpy())
            representations_for_probe[f"layers.{self.layer_idx}.self_attn.v_proj"].append(value_states.squeeze(0)[-1].detach().cpu().float().numpy())

        
        query_states = query_states.view(hidden_shape).transpose(1, 2)
        key_states = key_states.view(hidden_shape).transpose(1, 2)
        value_states = value_states.view(hidden_shape).transpose(1, 2)

        
        cos, sin = position_embeddings
        query_states, key_states = apply_rotary_pos_emb(query_states, key_states, cos, sin)

        if past_key_value is not None:
            # sin and cos are specific to RoPE models; cache_position needed for the static cache
            cache_kwargs = {"sin": sin, "cos": cos, "cache_position": cache_position}
            key_states, value_states = past_key_value.update(key_states, value_states, self.layer_idx, cache_kwargs)

        attention_interface: Callable = eager_attention_forward
        if self.config._attn_implementation != "eager":
            attention_interface = ALL_ATTENTION_FUNCTIONS[self.config._attn_implementation]

        attn_output, attn_weights = attention_interface(
            self,
            query_states,
            key_states,
            value_states,
            attention_mask,
            dropout=0.0 if not self.training else self.attention_dropout,
            scaling=self.scaling,
            **kwargs,
        )

        attn_output = attn_output.reshape(*input_shape, -1).contiguous()
        if f"layers.{self.layer_idx}.self_attn.o_proj" in contrastive_targets:
            attn_output = self.o_proj(
                attn_output,
                lora_B_idx,
                should_store=True,
                representations=representations,
            )
        else:
            attn_output = self.o_proj(
                attn_output,
                lora_B_idx,
                should_store=False,
            )
        if representations_for_probe is not None:
            representations_for_probe[f"layers.{self.layer_idx}.self_attn.o_proj"].append(attn_output.squeeze(0)[-1].detach().cpu().float().numpy())
        return attn_output, attn_weights


class LoraxLlamaDecoderLayer(LlamaDecoderLayer):
    def __init__(self, config: LlamaConfig, layer_idx: int):
        super().__init__(config, layer_idx)
        self.self_attn = LoraxLlamaAttention(config, layer_idx)
        self.mlp = LoraxLlamaMLP(config, layer_idx)
    
    def forward(
        self,
        hidden_states: torch.Tensor,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_value: Optional[Cache] = None,
        output_attentions: Optional[bool] = False,
        use_cache: Optional[bool] = False,
        cache_position: Optional[torch.LongTensor] = None,
        position_embeddings: Optional[Tuple[torch.Tensor, torch.Tensor]] = None,  # will become mandatory in v4.46
        lora_B_idx: torch.LongTensor = None,
        contrastive_targets: List[str] = None,
        representations: List[torch.Tensor] = None,
        representations_for_probe: Optional[Dict] = None,
        **kwargs,
    ) -> Tuple[torch.FloatTensor, Optional[Tuple[torch.FloatTensor, torch.FloatTensor]]]:
        
        residual = hidden_states

        hidden_states = self.input_layernorm(hidden_states)

        # Self Attention
        hidden_states, self_attn_weights = self.self_attn(
            hidden_states=hidden_states,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_value=past_key_value,
            output_attentions=output_attentions,
            use_cache=use_cache,
            cache_position=cache_position,
            position_embeddings=position_embeddings,
            lora_B_idx=lora_B_idx,
            contrastive_targets=contrastive_targets,
            representations=representations,
            representations_for_probe=representations_for_probe,
            **kwargs,
        )
        hidden_states = residual + hidden_states

        # Fully Connected
        residual = hidden_states
        hidden_states = self.post_attention_layernorm(hidden_states)
        hidden_states = self.mlp(
            hidden_states,
            lora_B_idx=lora_B_idx,
            contrastive_targets=contrastive_targets,
            representations=representations,
            representations_for_probe=representations_for_probe,
        )
        hidden_states = residual + hidden_states

        outputs = (hidden_states,)

        if output_attentions:
            outputs += (self_attn_weights,)

        return outputs
      

class LoraxLlamaModel(LlamaModel):
    def __init__(self, config: LlamaConfig):
        super().__init__(config)
        
        self.layers = nn.ModuleList(
            [LoraxLlamaDecoderLayer(config, layer_idx) for layer_idx in range(config.num_hidden_layers)]
        )
        
        # Initialize weights and apply final processing
        self.post_init()
    
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Union[Cache, List[torch.FloatTensor]]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        cache_position: Optional[torch.LongTensor] = None,
        lora_B_idx: torch.LongTensor = None,
        contrastive_targets: List[str] = None,
        representations: List[torch.Tensor] = None,
        representations_for_probe: Optional[Dict] = None,
        **flash_attn_kwargs: Unpack[FlashAttentionKwargs],
    ) -> BaseModelOutputWithPast:
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache

        if (input_ids is None) ^ (inputs_embeds is not None):
            raise ValueError("You must specify exactly one of input_ids or inputs_embeds")

        if self.gradient_checkpointing and self.training and use_cache:
            use_cache = False
        
        # TODO (joao): remove this exception in v4.56 -- it exists for users that try to pass a legacy cache
        if not isinstance(past_key_values, (type(None), Cache)):
            raise ValueError("The `past_key_values` should be either a `Cache` object or `None`.")

        if inputs_embeds is None:
            inputs_embeds = self.embed_tokens(input_ids)

        if use_cache and past_key_values is None:
            past_key_values = DynamicCache()

        if cache_position is None:
            past_seen_tokens = past_key_values.get_seq_length() if past_key_values is not None else 0
            cache_position = torch.arange(
                past_seen_tokens, past_seen_tokens + inputs_embeds.shape[1], device=inputs_embeds.device
            )

        if position_ids is None:
            position_ids = cache_position.unsqueeze(0)

        causal_mask = self._update_causal_mask(
            attention_mask, inputs_embeds, cache_position, past_key_values, output_attentions
        )

        hidden_states = inputs_embeds

        # create position embeddings to be shared across the decoder layers
        position_embeddings = self.rotary_emb(hidden_states, position_ids)

        # decoder layers
        all_hidden_states = () if output_hidden_states else None
        all_self_attns = () if output_attentions else None

        for decoder_layer in self.layers[: self.config.num_hidden_layers]:
            if output_hidden_states:
                all_hidden_states += (hidden_states,)

            if self.gradient_checkpointing and self.training:
                layer_outputs = self._gradient_checkpointing_func(
                    partial(decoder_layer.__call__, **flash_attn_kwargs),
                    hidden_states,
                    causal_mask,
                    position_ids,
                    past_key_values,
                    output_attentions,
                    use_cache,
                    cache_position,
                    position_embeddings,
                    lora_B_idx,
                    contrastive_targets,
                    representations,
                    representations_for_probe,
                )
            else:
                layer_outputs = decoder_layer(
                    hidden_states,
                    attention_mask=causal_mask,
                    position_ids=position_ids,
                    past_key_value=past_key_values,
                    output_attentions=output_attentions,
                    use_cache=use_cache,
                    cache_position=cache_position,
                    position_embeddings=position_embeddings,
                    lora_B_idx=lora_B_idx,
                    contrastive_targets=contrastive_targets,
                    representations=representations,
                    representations_for_probe=representations_for_probe,
                    **flash_attn_kwargs,
                )
            hidden_states = layer_outputs[0]

            if output_attentions:
                all_self_attns += (layer_outputs[1],)

        hidden_states = self.norm(hidden_states)

        # add hidden states from the last decoder layer
        if output_hidden_states:
            all_hidden_states += (hidden_states,)

        return BaseModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=past_key_values if use_cache else None,
            hidden_states=all_hidden_states,
            attentions=all_self_attns,
        )

class LoraxLlamaForCausalLM(LlamaForCausalLM):
    _tied_weights_keys = ["lm_head.weight"]
    _tp_plan = {"lm_head": "colwise_rep"}
    _pp_plan = {"lm_head": (["hidden_states"], ["logits"])}
    
    def __init__(self, config):
        super().__init__(config)
        self.model = LoraxLlamaModel(config)
        
        # Initialize weights and apply final processing
        self.post_init()
    
    def forward(
        self,
        input_ids: Optional[torch.LongTensor] = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Cache] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        cache_position: Optional[torch.LongTensor] = None,
        logits_to_keep: Union[int, torch.Tensor] = 0,
        lora_B_idx: torch.LongTensor = None,
        contrastive_targets: List[str] = None,
        representations: List[torch.Tensor] = None,
        representations_for_probe: Optional[Dict] = None,
        **kwargs: Unpack[KwargsForCausalLM],
    ) -> CausalLMOutputWithPast:
        
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )

        # decoder outputs consists of (dec_features, layer_state, dec_hidden, dec_attn)
        outputs: BaseModelOutputWithPast = self.model(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            cache_position=cache_position,
            lora_B_idx=lora_B_idx,
            contrastive_targets=contrastive_targets,
            representations=representations,
            representations_for_probe=representations_for_probe,
            **kwargs,
        )

        hidden_states = outputs.last_hidden_state
        # Only compute necessary logits, and do not upcast them to float if we are not computing the loss
        slice_indices = slice(-logits_to_keep, None) if isinstance(logits_to_keep, int) else logits_to_keep
        logits = self.lm_head(hidden_states[:, slice_indices, :])

        loss = None
        if labels is not None:
            loss = self.loss_function(logits=logits, labels=labels, vocab_size=self.config.vocab_size, **kwargs)

        return CausalLMOutputWithPast(
            loss=loss,
            logits=logits,
            past_key_values=outputs.past_key_values,
            hidden_states=outputs.hidden_states,
            attentions=outputs.attentions,
        ) 