from torch.utils.data import Dataset

from expelliarmus import Wizard
from expelliarmus.wizard.clib import event_t

import glob

import os

class NCars(Dataset):
    """
    Neuromorphic-Cars dataset loader.

    Code adapted from
        - https://github.com/neuromorphs/tonic/blob/develop/tonic/prototype/datasets/ncars.py
    due to the usage of the deprecated functionality in the `torchdata` library.
    """

    _DTYPE = event_t
    _TRAIN_PATH = "n-cars_train"
    _TEST_PATH = "n-cars_test"
    sensor_size = [120, 100, 2]

    def __init__(self, root: str, train=True, transform=None, target_transform=None, download=False):
        self.root = root
        self.train = train

        self.transform = transform
        self.target_transform = target_transform

        self._wizard = Wizard(encoding="dat")

        if not os.path.exists(f'{root}/NCARS'):
            print(f"Error: the dataset files could not be found in '{root}/NCARS'. You should download the dataset and manually extract it and, then, provide the path to the extracted archive as root.")
        
        if train:
            self.files = glob.glob(f"{self.root}/NCARS/{self._TRAIN_PATH}/**/*.dat", recursive=True)
        else:
            self.files = glob.glob(f"{self.root}/NCARS/{self._TEST_PATH}/**/*.dat", recursive=True)

    def __len__(self) -> int:
        return 7482 + 7940 if self.train else 4396 + 4211
    
    def __getitem__(self, index: int):
        data, target = self._wizard.read(self.files[index]), self._get_target(self.files[index])

        if self.transform is not None:
            data = self.transform(data)

        if self.target_transform is not None:
            target = self.target_transform(target)

        return data, target

    def _get_target(self, fname: str) -> int:
        folder_name = fname.split(os.sep)[-2]
        assert (
            folder_name == "background" or folder_name == "cars"
        ), f'Error, the folder name "{folder_name}" is wrong and cannot be associated to a label.'
        return 0 if folder_name == "background" else 1
