from recordclass import RecordClass

import torch

class QIFParameters(RecordClass):
    """Parametrization of an Quadratic Leaky Integrate and Fire neuron

    Parameters:
        tau_syn_inv (torch.Tensor): inverse synaptic time
                                    constant (:math:`1/\\tau_\\text{syn}`) in 1/ms
        tau_mem_inv (torch.Tensor): inverse membrane time
                                    constant (:math:`1/\\tau_\\text{mem}`) in 1/ms
        v_leak (torch.Tensor): leak potential in mV
        v_th (torch.Tensor): threshold potential in mV
        v_reset (torch.Tensor): reset potential in mV
        v_c (torch.Tensor): critical potential in mV
        a (torch.Tensor): parameter for the quadratic term
        method (str): method to determine the spike threshold
                      (relevant for surrogate gradients)
        alpha (float): hyper parameter to use in surrogate gradient computation
    """
    tau_syn_inv: torch.Tensor = torch.as_tensor(1.0 / 5e-3)
    tau_mem_inv: torch.Tensor = torch.as_tensor(1.0 / 1e-2)
    v_leak: torch.Tensor = torch.as_tensor(0.0)
    v_th: torch.Tensor = torch.as_tensor(1.0)
    v_reset: torch.Tensor = torch.as_tensor(0.0)
    v_c: torch.Tensor = torch.as_tensor(0.5)
    a: torch.Tensor = torch.as_tensor(1.0)
    method: str = "super"
    alpha: torch.Tensor = torch.as_tensor(1.0)