import json
from pathlib import Path

import download_and_preprocess_carte
import download_ttb
import download_wikidbs
import make_config_file
import preprocess_ttb
import preprocess_wikidbs
import check_linked_tables
import make_tables_with_linked_entities
import balance_classification_data


def main():
    """
    Main function to download and preprocess all datasets.
    """
    # 1. Download datasets
    print("--- Downloading TTB datasets ---")
    download_ttb.main()

    print("\n--- Downloading WikiDBs datasets ---")
    download_wikidbs.download_and_extract_wikidbs()
    download_wikidbs.remove_unecessary_files()

    print("\n--- Downloading CARTE datasets ---")
    download_and_preprocess_carte._download_raw()

    # 2. Preprocess datasets
    print("\n--- Preprocessing TTB datasets ---")
    with open(Path(__file__).parent / "ttb_data_config.json", "r") as f:
        data_configs = json.load(f)
    with open(Path(__file__).parent / "ttb_extra_config.json", "r") as f:
        extra_configs = json.load(f)
    ttb_configs = {**data_configs, **extra_configs}
    for dataset_name, dataset_config in ttb_configs.items():
        preprocess_ttb.preprocess_data(dataset_name, dataset_config)
        preprocess_ttb.add_label_encoding(dataset_name, dataset_config)

    print("\n--- Preprocessing WikiDBs datasets ---")
    preprocess_wikidbs.main()

    print("\n--- Preprocessing CARTE datasets ---")
    download_and_preprocess_carte._download_preprocessed(include_llm=False)
    download_and_preprocess_carte._process_carte_data()

    print("\n--- Creating config file ---")
    make_config_file.main()

    print("\n--- Linking tables ---")
    make_tables_with_linked_entities.link_tables()
    check_linked_tables.main()

    # 3. Balance classification datasets
    print("\n--- Balancing classification datasets ---")
    balance_classification_data.main()

    # 4. Create final config file
    print("\n--- Creating final config file ---")
    make_config_file.main()

    print("\n--- All datasets downloaded and preprocessed successfully! ---")


if __name__ == "__main__":
    main()
