# System Parser

A comprehensive library for parsing, validating, and analyzing power system configurations.

## Overview

The System Parser package provides a modular framework for working with power system configurations, transforming complex system definitions into validated, analyzable graph representations.

### Key Features

- **Modular Architecture**: Clean separation between parsing, validation, and analysis
- **Comprehensive Analysis**: Connectivity, electrical parameters, and topology analysis
- **Robust Validation**: Multi-layer validation with detailed error reporting
- **Pandapower Integration**: Seamless conversion to Pandapower format
- **Analysis Reporting**: Block effectiveness and voltage violation analysis

## Quick Start

```python
from system_parser import SystemGraph, GraphBuilder, SystemValidator

# Create system
system_dict = {
    "blocks": {
        "gen1": {"type": "Three-Phase Source", "params": {"voltage": 11000}},
        "load1": {"type": "Three-Phase Parallel RLC Load", "params": {"power": 1000}}
    },
    "connections": [{"from": "gen1/P1", "to": "load1/P1"}]
}

# Build and validate
builder = GraphBuilder(config_file="config/block_config.json")
system = builder.build_from_dict(system_dict)

validator = SystemValidator()
result = validator.validate(system)

if result.is_valid:
    print("✅ System is valid!")
    connectivity = system.analyze_connectivity()
    print(f"Load satisfaction: {connectivity.connectivity_ratio:.2%}")
else:
    print(f"❌ Validation failed: {result.summary}")
```

## Analysis Reporting

Generate concise analysis reports for system effectiveness and voltage violations:

```python
from system_parser.graph.analysis_reporter import (
    generate_effectiveness_report,
    generate_voltage_violations_report,
    generate_system_analysis_report
)

# Generate reports
effectiveness_report = generate_effectiveness_report(system_graph)
voltage_report = generate_voltage_violations_report(system_graph)
combined_report = generate_system_analysis_report(system_graph)
```

### Block Effectiveness Analysis

Evaluates how well blocks are connected based on their type:
- **Three-Phase V-I Measurement**: Both input and output sides should be connected
- **Three-Phase Parallel RLC Load**: All three phases should be connected

### Voltage Violations Analysis

Identifies voltage mismatches between connected blocks with severity classification.

## Architecture

```
system_parser/
├── core/                 # Base models and interfaces
├── graph/               # Graph construction and analysis
├── validation/          # Validation framework
├── analysis/            # Analysis components
├── config/              # Configuration management
├── validators/          # Specialized validators
├── visualization/       # Graph rendering
└── pandapower/         # Pandapower integration
```

## Core Classes

- **`SystemGraph`**: Main graph representation of the power system
- **`GraphBuilder`**: Constructs SystemGraph from various input formats
- **`SystemValidator`**: Orchestrates validation across multiple validators
- **`AnalysisReporter`**: Provides system effectiveness and voltage analysis

## Configuration

Block types and validation parameters are configured through JSON files:
- `config/block_config.json`: Block type definitions
- `config/validation_config.json`: Validation parameters

## Testing

```bash
# Run all tests
python -m pytest system_parser/tests/

# Run with coverage
python -m pytest system_parser/tests/ --cov=system_parser
```