#!/usr/bin/env python3
"""
Demonstration script for the Analysis Reporter functionality.

This script shows how to use the new concise reporting methods for:
1. Block effectiveness analysis
2. Voltage violations analysis
3. Combined system analysis
"""

import sys
import os
import json

# Add parent directories to path for imports
sys.path.append(os.path.join(os.path.dirname(__file__), '..', '..'))

from system_parser.system_graph import SystemGraph
from system_parser.graph.analysis_reporter import (
    generate_effectiveness_report,
    generate_voltage_violations_report,
    generate_system_analysis_report,
    AnalysisReporter
)

# Example system with both effective and ineffective blocks
EXAMPLE_SYSTEM = {
    "Blocks": {
        "Bus1": {
            "Type": "Three-Phase V-I Measurement"
        },
        "Bus2": {
            "Type": "Three-Phase V-I Measurement"
        },
        "Gen1": {
            "Active power P (W)": "50000",
            "Nominal phase-to-phase voltage Vn (Vrms)": 13800.0,
            "Type": "Three-Phase Source"
        },
        "Load1": {
            "Active power P (W)": "25000",
            "Nominal phase-to-phase voltage Vn (Vrms)": 480.0,
            "Type": "Three-Phase Parallel RLC Load"
        },
        "Load2": {
            "Active power P (W)": "25000",
            "Nominal phase-to-phase voltage Vn (Vrms)": 480.0,
            "Type": "Three-Phase Parallel RLC Load"
        }
    },
    "Connections": [
        # Bus2 and Load2 are fully connected
        [
            "Bus2/(a2,b2,c2)",
            "Load2/(a1,b1,c1)"
        ],
        [
            "Gen1/(a1,b1,c1)",
            "Bus2/(a1,b1,c1)"
        ],
        # Bus1 and Load1 are only partially connected (demonstrating ineffectiveness)
        [
            "Bus1/(a2)",
            "Load1/(a1)"
        ]
    ]
}

def main():
    """Demonstrate the analysis reporting functionality."""
    print("="*60)
    print("POWER SYSTEM ANALYSIS REPORTER DEMONSTRATION")
    print("="*60)
    print()
    
    # Create system graph
    print("Creating system graph from example data...")
    system_graph = SystemGraph(EXAMPLE_SYSTEM)
    print("✓ System graph created successfully")
    print()
    
    # Method 1: Using convenience functions
    print("--- METHOD 1: Using Convenience Functions ---")
    print()
    
    print("1. Block Effectiveness Report:")
    print("-" * 30)
    effectiveness_report = generate_effectiveness_report(system_graph)
    print(effectiveness_report)
    print()
    
    print("2. Voltage Violations Report:")
    print("-" * 30)
    voltage_report = generate_voltage_violations_report(system_graph)
    print(voltage_report)
    print()
    
    print("3. Combined System Analysis Report:")
    print("-" * 35)
    combined_report = generate_system_analysis_report(system_graph)
    print(combined_report)
    print()
    
    # Method 2: Using the AnalysisReporter class directly
    print("--- METHOD 2: Using AnalysisReporter Class ---")
    print()
    
    reporter = AnalysisReporter()
    
    # Get structured data for programmatic use
    effectiveness_data = reporter.generate_block_effectiveness_report(system_graph)
    voltage_data = reporter.generate_voltage_violations_report(system_graph)
    
    print("Structured Effectiveness Data:")
    print(f"  • Total blocks analyzed: {effectiveness_data.total_blocks}")
    print(f"  • Effective blocks: {effectiveness_data.effective_blocks}")
    print(f"  • System effectiveness score: {effectiveness_data.system_score:.3f}")
    print(f"  • Number of ineffective blocks: {len(effectiveness_data.ineffective_block_details)}")
    
    if effectiveness_data.ineffective_block_details:
        print("  • Ineffective block details:")
        for block in effectiveness_data.ineffective_block_details:
            print(f"    - {block['name']}: {block['effectiveness_score']:.2f} effectiveness")
    print()
    
    print("Structured Voltage Data:")
    print(f"  • Total violations: {voltage_data.total_violations}")
    print(f"  • Coherence score: {voltage_data.coherence_score:.3f}")
    print(f"  • Has violations: {voltage_data.has_violations}")
    print()
    
    # Method 3: Filtering by specific block types
    print("--- METHOD 3: Filtered Analysis ---")
    print()
    
    # Only analyze measurement blocks
    measurement_types = {"Three-Phase V-I Measurement"}
    measurement_report = generate_effectiveness_report(system_graph, measurement_types)
    print("Analysis for Three-Phase V-I Measurement blocks only:")
    print(measurement_report)
    print()
    
    # Only analyze load blocks
    load_types = {"Three-Phase Parallel RLC Load"}
    load_report = generate_effectiveness_report(system_graph, load_types)
    print("Analysis for Three-Phase Parallel RLC Load blocks only:")
    print(load_report)
    print()
    
    print("="*60)
    print("DEMONSTRATION COMPLETE")
    print("="*60)
    print()
    print("Key findings from this example:")
    print("• Bus1 is ineffective (only 1/6 ports connected)")
    print("• Load1 is ineffective (only 1/3 ports connected)")
    print("• Bus2 and Load2 are fully effective")
    print("• No voltage violations detected")
    print("• Overall system effectiveness: 40% (2/5 critical blocks effective)")

if __name__ == "__main__":
    main() 