"""
Analysis Reporter Module

Provides concise reporting utilities for power system analysis,
including block effectiveness and voltage violation reports.
"""

from typing import Dict, List, Optional, Set, Any
from dataclasses import dataclass, field


@dataclass
class BlockEffectivenessResult:
    """Result of block effectiveness evaluation."""
    block_name: str
    block_type: str
    effectiveness_score: float  # 0.0 to 1.0
    is_effective: bool
    connected_ports: Set[str]
    unconnected_ports: Set[str]
    total_ports: int
    evaluation_details: Dict[str, Any] = field(default_factory=dict)
    
    def __post_init__(self):
        if self.evaluation_details is None:
            self.evaluation_details = {}


@dataclass
class EffectivenessReport:
    """Concise block effectiveness analysis report."""
    
    total_blocks: int
    effective_blocks: int
    ineffective_blocks: int
    effectiveness_ratio: float
    ineffective_block_details: List[Dict[str, Any]]
    system_score: float
    
    def to_concise_text(self) -> str:
        """Generate a concise text report."""
        lines = [
            f"Block Effectiveness Summary:",
            f"  Total blocks: {self.total_blocks}",
            f"  Effective: {self.effective_blocks} ({self.effectiveness_ratio:.1%})",
            f"  Ineffective: {self.ineffective_blocks}",
            f"  System score: {self.system_score:.3f}",
        ]
        
        if self.ineffective_block_details:
            lines.append("  Ineffective blocks:")
            for block in self.ineffective_block_details:
                connected = block['connected_ports']
                total = block['total_ports']
                lines.append(f"    • {block['name']} ({block['type']}): {connected}/{total} ports connected")
                if block['critical_ports']:
                    lines.append(f"      Missing critical: {', '.join(block['critical_ports'])}")
        
        return '\n'.join(lines)


@dataclass
class VoltageReport:
    """Concise voltage violations analysis report."""
    
    total_violations: int
    has_violations: bool
    violation_details: List[Dict[str, Any]]
    coherence_score: float
    
    def to_concise_text(self) -> str:
        """Generate a concise text report."""
        lines = [
            f"Voltage Analysis Summary:",
            f"  Total violations: {self.total_violations}",
            f"  Coherence score: {self.coherence_score:.3f}",
        ]
        
        if self.has_violations:
            lines.append("  Voltage violations:")
            for violation in self.violation_details:
                lines.append(f"    • {violation['description']}")
                lines.append(f"      Severity: {violation['severity']}")
        else:
            lines.append("  ✓ No voltage violations detected")
        
        return '\n'.join(lines)


class AnalysisReporter:
    """
    Main analysis reporter for generating concise system analysis reports.
    
    This class provides methods to generate focused, actionable reports
    about system issues like ineffective blocks and voltage violations.
    """
    
    def __init__(self):
        pass
    
    def _evaluate_block_effectiveness(self, block_name: str, block_type: str, 
                                    all_ports: Set[str], unconnected_ports: Set[str]) -> BlockEffectivenessResult:
        """Evaluate a single block's effectiveness using simplified logic."""
        connected_ports = all_ports - unconnected_ports
        total_ports = len(all_ports)
        
        # Simplified effectiveness evaluation
        if block_type == "Three-Phase V-I Measurement":
            # Both input and output sides should be connected
            input_ports = {"a2", "b2", "c2"}
            output_ports = {"a1", "b1", "c1"}
            
            input_connected = len(input_ports.intersection(connected_ports)) > 0
            output_connected = len(output_ports.intersection(connected_ports)) > 0
            
            is_effective = input_connected and output_connected
            effectiveness_score = (int(input_connected) + int(output_connected)) / 2.0
            
        elif block_type == "Three-Phase Parallel RLC Load":
            # All three phases should be connected
            required_ports = {"a1", "b1", "c1"}
            connected_required = required_ports.intersection(connected_ports)
            
            is_effective = len(connected_required) == 3
            effectiveness_score = len(connected_required) / 3.0
            
        else:
            # Default: effectiveness based on connection ratio
            effectiveness_score = len(connected_ports) / total_ports if total_ports > 0 else 0.0
            is_effective = effectiveness_score >= 0.5
        
        return BlockEffectivenessResult(
            block_name=block_name,
            block_type=block_type,
            effectiveness_score=effectiveness_score,
            is_effective=is_effective,
            connected_ports=connected_ports,
            unconnected_ports=unconnected_ports,
            total_ports=total_ports
        )
    
    def generate_block_effectiveness_report(
        self, 
        system_graph,  # SystemGraph type hint avoided to prevent circular import
        target_block_types: Optional[Set[str]] = None
    ) -> EffectivenessReport:
        """
        Generate a concise report about block effectiveness issues.
        
        Args:
            system_graph: The system graph to analyze
            target_block_types: Block types to focus on. If None, uses default critical types.
            
        Returns:
            EffectivenessReport with detailed information about ineffective blocks
        """
        # Get unconnected ports grouped by block
        unconnected_by_block = system_graph.get_unconnected_ports_by_block()
        
        # Evaluate each block
        results = {}
        for block_name, block in system_graph.blocks.items():
            # Get all ports for this block
            all_ports = set(block.ports.keys())
            
            # Get unconnected ports for this block
            unconnected_ports_list = unconnected_by_block.get(block_name, [])
            unconnected_ports = {port.port_name for port in unconnected_ports_list}
            
            # Evaluate the block
            result = self._evaluate_block_effectiveness(
                block_name, block.type, all_ports, unconnected_ports
            )
            results[block_name] = result
        
        if not results:
            return EffectivenessReport(
                total_blocks=0, effective_blocks=0, ineffective_blocks=0,
                effectiveness_ratio=0.0, ineffective_block_details=[],
                system_score=0.0
            )
        
        # Filter by target types if specified
        if target_block_types:
            filtered_results = {
                name: result for name, result in results.items()
                if result.block_type in target_block_types
            }
        else:
            # Default to critical types (Three-Phase V-I Measurement and Parallel RLC Load)
            critical_types = {"Three-Phase V-I Measurement", "Three-Phase Parallel RLC Load"}
            filtered_results = {
                name: result for name, result in results.items()
                if result.block_type in critical_types
            }
        
        if not filtered_results:
            filtered_results = results  # Fall back to all blocks
        
        # Calculate statistics
        total_blocks = len(filtered_results)
        effective_blocks = sum(1 for r in filtered_results.values() if r.is_effective)
        ineffective_blocks = total_blocks - effective_blocks
        effectiveness_ratio = effective_blocks / total_blocks if total_blocks > 0 else 0.0
        
        # Calculate system score (ratio of effective blocks for critical types)
        if target_block_types is None:
            critical_types = {"Three-Phase V-I Measurement", "Three-Phase Parallel RLC Load"}
            critical_results = {
                name: result for name, result in filtered_results.items()
                if result.block_type in critical_types
            }
            if critical_results:
                system_score = sum(1 for r in critical_results.values() if r.is_effective) / len(critical_results)
            else:
                system_score = effectiveness_ratio
        else:
            system_score = effectiveness_ratio
        
        # Gather details about ineffective blocks
        ineffective_details = []
        for name, result in filtered_results.items():
            if not result.is_effective:
                # Identify critical missing ports based on block type
                critical_ports = self._identify_critical_missing_ports(result)
                
                ineffective_details.append({
                    'name': result.block_name,
                    'type': result.block_type,
                    'connected_ports': len(result.connected_ports),
                    'total_ports': result.total_ports,
                    'effectiveness_score': result.effectiveness_score,
                    'critical_ports': critical_ports,
                    'unconnected_ports': list(result.unconnected_ports)
                })
        
        # Sort by effectiveness score (worst first)
        ineffective_details.sort(key=lambda x: x['effectiveness_score'])
        
        return EffectivenessReport(
            total_blocks=total_blocks,
            effective_blocks=effective_blocks,
            ineffective_blocks=ineffective_blocks,
            effectiveness_ratio=effectiveness_ratio,
            ineffective_block_details=ineffective_details,
            system_score=system_score
        )
    
    def generate_voltage_violations_report(self, system_graph) -> VoltageReport:
        """
        Generate a concise report about voltage violations.
        
        Args:
            system_graph: The system graph to analyze
            
        Returns:
            VoltageReport with detailed information about voltage issues
        """
        violations = system_graph.voltage_violations
        total_violations = len(violations)
        
        # Calculate coherence score (higher is better)
        coherence_score = max(0.0, min(1.0, 1.0 / (1.0 + total_violations)))
        
        # Parse violation details
        violation_details = []
        for violation in violations:
            # Extract key information from violation string
            detail = self._parse_voltage_violation(violation)
            violation_details.append(detail)
        
        return VoltageReport(
            total_violations=total_violations,
            has_violations=total_violations > 0,
            violation_details=violation_details,
            coherence_score=coherence_score
        )
    
    def _identify_critical_missing_ports(self, result: BlockEffectivenessResult) -> List[str]:
        """Identify which missing ports are critical for block functionality."""
        critical_ports = []
        
        # Block-type specific critical port identification
        if result.block_type == "Three-Phase V-I Measurement":
            # For measurement blocks, both sides should be connected
            all_ports = {"a2", "b2", "c2", "a1", "b1", "c1"}
            missing_input = len(all_ports.intersection({"a2", "b2", "c2"}).intersection(result.unconnected_ports)) > 0
            missing_output = len(all_ports.intersection({"a1", "b1", "c1"}).intersection(result.unconnected_ports)) > 0
            
            if missing_input:
                critical_ports.extend([port for port in ["a2", "b2", "c2"] if port in result.unconnected_ports])
            if missing_output:
                critical_ports.extend([port for port in ["a1", "b1", "c1"] if port in result.unconnected_ports])
                
        elif result.block_type == "Three-Phase Parallel RLC Load":
            # For loads, all three phases should be connected
            critical_ports.extend([port for port in ["a1", "b1", "c1"] if port in result.unconnected_ports])
        
        else:
            # For other blocks, consider any unconnected port as potentially critical
            critical_ports = list(result.unconnected_ports)[:3]  # Limit to first 3 for brevity
        
        return critical_ports
    
    def _parse_voltage_violation(self, violation: str) -> Dict[str, Any]:
        """Parse a voltage violation string to extract structured information."""
        # Example: "Voltage mismatch between 'Bus1' (690 V) and 'Bus2' (6600 V): 90.5% difference"
        
        # Extract basic information
        if "Voltage mismatch between" in violation:
            # Simple parsing - could be enhanced with regex for more robustness
            parts = violation.split(":")
            description = parts[0] if parts else violation
            
            # Determine severity based on percentage
            severity = "high"
            if "%" in violation:
                try:
                    # Extract percentage value
                    pct_str = violation.split("%")[0].split()[-1]
                    percentage = float(pct_str)
                    if percentage < 30:
                        severity = "low"
                    elif percentage < 60:
                        severity = "medium"
                except:
                    pass
            
            return {
                'description': description,
                'severity': severity,
                'full_text': violation
            }
        
        return {
            'description': violation,
            'severity': "unknown",
            'full_text': violation
        }
    
    def generate_combined_report(
        self, 
        system_graph,
        target_block_types: Optional[Set[str]] = None
    ) -> str:
        """
        Generate a combined concise report covering both effectiveness and voltage issues.
        
        Args:
            system_graph: The system graph to analyze
            target_block_types: Block types to focus on for effectiveness analysis
            
        Returns:
            Formatted string with combined analysis
        """
        effectiveness_report = self.generate_block_effectiveness_report(system_graph, target_block_types)
        voltage_report = self.generate_voltage_violations_report(system_graph)
        
        lines = [
            "=== POWER SYSTEM ANALYSIS REPORT ===",
            "",
            effectiveness_report.to_concise_text(),
            "",
            voltage_report.to_concise_text(),
            "",
            "=== END REPORT ==="
        ]
        
        return '\n'.join(lines)


# Convenience functions for easy integration
def generate_effectiveness_report(
    system_graph,  # SystemGraph type hint avoided to prevent circular import
    target_block_types: Optional[Set[str]] = None
) -> str:
    """
    Generate a concise block effectiveness report.
    
    Args:
        system_graph: The system graph to analyze
        target_block_types: Block types to focus on
        
    Returns:
        Formatted effectiveness report string
    """
    reporter = AnalysisReporter()
    report = reporter.generate_block_effectiveness_report(system_graph, target_block_types)
    return report.to_concise_text()


def generate_voltage_violations_report(system_graph) -> str:
    """
    Generate a concise voltage violations report.
    
    Args:
        system_graph: The system graph to analyze
        
    Returns:
        Formatted voltage violations report string
    """
    reporter = AnalysisReporter()
    report = reporter.generate_voltage_violations_report(system_graph)
    return report.to_concise_text()


def generate_system_analysis_report(
    system_graph,
    target_block_types: Optional[Set[str]] = None
) -> str:
    """
    Generate a comprehensive system analysis report.
    
    Args:
        system_graph: The system graph to analyze
        target_block_types: Block types to focus on for effectiveness analysis
        
    Returns:
        Formatted comprehensive analysis report string
    """
    reporter = AnalysisReporter()
    return reporter.generate_combined_report(system_graph, target_block_types) 