import json
import sys
import os
from typing import Dict, List, Any, Optional, Union

# Add parent directory to path for imports
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from system_parser.system_graph import SystemGraph
from system_parser.analyzers.electrical_analyzers import ElectricalAnalysisSuite, AnalysisResult


def electrical_check(system_name: str, 
                    checks: Optional[List[str]] = None,
                    context: Optional[Dict[str, Any]] = None) -> str:
    """
    Perform electrical validation checks on a power system using the unified validation system.
    
    Args:
        system_name: Name of variable containing system definition (JSON string or dictionary)
        checks: List of checks to perform. Options: 
               ["voltage", "frequency", "connectivity", "port_connectivity", "completeness", "parameters", "validation"]
               Default None performs all available checks
        context: Execution context containing variables (automatically provided by environment)
    
    Returns:
        Detailed formatted results string
    
    Examples:
        <tool>{"name": "electrical_check", "args": {"system_name": "system_dict", "checks": ["voltage", "frequency"]}}</tool>
        <tool>{"name": "electrical_check", "args": {"system_name": "system_dict", "checks": ["connectivity", "port_connectivity"]}}</tool>
        <tool>{"name": "electrical_check", "args": {"system_name": "system_dict"}}</tool>
    """
    try:
        # Look up the variable by name in the execution context
        if context is None:
            return f"Error: No execution context provided. Cannot look up variable '{system_name}'"
        
        if system_name not in context:
            available_vars = [k for k in context.keys() if not k.startswith('__')]
            return f"Error: Variable '{system_name}' not found in execution context. Available variables: {available_vars}"
        
        system_dict = context[system_name]
        
        # Validate that the variable contains a valid system definition
        if not isinstance(system_dict, (dict, str)):
            return f"Error: Variable '{system_name}' must be a dictionary or JSON string, got {type(system_dict).__name__}"
        
        # Parse system_dict if it's a JSON string
        if isinstance(system_dict, str):
            try:
                system_dict = json.loads(system_dict)
            except json.JSONDecodeError as e:
                return f"Error: Invalid JSON in '{system_name}': {e}"
        
        # Create system graph for validation
        system_graph = SystemGraph(system_dict)
        
        # Create analysis suite and run checks
        analysis_suite = ElectricalAnalysisSuite()
        results = analysis_suite.run_analysis(system_graph, checks)
        
        # Format and return results
        return analysis_suite.format_results(results)
    
    except Exception as e:
        return f"Error during electrical check: {str(e)}"


# Backward compatibility: keep the old class structure but make it use the new system
class ElectricalChecker:
    """
    Legacy electrical checker class for backward compatibility.
    Now uses the unified validation system internally.
    """
    
    def __init__(self, system_dict: Union[str, Dict[str, Any]]):
        """
        Initialize the electrical checker.
        
        Args:
            system_dict: Either a JSON string or dictionary containing system definition
        """
        if isinstance(system_dict, str):
            self.system_dict = json.loads(system_dict)
        else:
            self.system_dict = system_dict
        
        self.system_graph = SystemGraph(self.system_dict)
        self.analysis_suite = ElectricalAnalysisSuite()
    
    def _convert_result(self, analysis_result: AnalysisResult) -> 'ElectricalCheckResult':
        """Convert ValidationResult to legacy ElectricalCheckResult format."""
        from dataclasses import dataclass
        
        @dataclass
        class ElectricalCheckResult:
            """Results from electrical checking operations."""
            check_type: str
            status: str  # "PASS", "FAIL", "WARNING", "INFO"
            violations: List[str]
            score: float  # 0.0 to 1.0
            details: Dict[str, Any]
            summary: str
        
        return ElectricalCheckResult(
            check_type=analysis_result.analyzer_name,
            status=analysis_result.status,
            violations=analysis_result.violations,
            score=analysis_result.score,
            details=analysis_result.details,
            summary=analysis_result.summary
        )
    
    def check_voltage_coherence(self, threshold: float = 0.2) -> 'ElectricalCheckResult':
        """Check voltage coherence across the electrical network."""
        # Update the analysis suite with the new threshold
        self.analysis_suite.analyzers["voltage"].threshold = threshold
        result = self.analysis_suite.run_analysis(self.system_graph, ["voltage"])
        return self._convert_result(result["voltage"])
    
    def check_frequency_coherence(self) -> 'ElectricalCheckResult':
        """Check frequency coherence across all blocks."""
        result = self.analysis_suite.run_analysis(self.system_graph, ["frequency"])
        return self._convert_result(result["frequency"])
    
    def check_connectivity(self) -> 'ElectricalCheckResult':
        """Check electrical connectivity between generators and loads."""
        result = self.analysis_suite.run_analysis(self.system_graph, ["connectivity"])
        return self._convert_result(result["connectivity"])
    
    def check_parameter_validation(self) -> 'ElectricalCheckResult':
        """Check parameter validation issues."""
        result = self.analysis_suite.run_analysis(self.system_graph, ["parameters"])
        return self._convert_result(result["parameters"])
    
    def check_system_completeness(self) -> 'ElectricalCheckResult':
        """Check system completeness by analyzing isolated blocks."""
        result = self.analysis_suite.run_analysis(self.system_graph, ["completeness"])
        return self._convert_result(result["completeness"])
    
    def check_port_connectivity(self) -> 'ElectricalCheckResult':
        """Check port connectivity effectiveness for all blocks."""
        result = self.analysis_suite.run_analysis(self.system_graph, ["port_connectivity"])
        return self._convert_result(result["port_connectivity"])
    
    def check_block_effectiveness(self, target_types: Optional[List[str]] = None) -> 'ElectricalCheckResult':
        """
        Check block effectiveness (port connections).
        Legacy method - now uses port connectivity analyzer.
        """
        result = self.analysis_suite.run_analysis(self.system_graph, ["port_connectivity"])
        converted_result = self._convert_result(result["port_connectivity"])
        # Update the check type to match legacy interface
        converted_result.check_type = "block_effectiveness"
        return converted_result
    
    def run_comprehensive_check(self, checks: Optional[List[str]] = None) -> Dict[str, 'ElectricalCheckResult']:
        """
        Run a comprehensive electrical check with specified aspects.
        
        Args:
            checks: List of checks to perform. Available options:
                   - "voltage": Voltage coherence
                   - "frequency": Frequency coherence  
                   - "connectivity": Generator-to-load connectivity
                   - "effectiveness": Block port effectiveness (basic validation)
                   - "parameters": Parameter validation
                   - "completeness": System completeness (isolated blocks analysis)
        
        Returns:
            Dictionary mapping check names to their results
        """
        if checks is None:
            checks = ["voltage", "frequency", "connectivity", "completeness", "parameters", "validation"]
        
        # Map legacy check names to new analyzer names
        check_mapping = {
            "voltage": "voltage",
            "frequency": "frequency",
            "connectivity": "connectivity",
            "port_connectivity": "port_connectivity",
            "effectiveness": "port_connectivity",  # Map legacy name to new port connectivity
            "parameters": "parameters",
            "completeness": "completeness"
        }
        
        # Convert check names
        analyzer_checks = []
        for check in checks:
            if check in check_mapping:
                analyzer_checks.append(check_mapping[check])
        
        # Remove duplicates while preserving order
        analyzer_checks = list(dict.fromkeys(analyzer_checks))
        
        # Run analysis
        results = self.analysis_suite.run_analysis(self.system_graph, analyzer_checks)
        
        # Convert results back to legacy format
        legacy_results = {}
        for check in checks:
            if check in check_mapping:
                analyzer_name = check_mapping[check]
                if analyzer_name in results:
                    legacy_result = self._convert_result(results[analyzer_name])
                    legacy_result.check_type = check  # Keep original check name
                    legacy_results[check] = legacy_result
        
        return legacy_results
