#!/usr/bin/env python3
"""Test script for the electrical_check tool functionality."""

import json
import sys
import os
from typing import Dict, Any

# Add current directory to path for tool imports
sys.path.append(os.path.dirname(os.path.abspath(__file__)))

from electrical_check import electrical_check

# Test system dictionary with voltage mismatches (same as test_reward_voltage.py)
test_system_dict = {
    "Blocks": {
        "Bus1": {
            "Type": "Three-Phase V-I Measurement"
        },
        "Bus2": {
            "Type": "Three-Phase V-I Measurement"
        },
        "Bus3": {
            "Type": "Three-Phase V-I Measurement"
        },
        "Gen1": {
            "Active power P (W)": "53700",
            "Nominal phase-to-phase voltage Vn (Vrms)": 13800.0,
            "Type": "Three-Phase Source"
        },
        "Line1": {
            "Frequency used for rlc specification (Hz)": 50,
            "Line length (km)": 1.0,
            "Type": "Three-Phase PI Section Line"
        },
        "Line2": {
            "Frequency used for rlc specification (Hz)": 50,
            "Line length (km)": 15.0,
            "Type": "Three-Phase PI Section Line"
        },
        "Load1": {
            "Active power P (W)": "7600",
            "Nominal phase-to-phase voltage Vn (Vrms)": 690,
            "Type": "Three-Phase Parallel RLC Load"
        },
        "Load2": {
            "Active power P (W)": "46000",
            "Nominal phase-to-phase voltage Vn (Vrms)": 6600.0,
            "Type": "Three-Phase Parallel RLC Load"
        }
    },
    "Connections": [
        ["Bus1/(a2,b2,c2)", "Load1/(a1,b1,c1)"],
        ["Bus2/(a2,b2,c2)", "Load2/(a1,b1,c1)"],
        ["Gen1/(a1,b1,c1)", "Bus3/(a1,b1,c1)"],
        ["Bus1/(a2,b2,c2)", "Line1/(a1,b1,c1)"],
        ["Line1/(a2,b2,c2)", "Bus3/(a1,b1,c1)"],
        ["Bus3/(a2,b2,c2)", "Line2/(a1,b1,c1)"],
        ["Line2/(a2,b2,c2)", "Bus2/(a1,b1,c1)"]
    ]
}

# Test system with good electrical coherence
good_system_dict = {
    "Blocks": {
        "Bus1": {
            "Type": "Three-Phase V-I Measurement"
        },
        "Bus2": {
            "Type": "Three-Phase V-I Measurement"
        },
        "Gen1": {
            "Active power P (W)": "50000",
            "Nominal phase-to-phase voltage Vn (Vrms)": 6600.0,
            "Type": "Three-Phase Source"
        },
        "Load1": {
            "Active power P (W)": "25000",
            "Nominal phase-to-phase voltage Vn (Vrms)": 6600.0,
            "Type": "Three-Phase Parallel RLC Load"
        },
        "Load2": {
            "Active power P (W)": "25000",
            "Nominal phase-to-phase voltage Vn (Vrms)": 6600.0,
            "Type": "Three-Phase Parallel RLC Load"
        }
    },
    "Connections": [
        ["Gen1/(a1,b1,c1)", "Bus1/(a1,b1,c1)"],
        ["Bus1/(a2,b2,c2)", "Load1/(a1,b1,c1)"],
        ["Bus1/(a2,b2,c2)", "Bus2/(a1,b1,c1)"],
        ["Bus2/(a2,b2,c2)", "Load2/(a1,b1,c1)"]
    ]
}

def test_summary_format():
    """Test the summary format output."""
    print("="*60)
    print("TEST 1: Summary Format (System with Voltage Issues)")
    print("="*60)
    
    result = electrical_check(test_system_dict, ["all"], "summary")
    print(result)
    print()

def test_detailed_format():
    """Test the detailed format output."""
    print("="*60)
    print("TEST 2: Detailed Format (System with Voltage Issues)")
    print("="*60)
    
    result = electrical_check(test_system_dict, ["all"], "detailed")
    print(result)
    print()

def test_specific_checks():
    """Test specific check types."""
    print("="*60)
    print("TEST 3: Specific Checks (Voltage & Connectivity Only)")
    print("="*60)
    
    result = electrical_check(test_system_dict, ["voltage", "connectivity"], "detailed")
    print(result)
    print()

def test_json_format():
    """Test JSON format output."""
    print("="*60)
    print("TEST 4: JSON Format (First 500 chars)")
    print("="*60)
    
    result = electrical_check(test_system_dict, ["voltage", "frequency"], "json")
    print(result[:500] + "..." if len(result) > 500 else result)
    print()

def test_good_system():
    """Test a system with good electrical coherence."""
    print("="*60)
    print("TEST 5: Good System (Should Pass Most Checks)")
    print("="*60)
    
    result = electrical_check(good_system_dict, ["all"], "summary")
    print(result)
    print()

def test_voltage_only():
    """Test voltage checking only."""
    print("="*60)
    print("TEST 6: Voltage Check Only")
    print("="*60)
    
    result = electrical_check(test_system_dict, ["voltage"], "detailed")
    print(result)
    print()

def test_string_input():
    """Test with JSON string input instead of dictionary."""
    print("="*60)
    print("TEST 7: String Input (JSON)")
    print("="*60)
    
    json_string = json.dumps(good_system_dict)
    result = electrical_check(json_string, ["connectivity", "effectiveness"], "summary")
    print(result)
    print()

def test_error_handling():
    """Test error handling with invalid input."""
    print("="*60)
    print("TEST 8: Error Handling (Invalid Input)")
    print("="*60)
    
    # Test with invalid JSON
    result = electrical_check("{invalid json", ["voltage"], "summary")
    print(result)
    print()

def main():
    """Run all tests."""
    print("ELECTRICAL CHECK TOOL TESTING")
    print("="*60)
    print("Testing the electrical_check tool with various scenarios")
    print("="*60)
    print()
    
    try:
        test_summary_format()
        test_detailed_format()
        test_specific_checks()
        test_json_format()
        test_good_system()
        test_voltage_only()
        test_string_input()
        test_error_handling()
        
        print("="*60)
        print("ALL TESTS COMPLETED")
        print("="*60)
        print()
        print("Key observations:")
        print("• The tool correctly identifies voltage violations")
        print("• Multiple output formats work (summary, detailed, json)")
        print("• Specific check types can be selected")
        print("• Both dictionary and JSON string inputs are supported")
        print("• Error handling works for invalid inputs")
        
    except Exception as e:
        print(f"Test execution failed: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    main() 