import yaml
import os
from typing import Dict, Any


def load_config(config_path: str) -> Dict[str, Any]:
    """
    Load configuration from YAML file.
    
    Args:
        config_path (str): Path to the YAML configuration file
        
    Returns:
        Dict[str, Any]: Parsed configuration dictionary
    """
    if not os.path.exists(config_path):
        raise FileNotFoundError(f"Configuration file not found: {config_path}")
    
    with open(config_path, 'r', encoding='utf-8') as file:
        config = yaml.safe_load(file)
    
    return config


def update_config_with_resume_path(config: Dict[str, Any], resume_path: str) -> Dict[str, Any]:
    """
    Update configuration with resume checkpoint path.
    
    Args:
        config (Dict[str, Any]): Configuration dictionary
        resume_path (str): Path to resume training from
        
    Returns:
        Dict[str, Any]: Updated configuration
    """
    config['resume']['checkpoint_path'] = resume_path
    config['resume']['resume_from_checkpoint'] = True
    
    # Update run name to indicate resume
    if 'project' in config and 'wandb_project' in config['project']:
        run_name_suffix = os.path.basename(resume_path)
        config['training']['run_name_suffix'] = f"resume-{run_name_suffix}"
    
    return config


def get_run_name(config: Dict[str, Any]) -> str:
    """
    Generate run name from configuration.
    
    Args:
        config (Dict[str, Any]): Configuration dictionary
        
    Returns:
        str: Generated run name
    """
    seed = config['training']['seed']
    ssh_alias = os.environ.get("SSH_ALIAS", "")
    
    base_name = f"grpo-seed{seed}-{ssh_alias}"
    
    # Add suffix if resuming
    if 'run_name_suffix' in config['training']:
        base_name += f"-{config['training']['run_name_suffix']}"
    
    return base_name 