import logging
import pathlib
from configparser import _UNSET, DEFAULTSECT, ConfigParser

# from abr_myant.paths import rc_dirs

logger = logging.getLogger(__name__)

# The default RC settings. Access with
#   abr_myant.RC_DEFAULTS[section_name][option_name]
RC_DEFAULTS = {}

# The RC files in the order in which they will be read.
RC_FILES = [
    # rc_dirs["user"],
    # rc_dirs["project"],
]


class _RC(ConfigParser):  # pylint: disable=too-many-ancestors
    """Allows reading from and writing to RC settings.

    This object is a :class:`configparser.ConfigParser`, which means that
    values can be accessed and manipulated like a dictionary:

    .. testcode::

       oldsize = abr_myant.rc["decoder_cache"]["size"]
       abr_myant.rc["decoder_cache"]["size"] = "2 GB"

    All values are stored as strings. If you want to store or retrieve a
    specific datatype, you should coerce it appropriately (e.g., with ``int()``).
    Booleans are more flexible, so you should use the ``getboolean`` method
    to access boolean values.

    .. testcode::

       simple = abr_myant.rc["exceptions"].getboolean("simplified")

    In addition to the normal :class:`configparser.ConfigParser` methods,
    this object also has a ``reload_rc`` method to reset ``abr_myant.rc``
    to default settings:

    .. testcode::

       abr_myant.rc.reload_rc()  # Reads defaults from configuration files
       abr_myant.rc.reload_rc(filenames=[])  # Ignores configuration files

    """

    def __init__(self):
        super().__init__()
        self.reload_rc()

    # @property
    # def float_dtype(self):
    #     bits = self.get("precision", "bits")
    #     return np.dtype(f"float{bits}")

    # @property
    # def int_dtype(self):
    #     bits = self.get("precision", "bits")
    #     return np.dtype(f"int{bits}")

    def _clear(self):
        self.remove_section(DEFAULTSECT)
        for s in self.sections():
            self.remove_section(s)

    def _init_defaults(self):
        for section, settings in RC_DEFAULTS.items():
            self.add_section(section)
            for k, v in settings.items():
                self.set(section, k, str(v))

    def getpath(self, section, option, *, raw=False, vars=None, fallback=_UNSET):
        return pathlib.Path(
            self.get(section, option, raw=raw, vars=vars, fallback=fallback)
        )

    def read_file(self, fp, filename=None):
        if filename is None:
            filename = fp.name if hasattr(fp, "name") else "<???>"

        logger.debug("Reading configuration from %s", filename)
        return super().read_file(fp, filename)

    def read(self, filenames):
        logger.debug("Reading configuration files %s", filenames)
        return super().read(filenames)

    def reload_rc(self, filenames=None):
        """Resets the currently loaded RC settings and loads new RC files.

        Parameters
        ----------
        filenames: iterable object
            Filenames of RC files to load.
        """
        if filenames is None:
            filenames = RC_FILES

        self._clear()
        self._init_defaults()
        self.read(filenames)


rc = _RC()
