import torch
import torch.nn as nn
import torch.nn.functional as F
from torch_geometric.nn import GCNConv, MessagePassing
from torch_geometric.nn.conv.gcn_conv import gcn_norm
from torch_sparse import SparseTensor
import numpy as np
from deeprobust.graph import utils

class ada_prop(MessagePassing):
    def __init__(self, P, coe, bias=True, **kwargs):
        super(ada_prop, self).__init__(aggr='add', **kwargs)
        self.P = P
        self.coe = coe

        coes = coe*(1-coe)**np.arange(P+1)
        coes[-1] = (1-coe)**P

        self.coes = nn.Parameter(torch.tensor(coes))

    def reset_parameters(self):
        nn.init.zeros_(self.coes)
        for p in range(self.P+1):
            self.coes.data[p] = self.coe*(1-self.coe)**p
        self.coes.data[-1] = (1-self.coe)**self.P

    def forward(self, x, edge_index, edge_weight=None):
        if isinstance(edge_index, torch.Tensor):
            edge_index, norm = gcn_norm(
                edge_index, edge_weight, num_nodes=x.size(0), dtype=x.dtype)
        elif isinstance(edge_index, SparseTensor):
            edge_index = gcn_norm(
                edge_index, edge_weight, num_nodes=x.size(0), dtype=x.dtype)
            norm = None

        hidden = x*(self.coes[0])
        for p in range(self.P):
            if norm is None:
                x = edge_index @ x
            else:
                x = self.propagate(edge_index, x=x, norm=norm)
            c = self.coes[p+1]
            hidden = hidden + c*x
        return hidden

    def message(self, x_j, norm):
        if norm is not None:
            return norm.view(-1, 1) * x_j
        else:
            return x_j

class GPRGNN(nn.Module):
    def __init__(self, in_channels, hidden_channels, out_channels, dropout=.5, coe=.5, P=10):
        super(GPRGNN, self).__init__()
        self.lin1 = nn.Linear(in_channels, hidden_channels)
        self.lin2 = nn.Linear(hidden_channels, out_channels)
        self.prop = ada_prop(P, coe)

        self.dropout = dropout

    def reset_parameters(self):
        self.lin1.reset_parameters()
        self.lin2.reset_parameters()
        self.prop.reset_parameters()

    def forward(self, x, edge_index):
        x = F.dropout(x, p=self.dropout, training=self.training)
        x = F.relu(self.lin1(x))
        x = F.dropout(x, p=self.dropout, training=self.training)
        x = self.lin2(x)


        x = F.dropout(x, p=self.dropout, training=self.training)
        x = self.prop(x, edge_index)
        return F.log_softmax(x, dim=1)
    
    def predict(self, features=None, adj=None):
        self.eval()
        if features is None and adj is None:
            return self.forward(self.features, self.adj_norm)
        else:
            if type(adj) is not torch.Tensor:
                features, adj = utils.to_tensor(features, adj, device=self.device)

            self.features = features
            if utils.is_sparse_tensor(adj):
                self.adj_norm = utils.normalize_adj_tensor(adj, sparse=True)
            else:
                self.adj_norm = utils.normalize_adj_tensor(adj)
            edge_index = SparseTensor.from_scipy(self.adj_norm).float().to(self.device)
            
            return self.forward(self.features, edge_index)

