import torch
import numpy as np
import torch.nn.functional as F
import torch.optim as optim
from deeprobust.graph.defense import GCN
from deeprobust.graph.targeted_attack import Nettack
from deeprobust.graph.utils import *
from deeprobust.graph.data import Dataset, Dpr2Pyg
import argparse
from tqdm import tqdm
import warnings
import copy
from deeprobust.graph import utils
from ogb.nodeproppred import NodePropPredDataset
from my_utils.utils import spade,hnsw,construct_adj, spectral_embedding_eig,SPF,construct_weighted_adj,spade_nonetworkx,rank_samples_by_variance
from scipy.sparse import load_npz
from torch_sparse import SparseTensor
from scipy.sparse import csr_matrix
import pickle

# Suppress all warnings
warnings.filterwarnings("ignore")

def normal_adj(adj):
    adj = SparseTensor.from_scipy(adj)
    deg = adj.sum(dim=1).to(torch.float)
    D_isqrt = deg.pow(-0.5)
    D_isqrt[D_isqrt == float('inf')] = 0
    DAD = D_isqrt.view(-1,1) * adj * D_isqrt.view(1,-1)

    return DAD.to_scipy(layout='csr')

parser = argparse.ArgumentParser()
parser.add_argument('--seed', type=int, default=15, help='Random seed.')
parser.add_argument('--dataset', type=str, default='citeseer', choices=['cora', 'cora_ml', 'citeseer', 'polblogs', 'pubmed','arxiv','products','chameleon', 'squirrel'], help='dataset')


args = parser.parse_args()
args.cuda = torch.cuda.is_available()
print('cuda: %s' % args.cuda)
device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")

np.random.seed(args.seed)
torch.manual_seed(args.seed)
if args.cuda:
    torch.cuda.manual_seed(args.seed)

if args.dataset in ['cora', 'cora_ml', 'citeseer', 'polblogs', 'pubmed']:
    data = Dataset(root='./tmp/', name=args.dataset)
    adj, features, labels = data.adj, data.features, data.labels
    orig_adj, orig_features = copy.copy(adj), copy.copy(features.todense())
    idx_train, idx_val, idx_test = data.idx_train, data.idx_val, data.idx_test

elif args.dataset in ['chameleon', 'squirrel']:
    
    with open(f'./tmp/{args.dataset}_data.pickle', 'rb') as handle:
            data = pickle.load(handle)
    features = data["features"]
    labels = data["labels"]
    idx_train = data["idx_train"].astype(np.int32)
    idx_val = data["idx_val"].astype(np.int32)
    idx_test = data["idx_test"].astype(np.int32)
    adj = load_npz(f'./tmp/{args.dataset}.npz')

    orig_adj, orig_features = copy.copy(adj), copy.copy(features)
    features = csr_matrix(features)


else:
    ogbn_dataset = NodePropPredDataset(name=f'ogbn-{args.dataset}', root='../../../ogbn/ogbn_data/')
    ogbn_data = ogbn_dataset[0]
    split_idx = ogbn_dataset.get_idx_split()
    idx_train = split_idx['train'].astype(np.int32)
    idx_val = split_idx['valid'].astype(np.int32)
    idx_test = split_idx['test'].astype(np.int32)
    features = ogbn_data[0]['node_feat']
    labels = ogbn_data[1].reshape(-1,)

    adj = load_npz(f'../../../ogbn/data/{args.dataset}/{args.dataset}_clean.npz')

    if args.dataset == "products":
        adj = normal_adj(adj)
        args.epochs = 300

    orig_adj, orig_features = copy.copy(adj), copy.copy(features)
    features = csr_matrix(features)

idx_unlabeled = np.union1d(idx_val, idx_test)

# Setup Surrogate model
surrogate = GCN(nfeat=features.shape[1], nclass=labels.max().item()+1, nhid=16, dropout=0, with_relu=False, with_bias=False, device=device)

surrogate = surrogate.to(device)
surrogate.fit(features, adj, labels, idx_train, idx_val, patience=30)

# Setup Attack Model
#target_node = 0
#assert target_node in idx_unlabeled

model = Nettack(surrogate, nnodes=adj.shape[0], attack_structure=True, attack_features=True, device=device)
model = model.to(device)

def main():
    degrees = adj.sum(0).A1
    # How many perturbations to perform. Default: Degree of the node
    n_perturbations = int(degrees[target_node])

    # direct attack
    model.attack(features, adj, labels, target_node, n_perturbations)
    # # indirect attack/ influencer attack
    # model.attack(features, adj, labels, target_node, n_perturbations, direct=False, n_influencers=5)
    modified_adj = model.modified_adj
    modified_features = model.modified_features
    print(model.structure_perturbations)
    print('=== testing GCN on original(clean) graph ===')
    test(adj, features, target_node)
    print('=== testing GCN on perturbed graph ===')
    test(modified_adj, modified_features, target_node)

def test(adj, features, target_node):
    ''' test on GCN '''
    gcn = GCN(nfeat=features.shape[1],
              nhid=16,
              nclass=labels.max().item() + 1,
              dropout=0.5, device=device)

    gcn = gcn.to(device)

    gcn.fit(features, adj, labels, idx_train, idx_val, patience=30)

    gcn.eval()
    output = gcn.predict()
    probs = torch.exp(output[[target_node]])[0]
    print('Target node probs: {}'.format(probs.detach().cpu().numpy()))
    acc_test = accuracy(output[idx_test], labels[idx_test])

    print("Overall test set results:",
          "accuracy= {:.4f}".format(acc_test.item()))

    return acc_test.item()

def select_nodes(target_gcn=None):
    '''
    selecting nodes as reported in nettack paper:
    (i) the 10 nodes with highest margin of classification, i.e. they are clearly correctly classified,
    (ii) the 10 nodes with lowest margin (but still correctly classified) and
    (iii) 20 more nodes randomly
    '''

    if target_gcn is None:
        target_gcn = GCN(nfeat=features.shape[1],
                  nhid=16,
                  nclass=labels.max().item() + 1,
                  dropout=0.5, device=device)
        target_gcn = target_gcn.to(device)
        target_gcn.fit(features, adj, labels, idx_train, idx_val, patience=30)
    target_gcn.eval()
    output = target_gcn.predict()

    margin_dict = {}
    for idx in idx_test:
        margin = classification_margin(output[idx], labels[idx])
        if margin < 0: # only keep the nodes correctly classified
            continue
        margin_dict[idx] = margin
    sorted_margins = sorted(margin_dict.items(), key=lambda x:x[1], reverse=True)
    high = [x for x, y in sorted_margins[: 10]]#10
    low = [x for x, y in sorted_margins[-10: ]]#-10
    other = [x for x, y in sorted_margins[10: -10]]#10: -10
    other = np.random.choice(other, 20, replace=False).tolist()

    return high + low + other


def select_low_nodes(target_gcn):

    target_gcn.eval()
    output = target_gcn.predict()
    margin_dict = {}
    for idx in idx_test:
        margin = classification_margin(output[idx], labels[idx])
        if margin < 0: # only keep the nodes correctly classified
            continue
        margin_dict[idx] = margin
    sorted_margins = sorted(margin_dict.items(), key=lambda x:x[1], reverse=True)
    low = [x for x, y in sorted_margins[-40: ]]

    return low

def multi_test_poison():
    # test on 40 nodes on poisoining attack
    cnt = 0
    degrees = adj.sum(0).A1
    node_list = select_nodes()
    num = len(node_list)
    print('=== [Poisoning] Attacking %s nodes respectively ===' % num)
    for target_node in tqdm(node_list):
        n_perturbations = int(degrees[target_node])
        model = Nettack(surrogate, nnodes=adj.shape[0], attack_structure=True, attack_features=True, device=device)
        model = model.to(device)
        model.attack(features, adj, labels, target_node, n_perturbations, verbose=False)
        modified_adj = model.modified_adj
        modified_features = model.modified_features
        acc = single_test(modified_adj, modified_features, target_node)
        if acc == 0:
            cnt += 1
    print('misclassification rate : %s' % (cnt/num))

def single_test(adj, features, target_node, gcn=None):
    if gcn is None:
        # test on GCN (poisoning attack)
        gcn = GCN(nfeat=features.shape[1],
                  nhid=16,
                  nclass=labels.max().item() + 1,
                  dropout=0.5, device=device)

        gcn = gcn.to(device)

        gcn.fit(features, adj, labels, idx_train, idx_val, patience=30)
        gcn.eval()
        output = gcn.predict()
    else:
        # test on GCN (evasion attack)
        gcn.eval()
        output = gcn.predict(features, adj)
    probs = torch.exp(output[[target_node]])

    # acc_test = accuracy(output[[target_node]], labels[target_node])
    acc_test = (output.argmax(1)[target_node] == labels[target_node])
    return acc_test.item()

def multi_test_evasion():
    # test on 40 nodes on evasion attack
    target_gcn = GCN(nfeat=features.shape[1],
              nhid=16,
              nclass=labels.max().item() + 1,
              dropout=0.5, device=device)

    target_gcn = target_gcn.to(device)

    target_gcn.fit(features, adj, labels, idx_train, idx_val, patience=30)
    
    SPADE_features, SPADE_adj = utils.to_tensor(features, adj, device=device)
    features1, adj1 = utils.to_tensor(features, adj, device=device)
    the_k = 50
    spec_embed = spectral_embedding_eig(orig_adj,orig_features,use_feature=True,adj_norm=False)#spectral_embedding_eig,spectral_embedding
    neighs, distance = hnsw(spec_embed, k=the_k)
    embed_adj_mtx,_,_ = construct_weighted_adj(neighs, distance)#construct_weighted_adj,construct_adj
    embed_adj_mtx = SPF(embed_adj_mtx, 4)
    SPADE_features, embed_adj_mtx1 = utils.to_tensor(features, embed_adj_mtx, device=device)
    embed_out =  target_gcn(SPADE_features,embed_adj_mtx1)
    orig_out =  target_gcn(features1,adj1)
    TopEig, _, TopNodeList, _, L_in, L_out = spade(embed_adj_mtx, embed_out.cpu().detach().numpy(), k=the_k)#spade

    variance_rank = rank_samples_by_variance(orig_out.cpu().detach().numpy())


    misclassified_ids=[]
    for idx in TopNodeList:
        margin = classification_margin(orig_out[idx], labels[idx])
        if margin < 0: 
            misclassified_ids.append(margin) 
    filtered_TopNodeList = [idx for idx in TopNodeList if idx not in misclassified_ids]
    spade_selected_node = filtered_TopNodeList[:40]

    _, predicted_labels = torch.max(orig_out, 1)
    misclassified_ids = [idx for idx in variance_rank if predicted_labels[idx] != labels[idx]]
    filtered_TopNodeList = [idx for idx in variance_rank if idx not in misclassified_ids]
    variance_selected_node = filtered_TopNodeList[:40]

    cnt = 0
    orig_cnt = 0
    degrees = adj.sum(0).A1
    node_list = select_nodes(target_gcn)
    #low_node_list = select_low_nodes(target_gcn)
    num = len(node_list)

    target_gcn.eval()
    output = target_gcn.predict()
    #probs = torch.exp(output[[node_list]])[0]
    #spade_probs = torch.exp(output[[spade_selected_node.copy()]])[0]
    #print('Target node probs: {}'.format(probs.detach().cpu().numpy()))
    #print('SPADE target node probs: {}'.format(spade_probs.detach().cpu().numpy()))
    
    '''''
    print('=== [Evasion] Attacking %s nodes respectively ===' % num)
    for target_node in tqdm(node_list):
        n_perturbations = int(degrees[target_node])
        model = Nettack(surrogate, nnodes=adj.shape[0], attack_structure=True, attack_features=True, device=device)
        model = model.to(device)
        model.attack(features, adj, labels, target_node, n_perturbations, verbose=False)
        modified_adj = model.modified_adj
        modified_features = model.modified_features
        acc = single_test(modified_adj, modified_features, target_node, gcn=target_gcn)
        orig_acc = single_test(orig_adj, orig_features, target_node, gcn=target_gcn)
        if acc == 0:
            cnt += 1

        if orig_acc == 0:
            orig_cnt += 1

    print('orig:  orig mis rate : {}, perturbed mis rate : {}'.format(orig_cnt/num,cnt/num))
    '''''
    cnt = 0
    orig_cnt = 0

    print('=== [Evasion] Attacking %s nodes respectively ===' % num)
    for target_node in tqdm(variance_selected_node.copy()):
        n_perturbations = int(degrees[target_node])
        model = Nettack(surrogate, nnodes=adj.shape[0], attack_structure=True, attack_features=True, device=device)
        model = model.to(device)
        model.attack(features, adj, labels, target_node, n_perturbations, verbose=False)
        modified_adj = model.modified_adj
        modified_features = model.modified_features
        acc = single_test(modified_adj, modified_features, target_node, gcn=target_gcn)
        orig_acc = single_test(orig_adj, orig_features, target_node, gcn=target_gcn)
        if acc == 0:
            cnt += 1

        if orig_acc == 0:
            orig_cnt += 1

    print('SAGMAN embedded mis rate : {}, perturbed mis rate : {}'.format(orig_cnt/num,cnt/num))

    cnt = 0
    orig_cnt = 0

    degrees = adj.sum(0).A1

    print('=== [Evasion] Attacking %s nodes respectively ===' % num)
    for target_node in tqdm(spade_selected_node.copy()):
        n_perturbations = int(degrees[target_node])
        model = Nettack(surrogate, nnodes=adj.shape[0], attack_structure=True, attack_features=True, device=device)
        model = model.to(device)
        model.attack(features, adj, labels, target_node, n_perturbations, verbose=False)
        modified_adj = model.modified_adj
        modified_features = model.modified_features
        acc = single_test(modified_adj, modified_features, target_node, gcn=target_gcn)
        orig_acc = single_test(orig_adj, orig_features, target_node, gcn=target_gcn)
 
        if orig_acc == 0:
            orig_cnt += 1

        if acc == 0:
            cnt += 1

    print('SAGMAN orig mis rate : {}, perturbed mis rate : {}'.format(orig_cnt/num,cnt/num))




if __name__ == '__main__':
    #main()
    #multi_test_poison()
    multi_test_evasion()


