from maxcut import H_graph, P_coloring
from collections import defaultdict
import random

eps = 1e-9

def reduce_maxcut_to_betweenness(vertices, edges, coloring):
    points = list(vertices) + ['x', 'y']
    constraints = []

    M = sum(w for _, _, w in edges) + 1

    # Type 1
    for v in vertices:
        constraints.append(('x', 'y', v, M))
        constraints.append(('y', 'x', v, M))

    # Type 2
    for u, v, w in edges:
        constraints.append((u, 'x', v, w))

    init = {}
    pos = set()

    def allocate(v):
        if v == 'x':
            return 0
        elif v == 'y':
            return 1
        elif coloring[v] == 0:
            while True:
                p = random.uniform(-1, -eps)
                if p not in pos:
                    pos.add(p)
                    return p
        else:
            while True:
                p = random.uniform(1 + eps, 2)
                if p not in pos:
                    pos.add(p)
                    return p

    for v in points:
        init[v] = allocate(v)

    return points, constraints, init


def local_search_betweenness(points, constraints, init):
    constraints_by_vertex = defaultdict(list)
    for a, b, c, w in constraints:
        constraints_by_vertex[a].append((a, b, c, w))
        constraints_by_vertex[b].append((a, b, c, w))
        constraints_by_vertex[c].append((a, b, c, w))

    solution = init
    pos = set(solution.values())

    def allocate(l, r):
        while True:
            p = random.uniform(l, r)
            if p not in pos:
                return p

    def local_contribution(v, new_pos):
        contribution = 0
        original_pos = solution[v]
        for a, b, c, w in constraints_by_vertex[v]:
            pa, pb, pc = solution[a], solution[b], solution[c]
            if pa < pb < pc or pc < pb < pa:
                contribution -= w
        solution[v] = new_pos
        for a, b, c, w in constraints_by_vertex[v]:
            pa, pb, pc = solution[a], solution[b], solution[c]
            if pa < pb < pc or pc < pb < pa:
                contribution += w
        solution[v] = original_pos
        return contribution

    def neighborhood_oracle():
        for v in points:
            if v in ['x', 'y']:
                continue
            new_pos = allocate(1 + eps, 2) if solution[v] < 0 else allocate(-1, -eps)
            if local_contribution(v, new_pos) > 0:
                return v, new_pos
        return None

    move_count = 0

    while True:
        move = neighborhood_oracle()
        if move is None:
            break
        v, new_pos = move
        solution[v] = new_pos
        move_count += 1

    return len(points), len(constraints), move_count


def main():
    for i in range(1, 16):
        edges = H_graph(i)
        vertices = set()
        for u, v, _ in edges:
            vertices.add(u)
            vertices.add(v)

        coloring = {v: P_coloring(v) for v in vertices}
        points, constraints, init = reduce_maxcut_to_betweenness(vertices, edges, coloring)
        num_vertices, num_constraints, move_count = local_search_betweenness(points, constraints, init)
        print(f"n={i}: {num_vertices} vertices, {num_constraints} constraints, {move_count} moves")


if __name__ == "__main__":
    main()