from maxcut import H_graph, P_coloring
from collections import defaultdict
import random

eps = 1e-9

def reduce_maxcut_to_contrastive(vertices, edges, coloring):
    points = list(vertices) + ['x', 'y', 'z']
    constraints = []

    M = sum(w for _, _, w in edges) + 1
    MM = (2 * len(vertices) + 1) * M

    # Type A
    for u, v, w in edges:
        constraints.append((u, 'x', v, w))

    # Type B
    for v in vertices:
        constraints.append(('x', 'y', v, M))
        constraints.append(('x', v, 'z', M))
    constraints.append(('y', 'z', 'x', MM))
    constraints.append(('x', 'y', 'z', MM))

    init = {}
    pos = set()

    def allocate(v):
        if v == 'x':
            return 0
        elif v == 'y':
            return 1
        elif v == 'z':
            return 1.5
        elif coloring[v] == 0:
            while True:
                p = random.uniform(-1.5 + eps, -1 - eps)
                if p not in pos:
                    pos.add(p)
                    return p
        else:
            while True:
                p = random.uniform(1 + eps, 1.5 - eps)
                if p not in pos:
                    pos.add(p)
                    return p

    for v in points:
        init[v] = allocate(v)

    return points, constraints, init


def local_search_contrastive(points, constraints, init):
    constraints_by_vertex = defaultdict(list)
    for a, b, c, w in constraints:
        constraints_by_vertex[a].append((a, b, c, w))
        constraints_by_vertex[b].append((a, b, c, w))
        constraints_by_vertex[c].append((a, b, c, w))

    solution = init
    pos = set(solution.values())

    def allocate(l, r):
        while True:
            p = random.uniform(l, r)
            if p not in pos:
                return p

    def local_contribution(v, new_pos):
        contribution = 0
        original_pos = solution[v]
        for a, b, c, w in constraints_by_vertex[v]:
            pa, pb, pc = solution[a], solution[b], solution[c]
            if abs(pa - pb) <= abs(pa - pc):
                contribution -= w
        solution[v] = new_pos
        for a, b, c, w in constraints_by_vertex[v]:
            pa, pb, pc = solution[a], solution[b], solution[c]
            if abs(pa - pb) <= abs(pa - pc):
                contribution += w
        solution[v] = original_pos
        return contribution

    def neighborhood_oracle():
        for v in points:
            if v in ['x', 'y']:
                continue
            new_pos = allocate(1 + eps, 1.5 - eps) if solution[v] < 0 else allocate(-1.5 + eps, -1 - eps)
            if local_contribution(v, new_pos) > 0:
                return v, new_pos
        return None

    move_count = 0

    while True:
        move = neighborhood_oracle()
        if move is None:
            break
        v, new_pos = move
        solution[v] = new_pos
        move_count += 1

    return len(points), len(constraints), move_count


def main():
    for i in range(1, 16):
        edges = H_graph(i)
        vertices = set()
        for u, v, _ in edges:
            vertices.add(u)
            vertices.add(v)

        coloring = {v: P_coloring(v) for v in vertices}
        points, constraints, init = reduce_maxcut_to_contrastive(vertices, edges, coloring)
        num_vertices, num_constraints, move_count = local_search_contrastive(points, constraints, init)
        print(f"n={i}: {num_vertices} vertices, {num_constraints} constraints, {move_count} moves")


if __name__ == "__main__":
    main()
