from typing import Any
import pandas as pd
from sklearn.metrics import f1_score

from mledojo.metrics.base import CompetitionMetrics, InvalidSubmissionError


class BirdSpeciesAudioClassificationMetrics(CompetitionMetrics):
    """
    Metric class for Bird Species Audio Identification using macro F1 score.

    Expected CSV schemas:
    - Ground truth (test_answer.csv): [id, label]
    - Submission (sample_submission.csv or user submission): [id, label]
    """

    def __init__(self, value: str = "label", higher_is_better: bool = True):
        super().__init__(higher_is_better)
        self.value = value

    def evaluate(self, y_true: pd.DataFrame, y_pred: pd.DataFrame) -> float:
        # Align by the first column (id) then compute macro F1 over the label column
        if not isinstance(y_true, pd.DataFrame) or not isinstance(y_pred, pd.DataFrame):
            raise InvalidSubmissionError("Both y_true and y_pred must be pandas DataFrames.")
        if y_true.shape[0] != y_pred.shape[0]:
            raise InvalidSubmissionError(
                f"Row count mismatch between y_true ({y_true.shape[0]}) and y_pred ({y_pred.shape[0]})."
            )

        # Sort by first column to align rows
        id_col_true = y_true.columns[0]
        id_col_pred = y_pred.columns[0]
        y_true_sorted = y_true.sort_values(by=id_col_true).reset_index(drop=True)
        y_pred_sorted = y_pred.sort_values(by=id_col_pred).reset_index(drop=True)

        # Validate that IDs match exactly
        if (y_true_sorted[id_col_true].values != y_pred_sorted[id_col_pred].values).any():
            raise InvalidSubmissionError("ID mismatch between y_true and y_pred after sorting by the first column.")

        # Compute macro F1 on label column
        return float(
            f1_score(
                y_true_sorted[self.value].astype(str).values,
                y_pred_sorted[self.value].astype(str).values,
                average="macro",
            )
        )

    def validate_submission(self, submission: Any, ground_truth: Any) -> str:
        # This validation follows the behavior in samples/sample_metric.py
        if not isinstance(submission, pd.DataFrame):
            raise InvalidSubmissionError(
                "Submission must be a pandas DataFrame. Please provide a valid pandas DataFrame."
            )
        if not isinstance(ground_truth, pd.DataFrame):
            raise InvalidSubmissionError(
                "Ground truth must be a pandas DataFrame. Please provide a valid pandas DataFrame."
            )

        if len(submission) != len(ground_truth):
            raise InvalidSubmissionError(
                f"Number of rows in submission ({len(submission)}) does not match ground truth ({len(ground_truth)}). Please ensure both have the same number of rows."
            )

        # Sort both by their first column to compare ids
        submission_sorted = submission.sort_values(by=submission.columns[0]).reset_index(drop=True)
        ground_truth_sorted = ground_truth.sort_values(by=ground_truth.columns[0]).reset_index(drop=True)

        if (
            submission_sorted[submission_sorted.columns[0]].values
            != ground_truth_sorted[ground_truth_sorted.columns[0]].values
        ).any():
            raise InvalidSubmissionError(
                "First column values (ids) do not match between submission and ground truth. Please ensure the first column values are identical."
            )

        sub_cols = set(submission.columns)
        true_cols = set(ground_truth.columns)
        missing_cols = true_cols - sub_cols
        extra_cols = sub_cols - true_cols

        if missing_cols:
            raise InvalidSubmissionError(
                f"Missing required columns in submission: {', '.join(sorted(missing_cols))}."
            )
        if extra_cols:
            raise InvalidSubmissionError(
                f"Extra unexpected columns found in submission: {', '.join(sorted(extra_cols))}."
            )

        return "Submission is valid."
