"""Base Agent Module.

Provides a base class for intelligent agents that use LLM reasoning
and function calling capabilities.
"""

import asyncio
import json
from typing import Any, Dict, Optional, List
from abc import ABC, abstractmethod

from utils.chat import ChatClient


class AgentBase(ABC):
    """Base class for intelligent agents using LLM and tools.
    
    This base class provides common functionality for agents that interact
    with LLMs and use various tools to accomplish tasks.
    
    Attributes:
        chat_client: LLM client for reasoning
        working_directory: Directory for all operations
        tools: List of tool schemas for LLM function calling
        tool_instances: Dictionary of initialized tool instances
    """
    
    def __init__(
        self,
        working_directory: Optional[str] = None,
        model: str = "gpt-4o-mini",
        chat_config: Optional[dict] = None
    ) -> None:
        """Initialize the base agent.
        
        Args:
            working_directory: Working directory path
            model: LLM model identifier
        """
        self.chat_client = ChatClient(model=model, **chat_config)
        self.working_directory = working_directory
        self.tools = []
        self.tool_instances = {}
        self._initialize_tools()
    
    @abstractmethod
    def _initialize_tools(self) -> None:
        """Initialize tool instances and schemas for LLM function calling.
        
        Subclasses should implement this method to set up:
        - self.tool_instances: Dictionary mapping tool names to tool instances
        - self.tools: List of tool schemas for LLM function calling
        """
        pass
    
    async def _call_tool(
        self,
        tool_name: str,
        arguments: Dict[str, Any]
    ) -> Dict[str, Any]:
        """Execute a tool with the specified arguments.
        
        Args:
            tool_name: Name of the tool to execute
            arguments: Arguments for the tool
            
        Returns:
            Tool execution result or error dictionary
        """
        try:
            tool = self.tool_instances.get(tool_name)
            if tool is None:
                return {"error": f"Unknown tool: {tool_name}"}
            
            return tool.execute(**arguments)
        except Exception as e:
            return {"error": f"Tool execution failed: {str(e)}"}
    
    @abstractmethod
    def _get_initial_prompt(self) -> str:
        """Get the initial prompt for the agent.
        
        Returns:
            Initial prompt string
        """
        pass
    
    async def execute_task(
        self,
        max_iterations: int = 5
    ) -> Dict[str, Any]:
        """Execute agent task using LLM and tools.
        
        Args:
            max_iterations: Maximum number of LLM-tool interaction cycles
            
        Returns:
            Dictionary containing:
                - success: Whether task completed successfully
                - conversation_history: Full conversation log
                - iterations_used: Number of iterations consumed
                - final_response: Last agent response
                - total_cost: Total cost of LLM API calls
        """
        conversation_history = []
        
        initial_prompt = self._get_initial_prompt()
        
        conversation_history.append({
            "role": "user",
            "content": initial_prompt
        })
        
        for iteration in range(max_iterations):
            try:
                response = await self._get_llm_response(
                    conversation_history,
                    iteration
                )
                
                if response is None:
                    break
                    
                assistant_message = self._process_response(response)
                conversation_history.append(assistant_message)
                
                if not response.tool_calls:
                    break
                
                tool_result = await self._execute_tool_call(
                    response.tool_calls[0],
                    iteration
                )
                conversation_history.append(tool_result)
                
                if iteration == max_iterations - 1:
                    final_message = await self._get_final_response(
                        conversation_history
                    )
                    conversation_history.append(final_message)
                    
            except Exception as e:
                error_message = self._create_error_message(str(e))
                conversation_history.append(error_message)
                break
        
        return {
            "success": True,
            "conversation_history": conversation_history,
            "iterations_used": iteration + 1,
            "final_response": conversation_history[-1] if conversation_history else None,
            "total_cost": getattr(self.chat_client, 'total_cost', 0.0)
        }
    
    async def _get_llm_response(
        self,
        conversation_history: List[Dict[str, Any]],
        iteration: int
    ) -> Optional[Any]:
        """Get response from the LLM.
        
        Args:
            conversation_history: Current conversation history
            iteration: Current iteration number
            
        Returns:
            LLM response or None if string response
        """
        response = await self.chat_client.chat_async(
            messages=conversation_history,
            tools=self.tools,
            tool_choice="auto"
        )
        
        print(f"Iteration {iteration}: {response}")
        
        if isinstance(response, str):
            conversation_history.append({
                "role": "assistant",
                "content": response
            })
            return None
            
        return response
    
    def _process_response(self, response: Any) -> Dict[str, Any]:
        """Process LLM response into assistant message.
        
        Args:
            response: LLM response object
            
        Returns:
            Formatted assistant message
        """
        assistant_message = {
            "role": "assistant",
            "content": response.content or ""
        }
        
        if response.tool_calls:
            assistant_message["tool_calls"] = [
                {
                    "id": tool_call.id,
                    "type": "function",
                    "function": {
                        "name": tool_call.function.name,
                        "arguments": tool_call.function.arguments
                    }
                }
                for tool_call in response.tool_calls
            ]
        
        return assistant_message
    
    async def _execute_tool_call(
        self,
        tool_call: Any,
        iteration: int
    ) -> Dict[str, Any]:
        """Execute a tool call and format the result.
        
        Args:
            tool_call: Tool call from LLM
            iteration: Current iteration number
            
        Returns:
            Formatted tool result message
        """
        tool_name = tool_call.function.name
        arguments = json.loads(tool_call.function.arguments)
        
        tool_result = await self._call_tool(tool_name, arguments)
        
        function_message = {
            "role": "tool",
            "tool_call_id": tool_call.id,
            "content": json.dumps(tool_result, ensure_ascii=False, indent=2)
        }
        
        print(f"Iteration {iteration}: Tool call '{tool_name}' with arguments {arguments}")
        print(f"Iteration {iteration}: Tool result: {tool_result}")
        return function_message
    
    async def _get_final_response(
        self,
        conversation_history: List[Dict[str, Any]]
    ) -> Dict[str, Any]:
        """Get final response from LLM.
        
        Args:
            conversation_history: Current conversation history
            
        Returns:
            Final assistant message
        """
        final_response = await self.chat_client.chat_async(
            messages=conversation_history
        )
        
        if isinstance(final_response, str):
            return {"role": "assistant", "content": final_response}
        
        return {
            "role": "assistant",
            "content": final_response.content or "Task completed"
        }
    
    def _create_error_message(self, error: str) -> Dict[str, Any]:
        """Create an error message.
        
        Args:
            error: Error description
            
        Returns:
            Formatted error message
        """
        return {
            "role": "assistant",
            "content": f"I encountered an error: {error}"
        }
