"""Executor Agent Module.

Provides an intelligent agent that generates Kaggle competition artifacts
using LLM reasoning and function calling capabilities.
"""

from typing import Optional

from components.agentbase import AgentBase
from prompt.prompts import initial_instruction_with_brainstorm
from tools.executetool import (
    ExecuteBashTool,
    ListDirectoryStructureTool,
    ReadFileTool,
    WriteFileTool,
)
from tools.workflowtool import WorkflowTools


class ExecutorAgent(AgentBase):
    """Agent for generating competition artifacts using LLM and tools.
    
    This agent orchestrates the creation of Kaggle competition materials
    including descriptions, metrics, data preparation scripts, and tests.
    
    Attributes:
        competition_name: Name of the competition
        workflow_tools: Tools for workflow operations
    """
    
    def __init__(
        self,
        competition_name: Optional[str] = None,
        working_directory: Optional[str] = None,
        model: str = "gpt-4o-mini",
        chat_config: Optional[dict] = None
    ) -> None:
        """Initialize the ExecutorAgent.
        
        Args:
            competition_name: Name of the competition to process
            working_directory: Working directory path
            model: LLM model identifier
        """
        self.competition_name = competition_name
        self.workflow_tools = WorkflowTools()
        super().__init__(working_directory=working_directory, model=model, chat_config=chat_config)
    
    def _initialize_tools(self) -> None:
        """Initialize tool instances and schemas for LLM function calling."""
        self.tool_instances = {
            'read_file': ReadFileTool(work_directory=self.working_directory),
            'write_file': WriteFileTool(work_directory=self.working_directory),
            'execute_bash': ExecuteBashTool(work_directory=self.working_directory),
            'list_directory_structure': ListDirectoryStructureTool(
                work_directory=self.working_directory
            )
        }
        
        self.tools = [
            {"type": "function", "function": tool.get_schema()}
            for tool in self.tool_instances.values()
        ]
    
    def _get_initial_prompt(self) -> str:
        """Get the initial prompt for the executor agent.
        
        Returns:
            Initial prompt string with competition information
        """
        competition_info = self.workflow_tools.get_detailed_competition_info(
            self.competition_name
        )
        
        return initial_instruction_with_brainstorm.format(
            working_directory=self.working_directory,
            dataset_information=competition_info
        )



