"""Refactor Agent Module.

Provides an intelligent agent that refactors Python code to meet
production standards using LLM reasoning and function calling.
"""

from typing import Optional

from components.agentbase import AgentBase
from prompt.prompts import refactor_instruction
from tools.executetool import (
    ExecuteBashTool,
    ListDirectoryStructureTool,
    ReadFileTool,
    WriteFileTool,
)


class RefactorAgent(AgentBase):
    """Agent for refactoring Python code to production standards.
    
    This agent takes generated competition code and refactors it to follow
    best practices, implement proper abstractions, and ensure code quality.
    """
    
    def __init__(
        self,
        working_directory: Optional[str] = None,
        model: str = "gpt-4o-mini",
        chat_config: Optional[dict] = None
    ) -> None:
        """Initialize the RefactorAgent.
        
        Args:
            working_directory: Working directory path
            model: LLM model identifier
        """
        super().__init__(working_directory=working_directory, model=model, chat_config=chat_config)
    
    def _initialize_tools(self) -> None:
        """Initialize tool instances and schemas for LLM function calling."""
        self.tool_instances = {
            'read_file': ReadFileTool(work_directory=self.working_directory),
            'write_file': WriteFileTool(work_directory=self.working_directory),
            'execute_bash': ExecuteBashTool(work_directory=self.working_directory),
            'list_directory_structure': ListDirectoryStructureTool(
                work_directory=self.working_directory
            )
        }
        
        self.tools = [
            {"type": "function", "function": tool.get_schema()}
            for tool in self.tool_instances.values()
        ]
    
    def _get_initial_prompt(self) -> str:
        """Get the initial prompt for the refactor agent.
        
        Returns:
            Initial prompt string for code refactoring
        """
        return refactor_instruction.format(
            working_directory=self.working_directory
        )




