"""Reviewer Agent Module.

Provides an intelligent agent that reviews and provides feedback on
competition artifacts using LLM reasoning and function calling.
"""

from typing import Optional

from components.agentbase import AgentBase
from prompt.prompts import feedback_instruction
from tools.executetool import (
    ExecuteBashTool,
    ListDirectoryStructureTool,
    ReadFileTool,
    WriteFileTool,
)
from tools.workflowtool import WorkflowTools


class ReviewerAgent(AgentBase):
    """Agent for reviewing competition artifacts and providing feedback.
    
    This agent analyzes generated competition materials and provides
    detailed feedback and suggestions for improvement.
    
    Attributes:
        competition_name: Name of the competition
        workflow_tools: Tools for workflow operations
    """
    
    def __init__(
        self,
        competition_name: Optional[str] = None,
        working_directory: Optional[str] = None,
        model: str = "gpt-4o-mini",
        chat_config: Optional[dict] = None
    ) -> None:
        """Initialize the ReviewerAgent.
        
        Args:
            competition_name: Name of the competition to review
            working_directory: Working directory path
            model: LLM model identifier
        """
        self.competition_name = competition_name
        self.workflow_tools = WorkflowTools()
        super().__init__(working_directory=working_directory, model=model, chat_config=chat_config)
    
    def _initialize_tools(self) -> None:
        """Initialize tool instances and schemas for LLM function calling."""
        self.tool_instances = {
            'read_file': ReadFileTool(work_directory=self.working_directory),
            'write_file': WriteFileTool(work_directory=self.working_directory),
            'execute_bash': ExecuteBashTool(work_directory=self.working_directory),
            'list_directory_structure': ListDirectoryStructureTool(
                work_directory=self.working_directory
            )
        }
        
        self.tools = [
            {"type": "function", "function": tool.get_schema()}
            for tool in self.tool_instances.values()
        ]
    
    def _get_initial_prompt(self) -> str:
        """Get the initial prompt for the reviewer agent.
        
        Returns:
            Initial prompt string with competition information
        """
        competition_info = self.workflow_tools.get_detailed_competition_info(
            self.competition_name
        )
        
        return feedback_instruction.format(
            working_directory=self.working_directory,
            max_iterations=5,  # Default value, could be made configurable
            dataset_information=competition_info
        )



