"""Competition Verifier Module.

Provides utilities to verify that competition directories contain
all required files for proper Kaggle competition structure.
"""

from pathlib import Path
from typing import Dict, List


class CompetitionVerifier:
    """Verifies competition directory structure and required files.
    
    Ensures that a competition directory contains all necessary files
    for a complete Kaggle competition setup.
    
    Attributes:
        REQUIRED_FILES: List of files required for valid competition
        directory: Path to the competition directory
    """
    
    REQUIRED_FILES = [
        'description.txt',
        'metric.py',
        'prepare.py',
        'test.py',
        'test_answer.csv',
        'sample_submission.csv'
    ]
    
    def __init__(self, directory: str) -> None:
        """Initialize verifier with target directory.
        
        Args:
            directory: Path to competition directory to verify
        """
        self.directory = Path(directory)
        
    def verify_files(self) -> Dict[str, bool]:
        """Verify existence of all required files.
        
        Returns:
            Dictionary mapping file names to their existence status
            
        Raises:
            ValueError: If directory doesn't exist or isn't a directory
        """
        self._validate_directory()
        
        return {
            file_name: (self.directory / file_name).exists()
            for file_name in self.REQUIRED_FILES
        }
    
    def _validate_directory(self) -> None:
        """Validate that directory exists and is actually a directory.
        
        Raises:
            ValueError: If directory validation fails
        """
        if not self.directory.exists():
            raise ValueError(f"Directory does not exist: {self.directory}")
        
        if not self.directory.is_dir():
            raise ValueError(f"Path is not a directory: {self.directory}")
        
    def get_missing_files(self) -> List[str]:
        """Get list of missing required files.
        
        Returns:
            List of file names that are missing from the directory
        """
        file_status = self.verify_files()
        return [name for name, exists in file_status.items() if not exists]
        
    def get_present_files(self) -> List[str]:
        """Get list of present required files.
        
        Returns:
            List of file names that exist in the directory
        """
        file_status = self.verify_files()
        return [name for name, exists in file_status.items() if exists]
        
    def is_complete(self) -> bool:
        """Check if all required files are present.
        
        Returns:
            True if all required files exist, False otherwise
        """
        return len(self.get_missing_files()) == 0
        
    def generate_report(self) -> str:
        """Generate detailed verification report.
        
        Returns:
            Formatted string report of verification status
        """
        try:
            if self.is_complete():
                return f"✓ All required files present in {self.directory.name}"
            
            missing = self.get_missing_files()
            present = self.get_present_files()
            
            report_lines = [f"✗ Verification failed for {self.directory.name}:"]
            
            if present:
                report_lines.append(f"  ✓ Present: {', '.join(present)}")
            
            if missing:
                report_lines.append(f"  ✗ Missing: {', '.join(missing)}")
            
            return '\n'.join(report_lines)
            
        except ValueError as e:
            return f"✗ Error: {e}"


class RefactoredCompetitionVerifier:
    """Verifies refactored competition directory structure and required files.
    
    Ensures that a refactored competition directory contains all necessary files
    and folders for a properly structured competition.
    
    Attributes:
        REQUIRED_FILES: List of files required for valid refactored competition
        REQUIRED_DIRECTORIES: List of directories required for valid refactored competition
        PRIVATE_FILES: List of files required in the private directory
        PUBLIC_FILES: List of files required in the public directory
        directory: Path to the competition directory
    """
    
    REQUIRED_FILES = [
        'description.txt',
        'metric.py',
        'prepare.py',
        'test.py'
    ]
    
    REQUIRED_DIRECTORIES = [
        'raw',
        'private',
        'public'
    ]
    
    PRIVATE_FILES = [
        'test_answer.csv'
    ]
    
    PUBLIC_FILES = [
        'sample_submission.csv',
        'description.txt'
    ]
    
    def __init__(self, directory: str) -> None:
        """Initialize verifier with target directory.
        
        Args:
            directory: Path to refactored competition directory to verify
        """
        self.directory = Path(directory)
        
    def verify_files(self) -> Dict[str, bool]:
        """Verify existence of all required files.
        
        Returns:
            Dictionary mapping file names to their existence status
            
        Raises:
            ValueError: If directory doesn't exist or isn't a directory
        """
        self._validate_directory()
        
        return {
            file_name: (self.directory / file_name).exists()
            for file_name in self.REQUIRED_FILES
        }
    
    def verify_directories(self) -> Dict[str, bool]:
        """Verify existence of all required directories.
        
        Returns:
            Dictionary mapping directory names to their existence status
        """
        self._validate_directory()
        
        return {
            dir_name: (self.directory / dir_name).exists() and (self.directory / dir_name).is_dir()
            for dir_name in self.REQUIRED_DIRECTORIES
        }
    
    def verify_private_files(self) -> Dict[str, bool]:
        """Verify existence of all required files in private directory.
        
        Returns:
            Dictionary mapping private file names to their existence status
        """
        self._validate_directory()
        private_dir = self.directory / 'private'
        
        if not private_dir.exists() or not private_dir.is_dir():
            return {file_name: False for file_name in self.PRIVATE_FILES}
        
        return {
            file_name: (private_dir / file_name).exists()
            for file_name in self.PRIVATE_FILES
        }
    
    def verify_public_files(self) -> Dict[str, bool]:
        """Verify existence of all required files in public directory.
        
        Returns:
            Dictionary mapping public file names to their existence status
        """
        self._validate_directory()
        public_dir = self.directory / 'public'
        
        if not public_dir.exists() or not public_dir.is_dir():
            return {file_name: False for file_name in self.PUBLIC_FILES}
        
        return {
            file_name: (public_dir / file_name).exists()
            for file_name in self.PUBLIC_FILES
        }
    
    def _validate_directory(self) -> None:
        """Validate that directory exists and is actually a directory.
        
        Raises:
            ValueError: If directory validation fails
        """
        if not self.directory.exists():
            raise ValueError(f"Directory does not exist: {self.directory}")
        
        if not self.directory.is_dir():
            raise ValueError(f"Path is not a directory: {self.directory}")
        
    def get_missing_files(self) -> List[str]:
        """Get list of missing required files.
        
        Returns:
            List of file names that are missing from the directory
        """
        file_status = self.verify_files()
        return [name for name, exists in file_status.items() if not exists]
        
    def get_missing_directories(self) -> List[str]:
        """Get list of missing required directories.
        
        Returns:
            List of directory names that are missing from the directory
        """
        dir_status = self.verify_directories()
        return [name for name, exists in dir_status.items() if not exists]
    
    def get_missing_private_files(self) -> List[str]:
        """Get list of missing required files in private directory.
        
        Returns:
            List of private file names that are missing
        """
        private_file_status = self.verify_private_files()
        return [name for name, exists in private_file_status.items() if not exists]
    
    def get_missing_public_files(self) -> List[str]:
        """Get list of missing required files in public directory.
        
        Returns:
            List of public file names that are missing
        """
        public_file_status = self.verify_public_files()
        return [name for name, exists in public_file_status.items() if not exists]
        
    def get_present_files(self) -> List[str]:
        """Get list of present required files.
        
        Returns:
            List of file names that exist in the directory
        """
        file_status = self.verify_files()
        return [name for name, exists in file_status.items() if exists]
        
    def get_present_directories(self) -> List[str]:
        """Get list of present required directories.
        
        Returns:
            List of directory names that exist in the directory
        """
        dir_status = self.verify_directories()
        return [name for name, exists in dir_status.items() if exists]
    
    def get_present_private_files(self) -> List[str]:
        """Get list of present required files in private directory.
        
        Returns:
            List of private file names that exist
        """
        private_file_status = self.verify_private_files()
        return [name for name, exists in private_file_status.items() if exists]
    
    def get_present_public_files(self) -> List[str]:
        """Get list of present required files in public directory.
        
        Returns:
            List of public file names that exist
        """
        public_file_status = self.verify_public_files()
        return [name for name, exists in public_file_status.items() if exists]
        
    def is_complete(self) -> bool:
        """Check if all required files and directories are present.
        
        Returns:
            True if all required files and directories exist, False otherwise
        """
        return (len(self.get_missing_files()) == 0 and 
                len(self.get_missing_directories()) == 0 and
                len(self.get_missing_private_files()) == 0 and
                len(self.get_missing_public_files()) == 0)
        
    def generate_report(self) -> str:
        """Generate detailed verification report.
        
        Returns:
            Formatted string report of verification status
        """
        try:
            if self.is_complete():
                return f"✓ All required files and directories present in {self.directory.name}"
            
            missing_files = self.get_missing_files()
            missing_dirs = self.get_missing_directories()
            missing_private_files = self.get_missing_private_files()
            missing_public_files = self.get_missing_public_files()
            
            present_files = self.get_present_files()
            present_dirs = self.get_present_directories()
            present_private_files = self.get_present_private_files()
            present_public_files = self.get_present_public_files()
            
            report_lines = [f"✗ Verification failed for {self.directory.name}:"]
            
            if present_files:
                report_lines.append(f"  ✓ Present files: {', '.join(present_files)}")
            
            if present_dirs:
                report_lines.append(f"  ✓ Present directories: {', '.join(present_dirs)}")
            
            if present_private_files:
                report_lines.append(f"  ✓ Present private files: {', '.join(present_private_files)}")
            
            if present_public_files:
                report_lines.append(f"  ✓ Present public files: {', '.join(present_public_files)}")
            
            if missing_files:
                report_lines.append(f"  ✗ Missing files: {', '.join(missing_files)}")
                
            if missing_dirs:
                report_lines.append(f"  ✗ Missing directories: {', '.join(missing_dirs)}")
            
            if missing_private_files:
                report_lines.append(f"  ✗ Missing private files: {', '.join(missing_private_files)}")
            
            if missing_public_files:
                report_lines.append(f"  ✗ Missing public files: {', '.join(missing_public_files)}")
            
            return '\n'.join(report_lines)
            
        except ValueError as e:
            return f"✗ Error: {e}"


def verify_competition_directory(directory: str) -> str:
    """Verify competition directory and return formatted report.
    
    Convenience function that creates a verifier instance and
    generates a verification report.
    
    Args:
        directory: Path to competition directory to verify
        
    Returns:
        Formatted verification report string
    """
    verifier = CompetitionVerifier(directory)
    return verifier.generate_report()


def verify_refactored_competition_directory(directory: str) -> str:
    """Verify refactored competition directory and return formatted report.
    
    Convenience function that creates a refactored verifier instance and
    generates a verification report.
    
    Args:
        directory: Path to refactored competition directory to verify
        
    Returns:
        Formatted verification report string
    """
    verifier = RefactoredCompetitionVerifier(directory)
    return verifier.generate_report()

