import asyncio
import shutil
import sys
import os
from pathlib import Path
from typing import Tuple
import re
import argparse

sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), '..')))

from components.verifier import verify_refactored_competition_directory


def format_refactored_competitions(workplace_dir: str = "./workplace"):
    """Format refactored competitions by verifying directories and updating imports.
    
    Args:
        workplace_dir: Base workplace directory containing refact subdirectory
    """
    refact_dir = Path(workplace_dir) / "refact"
    mle_dojo_competitions_dir = Path("./MLE-Dojo/mledojo/competitions")
    
    if not refact_dir.exists():
        print(f"Error: Directory {refact_dir} does not exist")
        return
    
    if not mle_dojo_competitions_dir.exists():
        print(f"Error: Directory {mle_dojo_competitions_dir} does not exist")
        return
    
    # Iterate through task directories in refact/
    for task_dir in refact_dir.iterdir():
        if not task_dir.is_dir():
            continue
            
        print(f"\nProcessing task: {task_dir.name}")
        
        # Look for refact directories (refact1, refact2, refact3, etc.)
        refact_subdirs = [d for d in task_dir.iterdir() if d.is_dir() and d.name.startswith('refact')]
        refact_subdirs.sort()  # Sort to process in order
        
        if not refact_subdirs:
            print(f"  No refact subdirectories found in {task_dir.name}")
            continue
        
        for refact_subdir in refact_subdirs:
            print(f"\n  Processing {refact_subdir.name} in {task_dir.name}")
            
            # Verify the competition directory
            verification_report = verify_refactored_competition_directory(str(refact_subdir))
            print(verification_report)
            
            # If verification passes, proceed with formatting
            if "✓ All required files and directories present" in verification_report:
                print(f"    Verification passed for {refact_subdir.name}")
                _update_imports(refact_subdir)
                _update_metric_class_name(refact_subdir)
                _copy_description_to_public(refact_subdir)
                _copy_to_mle_dojo(refact_subdir, mle_dojo_competitions_dir, task_dir.name)
            else:
                print(f"    Skipping {refact_subdir.name} due to missing files/directories")


def _copy_description_to_public(competition_dir: Path):
    """Copy description.txt to public/ subdirectory."""
    description_file = competition_dir / "description.txt"
    public_dir = competition_dir / "public"
    
    if description_file.exists():
        # Create public directory if it doesn't exist
        public_dir.mkdir(exist_ok=True)
        
        # Copy description.txt to public/
        shutil.copy2(description_file, public_dir / "description.txt")
        print(f"    Copied description.txt to {public_dir}")
    else:
        print(f"    Warning: description.txt not found in {competition_dir}")


def _copy_to_mle_dojo(competition_dir: Path, mle_dojo_competitions_dir: Path, task_name: str):
    """Copy competition files to MLE-Dojo structure."""
    # Create target directory with task name and refact suffix
    target_dir_name = f"{task_name}_{competition_dir.name}"
    target_dir = mle_dojo_competitions_dir / target_dir_name
    target_dir.mkdir(parents=True, exist_ok=True)
    
    # Create info/ and utils/ subdirectories
    info_dir = target_dir / "info"
    utils_dir = target_dir / "utils"
    info_dir.mkdir(exist_ok=True)
    utils_dir.mkdir(exist_ok=True)
    
    # Copy description.txt to info/
    description_file = competition_dir / "description.txt"
    if description_file.exists():
        shutil.copy2(description_file, info_dir / "description.txt")
        print(f"    Copied description.txt to {info_dir}")
    
    # Copy prepare.py to utils/
    prepare_file = competition_dir / "prepare.py"
    if prepare_file.exists():
        shutil.copy2(prepare_file, utils_dir / "prepare.py")
        print(f"    Copied prepare.py to {utils_dir}")
    
    # Copy metric.py to utils/
    metric_file = competition_dir / "metric.py"
    if metric_file.exists():
        shutil.copy2(metric_file, utils_dir / "metric.py")
        print(f"    Copied metric.py to {utils_dir}")
    
    print(f"    Created directory structure for {target_dir_name} in MLE-Dojo")


def _update_metric_class_name(competition_dir: Path):
    """Update metric class name to add Refacti suffix."""
    metric_file = competition_dir / "metric.py"
    if metric_file.exists():
        try:
            with open(metric_file, 'r', encoding='utf-8') as f:
                content = f.read()
            
            # Pattern to match class definitions that inherit from CompetitionMetrics
            pattern = r'class\s+([A-Za-z_][A-Za-z0-9_]*)\s*\(\s*CompetitionMetrics\s*\):'
            
            def replace_class_name(match):
                original_class_name = match.group(1)
                # Extract refact number from directory name (e.g., "refact1" -> "1")
                refact_num = re.search(r'refact(\d+)', competition_dir.name)
                if refact_num:
                    suffix = f"Refact{refact_num.group(1)}"
                else:
                    suffix = "Refacti"
                
                new_class_name = f"{original_class_name}{suffix}"
                return f"class {new_class_name}(CompetitionMetrics):"
            
            updated_content = re.sub(pattern, replace_class_name, content)
            
            if updated_content != content:
                with open(metric_file, 'w', encoding='utf-8') as f:
                    f.write(updated_content)
                print(f"    Updated metric class name in metric.py")
            
        except Exception as e:
            print(f"      Error updating metric class name in {metric_file}: {e}")


def _update_imports(competition_dir: Path):
    """Update imports in metric.py and prepare.py files."""
    # Update metric.py
    metric_file = competition_dir / "metric.py"
    if metric_file.exists():
        _update_file_imports(
            metric_file,
            "from samples.base_metric",
            "from mledojo.metrics.base"
        )
        print(f"    Updated imports in metric.py")
    
    # Update prepare.py
    prepare_file = competition_dir / "prepare.py"
    if prepare_file.exists():
        _update_file_imports(
            prepare_file,
            "from samples.sample_utils",
            "from mledojo.competitions.utils"
        )
        print(f"    Updated imports in prepare.py")


def _update_file_imports(file_path: Path, old_import: str, new_import: str):
    """Update import statements in a file."""
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        # Replace the import statement
        updated_content = content.replace(old_import, new_import)
        
        if updated_content != content:
            with open(file_path, 'w', encoding='utf-8') as f:
                f.write(updated_content)
    except Exception as e:
        print(f"      Error updating {file_path}: {e}")


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Format refactored competitions')
    parser.add_argument('--workplace_dir', type=str, default='./workplace',
                        help='Base workplace directory (default: ./workplace)')
    
    args = parser.parse_args()
    format_refactored_competitions(args.workplace_dir)