#!/usr/bin/env python3
"""
Basic Tool Framework
A unified framework for creating tools with both execution capabilities and LLM function schemas.
"""

import abc
import json
from typing import Any, Dict, List, Optional, Union
from pathlib import Path


class ToolError(Exception):
    """Base exception for tool operations."""
    pass


class BaseTool(abc.ABC):
    """
    Abstract base class for tools that can be executed directly and used with LLMs.
    
    This class provides a unified interface for tools that need to:
    1. Execute operations directly (like KaggleTools)
    2. Provide schema definitions for LLM function calling (like scaling_agent)
    """
    
    def __init__(self, name: str, description: str):
        """
        Initialize the base tool.
        
        Args:
            name: Tool name for function calling
            description: Tool description for LLM understanding
        """
        self.name = name
        self.description = description
    
    @abc.abstractmethod
    def execute(self, **kwargs) -> Dict[str, Any]:
        """
        Execute the tool with given parameters.
        
        Args:
            **kwargs: Tool-specific parameters
            
        Returns:
            Dict containing execution result with at least 'success' and 'result' keys
        """
        pass
    
    @abc.abstractmethod
    def get_schema(self) -> Dict[str, Any]:
        """
        Get the tool's function schema for LLM function calling.
        
        Returns:
            OpenAI function calling compatible schema
        """
        pass
    
    def get_function_definition(self) -> Dict[str, Any]:
        """
        Get the complete function definition for LLM tools array.
        
        Returns:
            Complete function definition with type and function schema
        """
        return {
            "type": "function",
            "function": self.get_schema()
        }
    
    def validate_parameters(self, parameters: Dict[str, Any]) -> Dict[str, Any]:
        """
        Validate parameters against the tool's schema.
        
        Args:
            parameters: Parameters to validate
            
        Returns:
            Validated parameters (may include defaults)
            
        Raises:
            ToolError: If validation fails
        """
        schema = self.get_schema()
        required_params = schema.get("parameters", {}).get("required", [])
        properties = schema.get("parameters", {}).get("properties", {})
        
        # Check required parameters
        for param in required_params:
            if param not in parameters:
                raise ToolError(f"Missing required parameter: {param}")
        
        # Apply defaults and validate types
        validated = {}
        for param_name, param_value in parameters.items():
            if param_name in properties:
                validated[param_name] = param_value
            else:
                raise ToolError(f"Unknown parameter: {param_name}")
        
        # Add default values for optional parameters
        for param_name, param_config in properties.items():
            if param_name not in validated and "default" in param_config:
                validated[param_name] = param_config["default"]
        
        return validated
    
    def safe_execute(self, **kwargs) -> Dict[str, Any]:
        """
        Execute the tool with parameter validation and error handling.
        
        Args:
            **kwargs: Tool parameters
            
        Returns:
            Standardized result dict with success, result, and optional error fields
        """
        try:
            validated_params = self.validate_parameters(kwargs)
            result = self.execute(**validated_params)
            
            # Ensure result has required fields
            if not isinstance(result, dict):
                result = {"success": True, "result": result}
            
            if "success" not in result:
                result["success"] = True
            
            return result
            
        except ToolError as e:
            return {
                "success": False,
                "error": str(e),
                "error_type": "validation_error"
            }
        except Exception as e:
            return {
                "success": False,
                "error": str(e),
                "error_type": "execution_error"
            }


class ToolRegistry:
    """Registry for managing multiple tools."""
    
    def __init__(self):
        """Initialize empty tool registry."""
        self._tools: Dict[str, BaseTool] = {}
    
    def register(self, tool: BaseTool) -> None:
        """
        Register a tool.
        
        Args:
            tool: Tool instance to register
            
        Raises:
            ToolError: If tool name already registered
        """
        if tool.name in self._tools:
            raise ToolError(f"Tool '{tool.name}' already registered")
        
        self._tools[tool.name] = tool
    
    def unregister(self, tool_name: str) -> None:
        """
        Unregister a tool.
        
        Args:
            tool_name: Name of tool to unregister
            
        Raises:
            ToolError: If tool not found
        """
        if tool_name not in self._tools:
            raise ToolError(f"Tool '{tool_name}' not found")
        
        del self._tools[tool_name]
    
    def get_tool(self, tool_name: str) -> BaseTool:
        """
        Get a registered tool.
        
        Args:
            tool_name: Name of tool to retrieve
            
        Returns:
            Tool instance
            
        Raises:
            ToolError: If tool not found
        """
        if tool_name not in self._tools:
            raise ToolError(f"Tool '{tool_name}' not found")
        
        return self._tools[tool_name]
    
    def list_tools(self) -> List[str]:
        """
        List all registered tool names.
        
        Returns:
            List of tool names
        """
        return list(self._tools.keys())
    
    def get_schemas(self) -> List[Dict[str, Any]]:
        """
        Get function schemas for all registered tools.
        
        Returns:
            List of OpenAI function calling compatible schemas
        """
        return [tool.get_function_definition() for tool in self._tools.values()]
    
    def execute_tool(self, tool_name: str, **kwargs) -> Dict[str, Any]:
        """
        Execute a registered tool by name.
        
        Args:
            tool_name: Name of tool to execute
            **kwargs: Tool parameters
            
        Returns:
            Tool execution result
        """
        tool = self.get_tool(tool_name)
        return tool.safe_execute(**kwargs)


class FileBasedTool(BaseTool):
    """
    Base class for tools that work with files and need path validation.
    
    Provides common functionality for file operations with security constraints.
    """
    
    def __init__(self, name: str, description: str, work_directory: Optional[str] = None):
        """
        Initialize file-based tool.
        
        Args:
            name: Tool name
            description: Tool description
            work_directory: Allowed working directory (defaults to current directory)
        """
        super().__init__(name, description)
        
        if work_directory is None:
            self.work_directory = Path.cwd()
        else:
            self.work_directory = Path(work_directory).resolve()
        
        # Ensure work directory exists
        self.work_directory.mkdir(parents=True, exist_ok=True)
    
    def _is_path_allowed(self, path: Union[str, Path]) -> bool:
        """Check if path is within the work directory."""
        try:
            path = Path(path).resolve()
            return path.is_relative_to(self.work_directory)
        except (ValueError, OSError):
            return False
    
    def _validate_path(self, path: Union[str, Path], operation: str) -> Path:
        """
        Validate and return resolved path if within work directory.
        
        Args:
            path: Path to validate
            operation: Operation description for error messages
            
        Returns:
            Validated resolved path
            
        Raises:
            ToolError: If path is outside work directory
        """
        resolved_path = Path(path).resolve()
        if not self._is_path_allowed(resolved_path):
            raise ToolError(
                f"{operation} operation denied: Path '{path}' is outside work directory '{self.work_directory}'"
            )
        return resolved_path
    
    def get_work_directory(self) -> str:
        """Get the current work directory."""
        return str(self.work_directory)
    
    def set_work_directory(self, new_work_dir: str) -> Dict[str, Any]:
        """
        Set a new work directory.
        
        Args:
            new_work_dir: Path to new work directory
            
        Returns:
            Dict with operation status
        """
        try:
            old_work_dir = self.work_directory
            new_path = Path(new_work_dir).resolve()
            new_path.mkdir(parents=True, exist_ok=True)
            self.work_directory = new_path
            
            return {
                "success": True,
                "old_work_directory": str(old_work_dir),
                "new_work_directory": str(new_path),
                "message": f"Work directory changed to: {new_path}"
            }
        except Exception as e:
            return {
                "success": False,
                "error": f"Failed to set work directory: {str(e)}",
                "current_work_directory": str(self.work_directory)
            }


# Global tool registry instance
default_registry = ToolRegistry()
