import shutil
import os
from pathlib import Path
from typing import Union


def copy_folder(src: Union[str, Path], dst: Union[str, Path]) -> None:
    """
    Copy an entire folder from source to destination.
    
    Args:
        src: Source folder path
        dst: Destination folder path
    """
    src_path = Path(src)
    dst_path = Path(dst)
    
    if not src_path.exists():
        raise FileNotFoundError(f"Source folder does not exist: {src_path}")
    
    if not src_path.is_dir():
        raise NotADirectoryError(f"Source is not a directory: {src_path}")
    
    # Create parent directories if they don't exist
    dst_path.parent.mkdir(parents=True, exist_ok=True)
    
    # Copy the entire folder
    shutil.copytree(src_path, dst_path, dirs_exist_ok=True)


def copy_file(src: Union[str, Path], dst: Union[str, Path]) -> None:
    """
    Copy a single file from source to destination.
    
    Args:
        src: Source file path
        dst: Destination file path
    """
    src_path = Path(src)
    dst_path = Path(dst)
    
    if not src_path.exists():
        raise FileNotFoundError(f"Source file does not exist: {src_path}")
    
    if not src_path.is_file():
        raise IsADirectoryError(f"Source is not a file: {src_path}")
    
    # Create parent directories if they don't exist
    dst_path.parent.mkdir(parents=True, exist_ok=True)
    
    # Copy the file
    shutil.copy2(src_path, dst_path)


def copy_all_files_in_folder(src_folder: Union[str, Path], dst_folder: Union[str, Path]) -> None:
    """
    Copy all files from source folder to destination folder (without copying the folder structure).
    Only copies files, not subdirectories.
    
    Args:
        src_folder: Source folder path
        dst_folder: Destination folder path
    """
    src_path = Path(src_folder)
    dst_path = Path(dst_folder)
    
    if not src_path.exists():
        raise FileNotFoundError(f"Source folder does not exist: {src_path}")
    
    if not src_path.is_dir():
        raise NotADirectoryError(f"Source is not a directory: {src_path}")
    
    # Create destination folder if it doesn't exist
    dst_path.mkdir(parents=True, exist_ok=True)
    
    # Copy all files in the source folder
    for item in src_path.iterdir():
        if item.is_file():
            dst_file = dst_path / item.name
            shutil.copy2(item, dst_file)
