_print() {
    local total_lines=$(awk 'END {print NR}' $CURRENT_FILE)
    echo "[File: $(realpath --relative-to=$ROOT $CURRENT_FILE) ($total_lines lines total)]"
    local effective_window=$((WINDOW * WINDOW_FACTOR))
    lines_above=$(jq -n "$CURRENT_LINE - $effective_window/2" | jq '[0, .] | max')
    lines_below=$(jq -n "$total_lines - $CURRENT_LINE - $effective_window/2" | jq '[0, .] | max')
    if [ $lines_above -gt 0 ]; then
        echo "($lines_above more lines above)"
    fi
    cat $CURRENT_FILE | grep -n $ | head -n $(jq -n "[$CURRENT_LINE + $effective_window/2, $effective_window/2] | max | floor") | tail -n $(jq -n "$effective_window")
    if [ $lines_below -gt 0 ]; then
        echo "($lines_below more lines below)"
    fi
}

_print_search() {
    if [ -z "$SEARCH_RESULTS" ]
    then
        echo "No search results found."
        return
    # else if number of search files is > 50, print "Too many search results. Please narrow your search."
    elif [ ${#SEARCH_RESULTS[@]} -gt 50 ]
    then
        echo "Found ${#SEARCH_RESULTS[@]} search results."
        echo "Too many search results. Please consider narrowing your search."
        return
    fi
    export CURRENT_FILE=$(realpath ${SEARCH_FILES[$SEARCH_INDEX]})
    export CURRENT_LINE=${SEARCH_RESULTS[$SEARCH_INDEX]}
    echo "[Result "$(jq -n "($SEARCH_INDEX + 1)")"/${#SEARCH_RESULTS[@]} (Line $CURRENT_LINE)]"
    _reset_window_size
    _constrain_line  # reset CURRENT_LINE if it is out of bounds
    _print
}

_constrain_line() {
    if [ -z "$CURRENT_FILE" ]
    then
        echo "No file open. Use the open command first."
        return
    fi
    local max_line=$(awk 'END {print NR}' $CURRENT_FILE)
    local effective_window=$((WINDOW * WINDOW_FACTOR))
    export CURRENT_LINE=$(jq -n "[$CURRENT_LINE, $max_line - $effective_window/2] | min")
    export CURRENT_LINE=$(jq -n "[$CURRENT_LINE, $effective_window/2] | max")
}

# @yaml
# signature: open <path>
# docstring: opens the file at the given path in the editor
open() {
    if [ -z "$1" ]
    then
        echo "Usage: open <file>"
        return
    fi

    if [ -f "$1" ]
    then
        if [ "$(realpath $1)" = "$CURRENT_FILE" ]
        then
            echo "$1"' is already open. Use `scroll_down`, `scroll_up`, `goto` and `search` to navigate file.'
            return
        fi
        export CURRENT_FILE=$(realpath $1)
        export CURRENT_LINE=$(jq -n "$WINDOW/2")
        _reset_window_size
        _constrain_line
        _print
    else
        echo "File $1 not found"
    fi
}

# @yaml
# signature: scroll_down
# docstring: expands the window down to show more lines of the file
scroll_down() {
    if [ -z "$CURRENT_FILE" ]
    then
        echo "No file open. Use the open command first."
        return
    fi
    # we want to expand the current window by WINDOW lines
    # increment WINDOW_FACTOR by 1
    # Move the CURRENT_LINE down by WINDOW/2
    # if the number of lines in the file is less than WINDOW, return an error
    max_lines=$(awk 'END {print NR}' $CURRENT_FILE)
    if [ $max_lines -lt $WINDOW ]
    then
        echo "Warning: All file contents are already displayed."
    fi
    export WINDOW_FACTOR=$((WINDOW_FACTOR + 1))
    export CURRENT_LINE=$(jq -n "$CURRENT_LINE + $WINDOW/2")
    _constrain_line
    _print
}

_reset_window_size() {
    export WINDOW_FACTOR=1
}

# @yaml
# signature: goto <n>
# docstring: moves the window to be centered on line n; resets the window size
goto() {
    if [ $# -gt 1 ]; then
        echo "goto allows only one line number at a time."
        return
    fi
    if [ -z "$CURRENT_FILE" ]
    then
        echo "No file open. Use the open command first."
        return
    fi
    if [ -z "$1" ]
    then
        echo "Usage: goto <line>"
        return
    fi
    if ! [[ $1 =~ ^[0-9]+$ ]]
    then
        echo "Usage: goto <line>"
        echo "Error: <line> must be a number"
        return
    fi
    local max_line=$(awk 'END {print NR}' $CURRENT_FILE)
    if [ $1 -gt $max_line ]
    then
        echo "Error: <line> must be less than or equal to $max_line"
        return
    fi
    export CURRENT_LINE=$1
    _reset_window_size
    _constrain_line
    _print
}

# @yaml
# signature: get_symbols [<path>]
# docstring: gets all of the symbols in the given file (defaults to current open file if no file is provided)
get_symbols() {
    if [ -z "$1" ]; then
        if [ -z "$CURRENT_FILE" ]; then
            echo "Usage: get_symbols <file>"
            return
        fi
        file_path="$CURRENT_FILE"
    else
        file_path="$1"
    fi

    if [ ! -f "$file_path" ]; then
        echo "File not found: $file_path"
        return
    fi

    /root/miniconda3/bin/python3 /root/tools/get_symbols.py "$file_path"
}

# @yaml
# signature: search <string> [<file_path>]
# docstring: searches for the given (case sensitive) string in the specified file or the open file
search() {
    if [ $# -lt 1 ] || [ $# -gt 2 ]; then
        echo "Usage: search <term> [<file_path>]"
        return
    fi

    search_term="$1"
    file_path="$2"

    if [ -z "$file_path" ]; then
        if [ -z "$CURRENT_FILE" ]; then
            echo "No file open. Provide a file path or use the open command first."
            return
        fi
        file_path="$CURRENT_FILE"
    fi

    if [ ! -f "$file_path" ]; then
        echo "The given file does not exist: $file_path"
        return
    fi

    export SEARCH_FILES=( $(grep -nIH "$search_term" "$file_path" | cut -d: -f1) )
    export SEARCH_RESULTS=( $(grep -nIH "$search_term" "$file_path" | cut -d: -f2) )
    export SEARCH_INDEX=0
    _print_search
}

# @yaml
# signature: search_all <string> [directory]
# docstring: searches for the given (case sensitive) string in all files in the specified directory (or the current working directory)
search_all() {
    if [ $# -lt 1 ] || [ $# -gt 2 ]; then
        echo "Usage: search_all <term> [<directory>]"
        return
    fi

    search_term="$1"
    directory="$2"

    if [ -z "$directory" ]; then
        # Use current directory if not provided
        directory="."
    fi

    if [ ! -d "$directory" ]; then
        # Check that the directory exists
        echo "The given directory does not exist: $directory"
        return
    fi

    export SEARCH_FILES=( $(grep -rnIH "$search_term" "$directory" | cut -d: -f1) )
    export SEARCH_RESULTS=( $(grep -rnIH "$search_term" "$directory" | cut -d: -f2) )
    export SEARCH_INDEX=0
    _print_search
}

# moves to the next search result
next() {
    if [ -z "$CURRENT_FILE" ]
    then
        echo "No file open. Use the open command first."
        return
    fi
    export SEARCH_INDEX=$(jq -n "($SEARCH_INDEX + 1) % ([${#SEARCH_RESULTS[@]}, 1] | max)")
    _print_search
}

# moves to the previous search result
prev() {
    if [ -z "$CURRENT_FILE" ]
    then
        echo "No file open. Use the open command first."
        return
    fi
    export SEARCH_INDEX=$(jq -n "($SEARCH_INDEX - 1) % ([${#SEARCH_RESULTS[@]}, 1] | max)")
    _print_search
}
