#PACKAGES -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

using Optim, Plots, DelimitedFiles, LinearAlgebra, Random, StatsBase, FiniteDifferences, LaTeXStrings , EasyFit, Printf, FFTW, Pkg, Noise, Clustering, Dierckx, BSplineKit, MultivariateStats, Flux, Combinatorics, Bigsimr, DataFrames, JLD, Distributions, FFTW, Statistics

#FUNCTIONS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#Functions for Todorov's and Numerical GD algorithms (1D)
cd(@__DIR__)#to go back to the directory of the script
include("../MyFunctions/functions_TOD_and_GD_1D.jl")

#Functions for full GD algorithms (1D and multiple dimensions)
cd(@__DIR__)#to go back to the directory of the script
include("../MyFunctions/functions_full_GD_algorithms.jl")

#TASK PARAMETERS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

Σ_x_0 = 0.0  # one of these two variances (note that they're variances) must be set at zero [probably we should always have Σ_z_0 = 0 to be consisten with (Todorov, 2005)--> see point (A) in "TO BE EXPLORED" session in the file "functions_full_GD_algorithms.jl"]
Σ_z_0 = 0.0
μ_x_0 = 1.0  # these two variables, the initial means of the state and state estimate, should be the same
μ_z_0 = μ_x_0
x_z_0_mean = μ_x_0
T =  100 #Note that T must be T>2 to have a non-trivial problem and to make the scripts run (with T=2 there is no estimation problem to solve)
q = 1
q_T = 20 #q_T is the weight of the terminal state in the cost function, higher than q to make the terminal state more important (being the task relevant cost in most of the cases)
r = 1
a = 1.0
b = 1.0
m = a
n = b
H = 1
σ_ξ = 0.5
σ_ϵ_control_dep_noise = 0.5
σ_ω_add_sensory_noise = 0.5
σ_ρ_mult_sensory_noise = 0.5 #NOTE: in the pdf this is called σ_ν
σ_η_internal_noise_min = 0.0 #minimum internal noise level 
σ_η_internal_noise_max = 2.0 #maximum internal noise level (2.0)
Δ_σ_η_internal_noise = 0.1 #step size for the internal noise level: we test the performance of the system for different values of the internal noise level

σ_η_internal_noise_min_adaptability = 0.0 #minimum internal noise level to test the adaptability of the system
σ_η_internal_noise_max_adaptability = 3.0 #maximum internal noise level to test the adaptability of the system (3.0)
Δ_σ_η_internal_noise_adaptability = 0.1 #step size for the internal noise level to test the adaptability of the system

#OPTIMIZATION PARAMETERS -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------

#Todorov
iterations_TOD = 5000

#Define the initial guess for the parameter vector to be optimized
k_0 = a                           
l_0 = -0.01  #l has opposite sign wrt Todorov paper, be careful {usually, the control is negative}

#Numerical GD 
algorithm_GD = GradientDescent()
iterations_GD = 100000
# Specify options for the optimization algorithm
options_GD = Optim.Options(
    # Step size for gradient descent
    iterations = iterations_GD,  # Number of iterations
    store_trace = false   # Show optimization trace
)

#x_0 is the initial condition for the vector optmised in the classic GD optimisation. Below we set the initial condition as Todorov's solutions for k_t and l_t.
x_0 = k_0 .* ones(2*(T-1))        #we assume uniform values for k_0 and l_0 as initial conditions before the optimization
x_0[T:2*(T-1)] = l_0 .* ones(T-1)

#Iterative GD

n_order_vec_iterative_GD = zeros(T-2)
for t in 1:T-2
    n_order_vec_iterative_GD[t] = T-t
end

iterations_full_optimization_iterative_GD = 1000
iterations_control_optimization_iterative_GD = 1
iterations_estimation_optimization_iterative_GD = 1

l_start_iterative_GD = zeros(T-1)
k_start_iterative_GD = zeros(T-1)

#MC simulations
realizations_for_averaged_cost = 10000 #it depends on the noise level, but to get accurate estimates through the MC method, it seems we should use realizations ≈ 10^4 at least (as soon 

#We create this dictonary to save all the used parameters
params_dict = Dict(
    "Σ_x_0" => Σ_x_0,
    "Σ_z_0" => Σ_z_0,
    "μ_x_0" => μ_x_0,
    "μ_z_0" => μ_z_0,
    "x_z_0_mean" => x_z_0_mean,
    "T" => T,
    "q" => q,
    "q_T" => q_T,
    "r" => r,
    "a" => a,
    "b" => b,
    "m" => m,
    "n" => n,
    "H" => H,
    "σ_ξ" => σ_ξ,
    "σ_ϵ_control_dep_noise" => σ_ϵ_control_dep_noise,
    "σ_ω_add_sensory_noise" => σ_ω_add_sensory_noise,
    "σ_ρ_mult_sensory_noise" => σ_ρ_mult_sensory_noise,
    "σ_η_internal_noise_min" => σ_η_internal_noise_min,
    "σ_η_internal_noise_max" => σ_η_internal_noise_max,
    "Δ_σ_η_internal_noise" => Δ_σ_η_internal_noise,
    "σ_η_internal_noise_min_adaptability" => σ_η_internal_noise_min_adaptability,
    "σ_η_internal_noise_max_adaptability" => σ_η_internal_noise_max_adaptability,
    "Δ_σ_η_internal_noise_adaptability" => Δ_σ_η_internal_noise_adaptability
)

##-------------------------------------------------------- # (1) ACCUMULATED COST AS A FUNCTION OF internal noise level ------------------------------------------------------------------------------------

println("\n------- (1) VARYING INTERNAL NOISE (optimization) --------------------------\n")

#internal noise levels:

σ_η_vec = σ_η_internal_noise_min:Δ_σ_η_internal_noise:σ_η_internal_noise_max

k_opt_iterative_GD = zeros(length(σ_η_vec), T-1)
l_opt_iterative_GD = zeros(length(σ_η_vec), T-1)

k_opt_TOD = zeros(length(σ_η_vec), T-1)
l_opt_TOD = zeros(length(σ_η_vec), T-1)

k_opt_GD = zeros(length(σ_η_vec), T-1)
l_opt_GD = zeros(length(σ_η_vec), T-1)

expected_cost_iterative_GD_optim = zeros(length(σ_η_vec))
expected_cost_TOD_optim = zeros(length(σ_η_vec))
expected_cost_GD_optim = zeros(length(σ_η_vec))

MC_cost_iterative_GD_optim = zeros(length(σ_η_vec))
MC_cost_TOD_optim = zeros(length(σ_η_vec))
MC_cost_GD_optim = zeros(length(σ_η_vec))

MC_std_cost_iterative_GD_optim = zeros(length(σ_η_vec))
MC_std_cost_TOD_optim = zeros(length(σ_η_vec))
MC_std_cost_GD_optim = zeros(length(σ_η_vec))

#Solve the control problem with the four algorithms: (1)Todorov (2)GD (3)iterative GD: "analytical GD" 

Threads.@threads for i in 1:length(σ_η_vec)
    
    t_each_iteration = @elapsed begin
        
        seed_MC = rand(1:1000000) #we use the same seed to compute the MC estimates of the accumulated cost in the different algorithms, in order to properly compare the results.

        #(1)TODOROV's algorithm

        k_opt_TOD[i,:], l_opt_TOD[i,:] = K_L_Todorov_optimization(iterations_TOD, x_z_0_mean, Σ_x_0, Σ_z_0, k_0, l_0, a, b, H, r, q, q_T, σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec[i])
        MC_cost_TOD_optim[i], MC_std_cost_TOD_optim[i], _ = cost_MC_sampling(realizations_for_averaged_cost, seed_MC, Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, l_opt_TOD[i,:], k_opt_TOD[i,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec[i])
        expected_cost_mom_prop_TOD = expected_cost_moments_propagation_k_l_separated(Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, k_opt_TOD[i,:], l_opt_TOD[i,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec[i])

        #(2)Numerical GD

        x_0[1:T-1] = k_opt_TOD[i,:]
        x_0[T:2*(T-1)] = l_opt_TOD[i,:]

        #define the cost function
        cost_function_optimization_GD(x) = expected_cost_moments_propagation(Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, x, σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec[i])
                
        result_GD = optimize(cost_function_optimization_GD, x_0, algorithm_GD, options_GD)
        x_opt_GD = result_GD.minimizer
        expected_cost_GD_optim[i] = result_GD.minimum

        k_opt_GD[i,:] = x_opt_GD[1:T-1]
        l_opt_GD[i,:] = x_opt_GD[T:2*(T-1)]

        MC_cost_GD_optim[i], MC_std_cost_GD_optim[i], _ = cost_MC_sampling(realizations_for_averaged_cost, seed_MC, Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, l_opt_GD[i,:], k_opt_GD[i,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec[i])

        #(3)ITERATIVE GD

        l_start_iterative_GD[:] .= l_opt_TOD[i,:]
        k_start_iterative_GD[:] .= k_opt_TOD[i,:]

        l_opt_iterative_GD[i,:], k_opt_iterative_GD[i,:] = Iterative_GD_control_and_estimation_backward_optimization_coordinate_descend(n_order_vec_iterative_GD, iterations_full_optimization_iterative_GD, iterations_control_optimization_iterative_GD, iterations_estimation_optimization_iterative_GD, l_start_iterative_GD, k_start_iterative_GD, Σ_x_0, Σ_z_0, μ_z_0, T, q, q_T, r, a, b, H, σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec[i])
        MC_cost_iterative_GD_optim[i], MC_std_cost_iterative_GD_optim[i], _ = cost_MC_sampling(realizations_for_averaged_cost, seed_MC, Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, l_opt_iterative_GD[i,:], k_opt_iterative_GD[i,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec[i])
        expected_cost_iterative_GD_optim[i] = expected_cost_moments_propagation_k_l_separated(Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, k_opt_iterative_GD[i,:], l_opt_iterative_GD[i,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec[i])
    
    end
    σ_η_tmp = σ_η_vec[i]
    println("Execution time for σ_η = $σ_η_tmp: $t_each_iteration seconds\n")

end

###SAVE THE DATA ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

cd(@__DIR__)#to go back to the directory of the script
folder_location = "../Data/1D_problem_changing_internal_noise/New_Data"

# Create a dictionary to store the matrices

dict_to_save_study_cost = Dict(
    "σ_η_vec" => σ_η_vec,
    "k_opt_iterative_GD" => k_opt_iterative_GD,
    "l_opt_iterative_GD" => l_opt_iterative_GD,
    "k_opt_TOD" => k_opt_TOD,
    "l_opt_TOD" => l_opt_TOD,
    "k_opt_GD" => k_opt_GD,
    "l_opt_GD" => l_opt_GD,
    "expected_cost_iterative_GD_optim" => expected_cost_iterative_GD_optim,
    "expected_cost_TOD_optim" => expected_cost_TOD_optim,
    "expected_cost_GD_optim" => expected_cost_GD_optim,
    "MC_cost_iterative_GD_optim" => MC_cost_iterative_GD_optim,
    "MC_cost_TOD_optim" => MC_cost_TOD_optim,
    "MC_cost_GD_optim" => MC_cost_GD_optim,
    "MC_std_cost_iterative_GD_optim" => MC_std_cost_iterative_GD_optim,
    "MC_std_cost_TOD_optim" => MC_std_cost_TOD_optim,
    "MC_std_cost_GD_optim" => MC_std_cost_GD_optim,
)

# Specify the file path
file_name_save_study_cost = "1D_problem_data_changing_internal_noise.jld2"
file_path_save_study_cost = folder_location * "/" *file_name_save_study_cost

# Save the dictionary to the file
@save file_path_save_study_cost dict_to_save_study_cost

# Save the parameters 
file_name_save_parameters = "1D_problem_data_parameters.jld2"
file_path_save_parameters = folder_location * "/" * file_name_save_parameters
@save file_path_save_parameters params_dict

##-------------------------------------------------------- # (2) ADAPTABILITY TO INTERNAL NOISE ------------------------------------------------------------------------------------

println("\n------- (2) ADAPTABILITY TO INTERNAL NOISE --------------------------\n")

#internal noise levels:
 
σ_η_vec_adaptability = σ_η_internal_noise_min_adaptability:Δ_σ_η_internal_noise_adaptability:σ_η_internal_noise_max_adaptability

#Here, each column represents the performances of the system optimised for internal noise  = σ_η_vec[i=column], varying the internal noise level, following the values of σ_η_vec_adaptability[:]
expected_cost_prop_moments_iterative_GD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))
expected_cost_prop_moments_TOD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))
expected_cost_prop_moments_GD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))

MC_cost_iterative_GD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))
MC_cost_TOD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))
MC_cost_GD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))

MC_std_cost_iterative_GD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))
MC_std_cost_TOD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))
MC_std_cost_GD_noise_adapt = zeros(length(σ_η_vec_adaptability), length(σ_η_vec))

t_parfor_σ_η = @elapsed begin   

    Threads.@threads for i in 1:length(σ_η_vec_adaptability)
        
        for j in 1:length(σ_η_vec)

            seed_MC = rand(1:1000000)

            expected_cost_prop_moments_iterative_GD_noise_adapt[i,j] = expected_cost_moments_propagation_k_l_separated(Σ_x_0, Σ_z_0, μ_x_0, μ_z_0 , T, q, q_T, r, a, b, m, n, H, k_opt_iterative_GD[j,:], l_opt_iterative_GD[j,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec_adaptability[i])
            expected_cost_prop_moments_TOD_noise_adapt[i,j] = expected_cost_moments_propagation_k_l_separated(Σ_x_0, Σ_z_0, μ_x_0, μ_z_0 , T, q, q_T, r, a, b, m, n, H, k_opt_TOD[j,:], l_opt_TOD[j,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec_adaptability[i])
            expected_cost_prop_moments_GD_noise_adapt[i,j] = expected_cost_moments_propagation_k_l_separated(Σ_x_0, Σ_z_0, μ_x_0, μ_z_0 , T, q, q_T, r, a, b, m, n, H, k_opt_GD[j,:], l_opt_GD[j,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec_adaptability[i])

            MC_cost_iterative_GD_noise_adapt[i,j], MC_std_cost_iterative_GD_noise_adapt[i,j], _ = cost_MC_sampling(realizations_for_averaged_cost, seed_MC, Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, l_opt_iterative_GD[j,:], k_opt_iterative_GD[j,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec_adaptability[i])
            MC_cost_TOD_noise_adapt[i,j], MC_std_cost_TOD_noise_adapt[i,j], _ = cost_MC_sampling(realizations_for_averaged_cost, seed_MC, Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, l_opt_TOD[j,:], k_opt_TOD[j,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec_adaptability[i])
            MC_cost_GD_noise_adapt[i,j], MC_std_cost_GD_noise_adapt[i,j], _ = cost_MC_sampling(realizations_for_averaged_cost, seed_MC, Σ_x_0, Σ_z_0, μ_x_0, μ_z_0, T, q, q_T, r, a, b, m, n, H, l_opt_GD[j,:], k_opt_GD[j,:], σ_ξ, σ_ϵ_control_dep_noise, σ_ω_add_sensory_noise, σ_ρ_mult_sensory_noise, σ_η_vec_adaptability[i])

        end

    end

end

println("Execution time (test adaptability over different values of σ_η): $t_parfor_σ_η seconds\n")

# Create a dictionary to store the matrices

dict_to_save_adaptability = Dict(
    "expected_cost_prop_moments_iterative_GD_noise_adapt" => expected_cost_prop_moments_iterative_GD_noise_adapt,
    "expected_cost_prop_moments_TOD_noise_adapt" => expected_cost_prop_moments_TOD_noise_adapt,
    "expected_cost_prop_moments_GD_noise_adapt" => expected_cost_prop_moments_GD_noise_adapt,
    "MC_cost_iterative_GD_noise_adapt" => MC_cost_iterative_GD_noise_adapt,
    "MC_cost_TOD_noise_adapt" => MC_cost_TOD_noise_adapt,
    "MC_cost_GD_noise_adapt" => MC_cost_GD_noise_adapt,
    "MC_std_cost_iterative_GD_noise_adapt" => MC_std_cost_iterative_GD_noise_adapt,
    "MC_std_cost_TOD_noise_adapt" => MC_std_cost_TOD_noise_adapt,
    "MC_std_cost_GD_noise_adapt" => MC_std_cost_GD_noise_adapt,
)

# Specify the file path
file_name_save_adaptability = "1D_problem_data_adaptability.jld2"
file_path_save_adaptability = folder_location * "/" * file_name_save_adaptability

# Save the dictionary to the file
@save file_path_save_adaptability dict_to_save_adaptability
