# Copyright 2024 Bytedance Ltd. and/or its affiliates
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
Note that we don't combine the main with ray_trainer as ray_trainer is used by other main.
"""
from pathlib import Path
from verl import DataProto
import torch
import ray
import hydra
from verl.utils.reward_score import gsm8k, math, multiply, countdown, mlebench
from verl.trainer.ppo.ray_trainer import RayPPOTrainer
import shutil
import os
from absl import app
from absl import flags
from omegaconf import OmegaConf
import time
from funcy import notnone, once, select_values

FLAGS = flags.FLAGS
flags.DEFINE_string('output_dir', None, 'Output directory.')
flags.DEFINE_string('train_tasks', 'random-acts-of-pizza', 'Comma-separated list of tasks.')
flags.DEFINE_string('val_tasks', 'random-acts-of-pizza', 'Comma-separated list of tasks.')

flags.DEFINE_integer('train_batch_size', 256, 'Train batch size.')
flags.DEFINE_integer('ppo_mini_batch_size', 128, 'Mini batch size.')
flags.DEFINE_float('gpu_memory_utilization', 0.4, 'GPU utilization.')
flags.DEFINE_integer('micro_batch_size', 8, 'Micro batch size.')
flags.DEFINE_float('improve_rate', 0.0, 'Improvement rate.')
flags.DEFINE_float('debug_rate', 0.0, 'Debug rate.')
flags.DEFINE_integer('max_prompt_length', 768, 'Max prompt length.')
flags.DEFINE_integer('max_response_length', 1024, 'Max response length.')
flags.DEFINE_integer('timeout', 300, 'Reward timeout in seconds.')

flags.DEFINE_boolean('dense_reward', False, 'Whether to use line number as reward.')
flags.DEFINE_boolean('duration_aware_gradient', False, 'Whether to use duration aware gradient.')

WORST_SCORE = -10

def _select_rm_score_fn(data_source):
    if data_source == 'openai/gsm8k':
        return gsm8k.compute_score
    elif data_source == 'lighteval/MATH':
        return math.compute_score
    elif "multiply" in data_source or "arithmetic" in data_source:
        return multiply.compute_score
    elif "countdown" in data_source:
        return countdown.compute_score
    else:
        return mlebench.compute_score


@ray.remote
def compute_single_reward(
    data_item,
    tokenizer,
    max_prints_per_data_source,
    data_source_print_counter,
    output_dir,
    timeout,
    dense_reward,
    is_eval,
):
    """
    This function will run on a Ray worker. It takes a single data_item and
    returns (score, data_source_print_counter_update, index) so the calling
    code knows where to store the score and how many times we printed.
    """
    # Extract relevant pieces
    prompt_ids = data_item.batch['prompts']
    prompt_length = prompt_ids.shape[-1]
    valid_prompt_length = data_item.batch['attention_mask'][:prompt_length].sum()
    valid_prompt_ids = prompt_ids[-valid_prompt_length:]

    response_ids = data_item.batch['responses']
    valid_response_length = data_item.batch['attention_mask'][prompt_length:].sum()
    valid_response_ids = response_ids[:valid_response_length]

    # Decode
    sequences = torch.cat((valid_prompt_ids, valid_response_ids))
    sequences_str = tokenizer.decode(sequences)

    ground_truth = data_item.non_tensor_batch['reward_model']['ground_truth']
    data_source = data_item.non_tensor_batch['data_source']

    # Your existing score fn
    from verl.utils.reward_score import gsm8k, math, multiply, countdown, mlebench
    def _select_rm_score_fn(source):
        if source == 'openai/gsm8k':
            return gsm8k.compute_score
        elif source == 'lighteval/MATH':
            return math.compute_score
        elif "multiply" in source or "arithmetic" in source:
            return multiply.compute_score
        elif "countdown" in source:
            return countdown.compute_score
        else:
            return mlebench.compute_score

    compute_score_fn = _select_rm_score_fn(data_source)
    score = compute_score_fn(solution_str=sequences_str, ground_truth=ground_truth, output_dir=output_dir,
                             timeout=timeout, dense_reward=dense_reward, is_eval=is_eval)

    # Handle printing. Because we are in parallel, you need a concurrency-
    # safe way to limit prints. This is simplified; you may want a more
    # robust approach (e.g. a central coordinator).
    old_count = data_source_print_counter.get(data_source, 0)
    if old_count < max_prints_per_data_source:
        print(f"[DataSource: {data_source}, Count={old_count}] {sequences_str}")
        data_source_print_counter[data_source] = old_count + 1

    # Return all needed info
    return score, data_source_print_counter


class RewardManagerSingleThread():
    """The reward manager.
    """

    def __init__(self, tokenizer, num_examine) -> None:
        self.tokenizer = tokenizer
        self.num_examine = num_examine  # the number of batches of decoded responses to print to the console

    def __call__(self, data: DataProto):
        """We will expand this function gradually based on the available datasets"""

        # If there is rm score, we directly return rm score. Otherwise, we compute via rm_score_fn
        if 'rm_scores' in data.batch.keys():
            return data.batch['rm_scores']

        reward_tensor = torch.zeros_like(data.batch['responses'], dtype=torch.float32)

        already_print_data_sources = {}

        for i in range(len(data)):
            data_item = data[i]  # DataProtoItem

            prompt_ids = data_item.batch['prompts']

            prompt_length = prompt_ids.shape[-1]

            valid_prompt_length = data_item.batch['attention_mask'][:prompt_length].sum()
            valid_prompt_ids = prompt_ids[-valid_prompt_length:]

            response_ids = data_item.batch['responses']
            valid_response_length = data_item.batch['attention_mask'][prompt_length:].sum()
            valid_response_ids = response_ids[:valid_response_length]

            # decode
            sequences = torch.cat((valid_prompt_ids, valid_response_ids))
            sequences_str = self.tokenizer.decode(sequences)

            ground_truth = data_item.non_tensor_batch['reward_model']['ground_truth']

            # select rm_score
            data_source = data_item.non_tensor_batch['data_source']
            compute_score_fn = _select_rm_score_fn(data_source)

            score = compute_score_fn(solution_str=sequences_str, ground_truth=ground_truth)
            reward_tensor[i, valid_response_length - 1] = score

            if data_source not in already_print_data_sources:
                already_print_data_sources[data_source] = 0

            if already_print_data_sources[data_source] < self.num_examine:
                already_print_data_sources[data_source] += 1
                print(sequences_str)

        return reward_tensor

class RewardManager():
    """The reward manager.
    """

    def __init__(self, tokenizer, num_examine, output_dir, timeout=300, dense_reward=False, is_eval=False) -> None:
        self.tokenizer = tokenizer
        self.num_examine = num_examine  # times to print for each data source
        self.timeout = timeout
        self.output_dir = output_dir
        self.dense_reward = dense_reward
        self.is_eval = is_eval

    def __call__(self, data: DataProto):
        # If there is rm score, we directly return it.
        if 'rm_scores' in data.batch:
            return data.batch['rm_scores']

        # Prepare a tensor for final rewards
        reward_tensor = torch.zeros_like(data.batch['responses'], dtype=torch.float32)
        time_tensor = torch.zeros_like(data.batch['responses'][:, 0], dtype=torch.float32)
        actual_score_tensor = torch.zeros_like(data.batch['responses'][:, 0], dtype=torch.float32)

        # We'll store references to Ray tasks here
        task_refs = []

        # We also keep track of counters in each task
        # so that each item can decide whether it should print.
        # Each remote call modifies its local copy and returns it,
        # but to keep logic simple we can just pass an empty dict
        # each time or incorporate some more advanced approach.
        # Here, we pass an empty dict so that each item can do
        # printing up to `self.num_examine` times. You probably
        # want something more global than that, but this is an example.
        data_source_print_counter = {}

        # Launch all tasks in parallel
        for i in range(len(data)):
            data_item = data[i]  # DataProtoItem
            # We pass everything needed to the remote function
            ref = compute_single_reward.remote(
                data_item=data_item,
                tokenizer=self.tokenizer,
                max_prints_per_data_source=self.num_examine,
                data_source_print_counter=data_source_print_counter,
                output_dir=self.output_dir,
                timeout=self.timeout,
                dense_reward=self.dense_reward,
                is_eval=self.is_eval,
            )
            task_refs.append((i, ref))

        results = []
        for i, ref in task_refs:
            try:
                result = ray.get(ref, timeout=self.timeout)
                results.append((i, result))
            except ray.exceptions.GetTimeoutError:
                # Handle the timeout
                print(f"Warning: compute_single_reward timed out after {self.timeout} seconds.")
                # Optional:
                ray.cancel(ref, force=True)
                # Default result if timed out:
                results.append((i, ((WORST_SCORE, WORST_SCORE, 1.), {})))
            except ray.exceptions.RayTaskError as e:
                print(f"Warning: Worker crashed or task failed with error: {e} for index {i}.")
                results.append((i, ((WORST_SCORE, WORST_SCORE, 1.), {})))
            except Exception as e:
                print(f"Warning: Unknown error: {e} for index {i}.")
                results.append((i, ((WORST_SCORE, WORST_SCORE, 1.), {})))
        for i, (score, _) in results:
            # For simplicity, put the entire score at the last token index
            # or however you did it in your code
            prompt_length = data[i].batch['prompts'].shape[-1]
            valid_response_length = data[i].batch['attention_mask'][prompt_length:].sum()
            reward_tensor[i, valid_response_length - 1] = score[0]
            actual_score_tensor[i] = score[1]
            time_tensor[i] = score[2]

        return reward_tensor, actual_score_tensor, time_tensor

#@hydra.main(config_path='config', config_name='ppo_trainer', version_base=None)
def hydra_main(config):
    if not ray.is_initialized():
        # this is for local ray cluster
        ray.init(runtime_env={'env_vars': {'TOKENIZERS_PARALLELISM': 'true', 'NCCL_DEBUG': 'WARN'}})

    while True:
      try:
        ray.get(main_task.remote(config))
        break
      except Exception as e:
        print(e)

        # Reset ray cluster
        ray.shutdown()
        # Wait to ensure clean shutdown
        time.sleep(10)
        # Restart Ray
        ray.init(runtime_env={'env_vars': {'TOKENIZERS_PARALLELISM': 'true', 'NCCL_DEBUG': 'WARN'}})

        continue


@ray.remote
def main_task(config):
    from verl.utils.fs import copy_local_path_from_hdfs
    from transformers import AutoTokenizer

    # print initial config
    from pprint import pprint
    from omegaconf import OmegaConf
    pprint(OmegaConf.to_container(config, resolve=True))  # resolve=True will eval symbol values
    OmegaConf.resolve(config)

    # download the checkpoint from hdfs
    local_path = copy_local_path_from_hdfs(config.actor_rollout_ref.model.path)

    # instantiate tokenizer
    from verl.utils import hf_tokenizer
    tokenizer = hf_tokenizer(local_path)

    # define worker classes
    if config.actor_rollout_ref.actor.strategy == 'fsdp':
        assert config.actor_rollout_ref.actor.strategy == config.critic.strategy
        from verl.workers.fsdp_workers import ActorRolloutRefWorker, CriticWorker
        from verl.single_controller.ray import RayWorkerGroup
        ray_worker_group_cls = RayWorkerGroup

    elif config.actor_rollout_ref.actor.strategy == 'megatron':
        assert config.actor_rollout_ref.actor.strategy == config.critic.strategy
        from verl.workers.megatron_workers import ActorRolloutRefWorker, CriticWorker
        from verl.single_controller.ray.megatron import NVMegatronRayWorkerGroup
        ray_worker_group_cls = NVMegatronRayWorkerGroup

    else:
        raise NotImplementedError

    from verl.trainer.ppo.ray_trainer import ResourcePoolManager, Role

    role_worker_mapping = {
        Role.ActorRollout: ray.remote(max_restarts=3)(ActorRolloutRefWorker),
        Role.Critic: ray.remote(max_restarts=3)(CriticWorker),
        Role.RefPolicy: ray.remote(max_restarts=3)(ActorRolloutRefWorker)
    }

    global_pool_id = 'global_pool'
    resource_pool_spec = {
        global_pool_id: [config.trainer.n_gpus_per_node] * config.trainer.nnodes,
    }
    mapping = {
        Role.ActorRollout: global_pool_id,
        Role.Critic: global_pool_id,
        Role.RefPolicy: global_pool_id,
    }

    # we should adopt a multi-source reward function here
    # - for rule-based rm, we directly call a reward score
    # - for model-based rm, we call a model
    # - for code related prompt, we send to a sandbox if there are test cases
    # - finally, we combine all the rewards together
    # - The reward type depends on the tag of the data
    if config.reward_model.enable:
        if config.reward_model.strategy == 'fsdp':
            from verl.workers.fsdp_workers import RewardModelWorker
        elif config.reward_model.strategy == 'megatron':
            from verl.workers.megatron_workers import RewardModelWorker
        else:
            raise NotImplementedError
        role_worker_mapping[Role.RewardModel] = ray.remote(RewardModelWorker)
        mapping[Role.RewardModel] = global_pool_id


    Path(f'{config.trainer.output_dir}/generation').mkdir(parents=True, exist_ok=True)

    reward_fn = RewardManager(tokenizer=tokenizer, num_examine=0, output_dir=config.trainer.output_dir, timeout=config.data.timeout, dense_reward=config.data.dense_reward)

    # Note that we always use function-based RM for validation
    val_reward_fn = RewardManager(tokenizer=tokenizer, num_examine=1, output_dir=config.trainer.output_dir, timeout=config.data.timeout, dense_reward=config.data.dense_reward, is_eval=True)

    resource_pool_manager = ResourcePoolManager(resource_pool_spec=resource_pool_spec, mapping=mapping)

    trainer = RayPPOTrainer(config=config,
                            tokenizer=tokenizer,
                            role_worker_mapping=role_worker_mapping,
                            resource_pool_manager=resource_pool_manager,
                            ray_worker_group_cls=ray_worker_group_cls,
                            reward_fn=reward_fn,
                            val_reward_fn=val_reward_fn,
                            output_dir=config.trainer.output_dir)
    trainer.init_workers()
    trainer.fit()

def main(_):
  # Config override
  base_config = OmegaConf.load("./config/ppo_trainer.yaml")
  base_config.trainer.output_dir = FLAGS.output_dir
  base_config.trainer.default_local_dir = f"{FLAGS.output_dir}/ckpt"

  # Tasks
  data_dir = '/workdir/data'
  train_tasks = FLAGS.train_tasks.split(',')
  base_config.data.train_files = [f"{data_dir}/{task}/train.parquet" for task in train_tasks]
  val_tasks = FLAGS.val_tasks.split(',')
  base_config.data.val_files = [f"{data_dir}/{task}/test.parquet" for task in val_tasks]

  # Hyperparameters
  base_config.data.train_batch_size = FLAGS.train_batch_size
  base_config.data.timeout = FLAGS.timeout
  base_config.data.dense_reward = FLAGS.dense_reward
  base_config.data.max_prompt_length = FLAGS.max_prompt_length
  base_config.data.max_response_length = FLAGS.max_response_length
  base_config.actor_rollout_ref.actor.ppo_mini_batch_size = FLAGS.ppo_mini_batch_size
  base_config.actor_rollout_ref.rollout.gpu_memory_utilization = FLAGS.gpu_memory_utilization

  base_config.actor_rollout_ref.actor.ppo_micro_batch_size = FLAGS.micro_batch_size
  base_config.actor_rollout_ref.rollout.log_prob_micro_batch_size = FLAGS.micro_batch_size
  base_config.actor_rollout_ref.ref.log_prob_micro_batch_size = FLAGS.micro_batch_size
  base_config.critic.ppo_micro_batch_size = FLAGS.micro_batch_size
  base_config.trainer.improve_rate = FLAGS.improve_rate
  base_config.trainer.debug_rate = FLAGS.debug_rate
  base_config.trainer.duration_aware_gradient = FLAGS.duration_aware_gradient

  for task in set(train_tasks + val_tasks):
    # Setup working dir by copying over files
    shutil.copytree(f'/workdir/mle-bench/data/{task}', f'/workdir/{task}')

  hydra_main(base_config)

if __name__ == '__main__':
  app.run(main)
