import glob
import os


from scipy import signal
from torch.utils.data import DataLoader
from torchvision.datasets.utils import list_dir
from torchvision.datasets.folder import make_dataset
# from torchvision.datasets.video_utils import VideoClips
from torchvision.datasets.vision import VisionDataset

try:
    from VideoClips import VideoClips, few_shot_setup
except:
    from .VideoClips import VideoClips, few_shot_setup


class HMDB51(VisionDataset):
    """
    `HMDB51 <http://serre-lab.clps.brown.edu/resource/hmdb-a-large-human-motion-database/>`_
    dataset.
    HMDB51 is an action recognition video dataset.
    This dataset consider every video as a collection of video clips of fixed size, specified
    by ``frames_per_clip``, where the step in frames between each clip is given by
    ``step_between_clips``.
    To give an example, for 2 videos with 10 and 15 frames respectively, if ``frames_per_clip=5``
    and ``step_between_clips=5``, the dataset size will be (2 + 3) = 5, where the first two
    elements will come from video 1, and the next three elements from video 2.
    Note that we drop clips which do not have exactly ``frames_per_clip`` elements, so not all
    frames in a video might be present.
    Internally, it uses a VideoClips object to handle clip creation.
    Args:
        root (string): Root directory of the HMDB51 Dataset.
        annotation_path (str): path to the folder containing the split files
        frames_per_clip (int): number of frames in a clip.
        step_between_clips (int): number of frames between each clip.
        fold (int, optional): which fold to use. Should be between 1 and 3.
        train (bool, optional): if ``True``, creates a dataset from the train split,
            otherwise from the ``test`` split.
        transform (callable, optional): A function/transform that  takes in a TxHxWxC video
            and returns a transformed version.
    Returns:
        video (Tensor[T, H, W, C]): the `T` video frames
        audio(Tensor[K, L]): the audio frames, where `K` is the number of channels
            and `L` is the number of points
        label (int): class of the video clip
    """

    data_url = "http://serre-lab.clps.brown.edu/wp-content/uploads/2013/10/hmdb51_org.rar"
    splits = {
        "url": "http://serre-lab.clps.brown.edu/wp-content/uploads/2013/10/test_train_splits.rar",
        "md5": "15e67781e70dcfbdce2d7dbb9b3344b5"
    }

    def __init__(
        self, 
        root='/datasets01_101/hmdb51/112018/data/', 
        annotation_path='/datasets01_101/hmdb51/112018/splits/', 
        frames_per_clip=16, 
        step_between_clips=1,
        frame_rate=None,
        subsample=False, 
        fold=1, 
        train=True, 
        transform=None,
        num_shots=-1,
        _precomputed_metadata=None, 
        num_workers=1, 
        _video_width=0,
        _video_height=0, 
        _video_min_dimension=0, 
        _audio_samples=0
    ):
        super(HMDB51, self).__init__(root)
        if not 1 <= fold <= 3:
            raise ValueError("fold should be between 1 and 3, got {}".format(fold))

        extensions = ('avi',)
        self.fold = fold
        self.train = train

        classes = list(sorted(list_dir(root)))
        class_to_idx = {classes[i]: i for i in range(len(classes))}
        self.samples = make_dataset(self.root, class_to_idx, extensions, is_valid_file=None)
        self.classes = classes
        self.subsample = subsample
        self.num_shots = num_shots
        video_list = [x[0] for x in self.samples]
        video_clips = VideoClips(
            video_list,
            frames_per_clip,
            step_between_clips,
            frame_rate,
            _precomputed_metadata,
            num_workers=num_workers,
            _video_width=_video_width,
            _video_height=_video_height,
            _video_min_dimension=_video_min_dimension,
            _audio_samples=_audio_samples,
        )
        self.video_clips_metadata = video_clips.metadata
        self.indices = self._select_fold(video_list, annotation_path, fold, train)            
        self.video_clips = video_clips.subset(self.indices)
        self.transform = transform
        print(f"Total Number of clips: {len(self.video_clips)}", flush=True)
        print(f"Total Number of videos: {len(video_list)}", flush=True)

    @property
    def metadata(self):
        return self.video_clips_metadata

    def _select_fold(self, video_list, annotation_path, fold, train):
        print(f"Getting HMDB51 dataset. Train Mode: {train}, fold: {fold}", flush=True)
        target_tag = 1 if train else 2
        name = "*test_split{}.txt".format(fold)
        files = glob.glob(os.path.join(annotation_path, name))
        selected_files = []
        for f in files:
            with open(f, "r") as fid:
                data = fid.readlines()
                data = [x.strip().split(" ") for x in data]
                data = [x[0] for x in data if int(x[1]) == target_tag]
                selected_files.extend(data)
        selected_files = set(selected_files)
        if self.num_shots != -1 and self.num_shots > 0:
            indices = [i for i in range(len(video_list)) if os.path.basename(video_list[i]) in selected_files]
            indices = few_shot_setup(self.samples, indices, num_shots=self.num_shots)
            print(f"Number of videos: {len(indices)}")
        else:
            indices = [i for i in range(len(video_list)) if os.path.basename(video_list[i]) in selected_files]
        return indices

    def __len__(self):
        return self.video_clips.num_clips()

    def __getitem__(self, idx):
        video, _, info, video_idx = self.video_clips.get_clip(idx)
        # subsample the input if that's what we want
        if self.subsample:
            video = video[::4, :, :, :] #  time x channels x h x w

        label = self.samples[self.indices[video_idx]][1]

        if self.transform is not None:
            video = self.transform(video)

        return video, label, idx, video_idx


if __name__ == '__main__':

    import time
    val_dataset = HMDB51(
        train=False,
        subsample=True,
        num_shots=2
    )
    print(len(val_dataset))
    val_loader = DataLoader(
        val_dataset, 
        batch_size=32, 
        num_workers=0, 
    )

    tic = time.time()
    for idx, batch in enumerate(val_loader):
        if batch is not None:
            video, label, idx, video_idx = batch
            print(idx, video.size(), label.size(), idx.size(), video_idx.size(), time.time() - tic)
            print(f'Batch time (s): {time.time() - tic}')
            tic = time.time()