#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h> // for unlink()
#include <sys/wait.h> // for WEXITSTATUS, etc.

typedef unsigned char Bool;
#define TRUE 1
#define FALSE 0

#define MAX_DEPTH 20
#define MAX_SONS 50
#define MAX_ORDERINGS 1000
#define MAX_RETRIES 1000 /* max generation retries before giving up */

typedef struct Assembly {
    int row, col;
    int requires; /* resource index or -1 */
    int num_sons, start_son, end_son;
    int part_of_row, part_of_col;

    int num_ao;
    int aop_row[MAX_ORDERINGS], aop_col[MAX_ORDERINGS];

    int num_tp;
    int tp_row[MAX_ORDERINGS], tp_col[MAX_ORDERINGS];

    int num_ro;
    int rop_row[MAX_ORDERINGS], rop_col[MAX_ORDERINGS];
} Assembly, *Assembly_pointer;

/* Globals */
int gdepth, gmax_sons, gresources;
int gp_has_sons, gp_requires, gp_transient_part, gp_assemble_order, gp_remove_order;
Assembly_pointer gtree[MAX_DEPTH + 1];
int gnum_cols[MAX_DEPTH + 1];

/* Prototypes */
void initialize_parameters(int problem_index);
void create_tree(void);
void setup_transient_parts(void);
void setup_orderings(void);
Bool validate_problem(void);
void print_problem(const char *filename);

/* 
 * Paths and templates. 
 * Adjust as needed for your setup.
 */
static const char *DOMAIN_PATH = "/ssd1/rh38988/projects/LLm-pddl-benchmark/data/pddl/assembly/domain.pddl";

/*
 * Use `timeout 60s` to cap each FD run to 60 seconds.
 * A single-phase lazy_greedy([ff()], preferred=[ff()]) search 
 * will stop as soon as it finds a solution.
 */
static const char *PLANNER_CMD_TEMPLATE = 
    "timeout 60s python /ssd1/rh38988/projects/LLm-pddl-benchmark/submodule/downward/fast-downward.py "
    "%s %s "
    "--search \"lazy_greedy([ff()], preferred=[ff()])\"";

/* Main */
int main(void) {
    int successful_problems = 0;
    int attempts = 0;

    while (successful_problems < 100) {
        if (++attempts > MAX_RETRIES) {
            fprintf(stderr, "Exceeded MAX_RETRIES for problem generation.\n");
            exit(1);
        }

        /* 1) Generate random parameters for this problem index. */
        initialize_parameters(successful_problems);

        /* 2) Create the tree, transient parts, orderings. */
        create_tree();
        setup_transient_parts();
        setup_orderings();

        /* 3) Quick validation. */
        if (!validate_problem()) {
            printf("Problem %d not valid. Retrying...\n", successful_problems + 1);
            continue;
        }

        /* 4) Print out the problem file pNNN.pddl. */
        char problem_path[512];
        snprintf(problem_path, sizeof(problem_path),
                 "/ssd1/rh38988/projects/LLm-pddl-benchmark/data/pddl/assembly/p%03d.pddl",
                 successful_problems + 1);
        print_problem(problem_path);
        printf("Generated problem %d -> %s\n", successful_problems + 1, problem_path);

        /* 5) Invoke FD with a time limit using `timeout`. */
        char cmd[2048];
        snprintf(cmd, sizeof(cmd), PLANNER_CMD_TEMPLATE, DOMAIN_PATH, problem_path);
        printf("Running FD: %s\n", cmd);

        int ret = system(cmd);

        /*
         * system() returns:
         *   - 0 if command exited with code 0
         *   - 256*N if command exited with code N
         *   - -1 if system() call fails
         *
         * We'll decode with WEXITSTATUS on ret.
         */
        if (ret == -1) {
            printf("Error calling system() for FD. Removing %s.\n", problem_path);
            unlink(problem_path);
            continue;
        }

        int exit_code = WEXITSTATUS(ret);
        if (exit_code == 0) {
            /* Found a solution => keep it. */
            printf("Problem %d is solvable. Keeping it.\n", successful_problems + 1);
            successful_problems++;
            attempts = 0; /* reset attempts for next problem */
        } else if (exit_code == 124) {
            /* Timed out => remove file. */
            printf("Problem %d timed out after 60s (exit=124). Removing file.\n", successful_problems + 1);
            unlink(problem_path);
        } else {
            /* Non-zero exit code => likely unsolvable or error => remove. */
            printf("Problem %d not solved (exit=%d). Removing file.\n", successful_problems + 1, exit_code);
            unlink(problem_path);
        }
    }

    printf("100 solvable problems successfully generated.\n");
    return 0;
}

/* Random parameter init */
void initialize_parameters(int problem_index) {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    srandom(tv.tv_usec + problem_index);

    gdepth            = 3 + (random() % 3);          
    gmax_sons         = 3 + (random() % 3);          
    gresources        = 2 + (random() % 3);          
    gp_has_sons       = 70 + (random() % 20);        
    gp_requires       = 5 + (random() % 20);         
    gp_transient_part = 10 + (random() % 20);        
    gp_assemble_order = 20 + (random() % 20);        
    gp_remove_order   = 5 + (random() % 10);         
}

/* Tree building */
void create_tree(void) {
    int size = (gmax_sons / gdepth > 0) ? (gmax_sons / gdepth) : 1;
    int current_range_low = gmax_sons - size + 1;

    gnum_cols[0] = 1;
    gtree[0] = (Assembly *)calloc(1, sizeof(Assembly));
    gtree[0][0].row = 0;
    gtree[0][0].col = 0;

    for (int i = 0; i <= gdepth; i++) {
        if (i < gdepth) gnum_cols[i+1] = 0;

        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *n = &gtree[i][j];
            n->requires = -1;
            n->num_sons = 0;
            n->num_ao = n->num_tp = n->num_ro = 0;
            n->row = i;
            n->col = j;

            if (i < gdepth) {
                if ((random() % 100) < gp_requires) {
                    n->requires = random() % gresources;
                }
                if ((i == 0) || ((random() % 100) < gp_has_sons)) {
                    n->num_sons = (random() % size) + current_range_low;
                    if (n->num_sons > 0) {
                        n->start_son = gnum_cols[i+1];
                        gnum_cols[i+1] += n->num_sons;
                        n->end_son = gnum_cols[i+1] - 1;
                    }
                }
            } else {
                /* Leaves can also require a resource. */
                if ((random() % 100) < gp_requires) {
                    n->requires = random() % gresources;
                }
                n->num_sons = 0;
            }
        }

        if (i < gdepth && gnum_cols[i+1] > 0) {
            gtree[i+1] = (Assembly *)calloc(gnum_cols[i+1], sizeof(Assembly));
            for (int j = 0; j < gnum_cols[i]; j++) {
                Assembly *par = &gtree[i][j];
                for (int c = par->start_son; c <= par->end_son; c++) {
                    gtree[i+1][c].part_of_row = i;
                    gtree[i+1][c].part_of_col = j;
                }
            }
        }

        current_range_low -= (size + 1);
        if (current_range_low < 0) current_range_low = 0;
    }

    /* Ensure at least one leaf node at depth gdepth. */
    if (gnum_cols[gdepth] == 0) {
        gnum_cols[gdepth] = 1;
        gtree[gdepth] = (Assembly*)calloc(1, sizeof(Assembly));
        gtree[gdepth][0].row = gdepth;
        gtree[gdepth][0].col = 0;
        gtree[gdepth][0].num_sons = 0;
        gtree[gdepth][0].requires = -1;
        if ((random() % 100) < gp_requires) {
            gtree[gdepth][0].requires = random() % gresources;
        }
    }
}

/* Transient parts */
void setup_transient_parts(void) {
    for (int i = 1; i <= gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *child = &gtree[i][j];
            int pr = child->part_of_row;
            int pc = child->part_of_col;

            child->num_tp = 0;
            for (int up = 0; up < i; up++) {
                for (int uc = 0; uc < gnum_cols[up]; uc++) {
                    /* Don't add the direct parent as transient. */
                    if (up == pr && uc == pc) continue;
                    if ((random() % 100) < gp_transient_part) {
                        child->tp_row[child->num_tp] = up;
                        child->tp_col[child->num_tp] = uc;
                        child->num_tp++;
                    }
                }
            }
        }
    }
}

/* Assemble/remove order constraints among siblings */
void setup_orderings(void) {
    for (int i = 0; i < gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *parent = &gtree[i][j];
            if (parent->num_sons <= 1) continue;

            for (int c1 = parent->start_son; c1 <= parent->end_son; c1++) {
                for (int c2 = c1 + 1; c2 <= parent->end_son; c2++) {
                    /* assemble-order with probability gp_assemble_order. */
                    if ((random() % 100) < gp_assemble_order) {
                        if (random() % 2) {
                            gtree[i+1][c2].aop_row[ gtree[i+1][c2].num_ao ] = i+1;
                            gtree[i+1][c2].aop_col[ gtree[i+1][c2].num_ao ] = c1;
                            gtree[i+1][c2].num_ao++;
                        } else {
                            gtree[i+1][c1].aop_row[ gtree[i+1][c1].num_ao ] = i+1;
                            gtree[i+1][c1].aop_col[ gtree[i+1][c1].num_ao ] = c2;
                            gtree[i+1][c1].num_ao++;
                        }
                    }
                    /* remove-order with probability gp_remove_order. */
                    if ((random() % 100) < gp_remove_order) {
                        if (random() % 2) {
                            gtree[i+1][c2].rop_row[ gtree[i+1][c2].num_ro ] = i+1;
                            gtree[i+1][c2].rop_col[ gtree[i+1][c2].num_ro ] = c1;
                            gtree[i+1][c2].num_ro++;
                        } else {
                            gtree[i+1][c1].rop_row[ gtree[i+1][c1].num_ro ] = i+1;
                            gtree[i+1][c1].rop_col[ gtree[i+1][c1].num_ro ] = c2;
                            gtree[i+1][c1].num_ro++;
                        }
                    }
                }
            }
        }
    }
}

/* Simple validation */
Bool validate_problem(void) {
    if (gnum_cols[gdepth] < 1) {
        fprintf(stderr, "Validation failed: no leaf nodes.\n");
        return FALSE;
    }
    for (int i = 0; i <= gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *n = &gtree[i][j];
            if (n->requires >= gresources) {
                fprintf(stderr, "Invalid resource required at (%d,%d)\n", i, j);
                return FALSE;
            }
        }
    }
    return TRUE;
}

/* Print problem */
void print_problem(const char *filename) {
    FILE *fp = fopen(filename, "w");
    if (!fp) {
        perror("Error writing file");
        exit(1);
    }

    fprintf(fp, "(define (problem auto-assembly)\n");
    fprintf(fp, "   (:domain assembly)\n\n");

    fprintf(fp, "   (:objects ");
    for (int i = 0; i < gresources; i++) {
        fprintf(fp, "r%d ", i);
    }
    fprintf(fp, "- resource ");

    for (int i = 0; i <= gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            fprintf(fp, "a-%d-%d ", i, j);
        }
    }
    fprintf(fp, "- assembly)\n\n");

    fprintf(fp, "   (:init\n");
    /* part-of direct parent->child */
    for (int i = 0; i < gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *p = &gtree[i][j];
            if (p->num_sons == 0) continue;
            for (int c = p->start_son; c <= p->end_son; c++) {
                fprintf(fp, "      (part-of a-%d-%d a-%d-%d)\n", i+1, c, i, j);
            }
        }
    }

    /* transient-part relationships */
    for (int i = 1; i <= gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *ch = &gtree[i][j];
            for (int t = 0; t < ch->num_tp; t++) {
                int rr = ch->tp_row[t];
                int cc = ch->tp_col[t];
                fprintf(fp, "      (transient-part a-%d-%d a-%d-%d)\n", i, j, rr, cc);
            }
        }
    }

    /* resources and assemblies all available initially */
    for (int i = 0; i < gresources; i++) {
        fprintf(fp, "      (available r%d)\n", i);
    }
    for (int i = 0; i <= gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            fprintf(fp, "      (available a-%d-%d)\n", i, j);
        }
    }

    /* requires constraints */
    for (int i = 0; i <= gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *n = &gtree[i][j];
            if (n->requires >= 0) {
                fprintf(fp, "      (requires a-%d-%d r%d)\n", i, j, n->requires);
            }
        }
    }

    /* assemble-order constraints */
    for (int i = 1; i <= gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *ch = &gtree[i][j];
            int pr = ch->part_of_row;
            int pc = ch->part_of_col;
            for (int a = 0; a < ch->num_ao; a++) {
                int rr = ch->aop_row[a];
                int cc = ch->aop_col[a];
                fprintf(fp, "      (assemble-order a-%d-%d a-%d-%d a-%d-%d)\n",
                        rr, cc, i, j, pr, pc);
            }
        }
    }

    /* remove-order constraints */
    for (int i = 1; i <= gdepth; i++) {
        for (int j = 0; j < gnum_cols[i]; j++) {
            Assembly *ch = &gtree[i][j];
            int pr = ch->part_of_row;
            int pc = ch->part_of_col;
            for (int r = 0; r < ch->num_ro; r++) {
                int rr = ch->rop_row[r];
                int cc = ch->rop_col[r];
                fprintf(fp, "      (remove-order a-%d-%d a-%d-%d a-%d-%d)\n",
                        rr, cc, i, j, pr, pc);
            }
        }
    }

    fprintf(fp, "   )\n\n");

    /* Goal: complete the root */
    fprintf(fp, "   (:goal (complete a-0-0))\n");
    fprintf(fp, ")\n");
    fclose(fp);
}
